/**
 * SimpleCommenter Admin JavaScript
 */

(function($) {
    'use strict';

    const SC = {
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.refreshAccount(); // Refresh account on load (upgrades token permissions if needed)
            this.loadDomains();
            this.loadMembers();
            this.loadClients();
            this.initComments(); // Initialize comments if on comments tab
        },

        /**
         * Refresh account info (also upgrades token permissions)
         */
        refreshAccount: function() {
            // Only refresh if connected (check if settings section exists)
            if (!$('#sc-settings-form').length) {
                return;
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_refresh_account',
                    nonce: scAdmin.nonce
                }
            });
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Tab switching (login/register)
            $('.sc-tab').on('click', this.handleTabSwitch);

            // Settings tab switching
            $('.sc-settings-tab').on('click', this.handleSettingsTabSwitch);

            // Login form
            $('#sc-login-form').on('submit', this.handleLogin.bind(this));

            // Register form
            $('#sc-register-form').on('submit', this.handleRegister.bind(this));

            // Google login/register buttons
            $('#sc-google-login').on('click', this.handleGoogleLogin.bind(this));
            $('#sc-google-register').on('click', this.handleGoogleRegister.bind(this));

            // Open Dashboard
            $('#sc-open-dashboard').on('click', this.handleOpenDashboard.bind(this));

            // Sync current user (Connect to SimpleCommenter button)
            $('#sc-sync-me').on('click', this.handleSyncCurrentUser.bind(this));

            // Disconnect (in settings, header, and project selection page)
            $('#sc-disconnect, #sc-disconnect-header, #sc-disconnect-select').on('click', this.handleDisconnect.bind(this));

            // Upgrade button
            $('#sc-upgrade-btn').on('click', this.handleUpgrade.bind(this));

            // Change project button (shown on domain mismatch)
            $('#sc-change-project-btn').on('click', this.handleChangeProject.bind(this));

            // Add domain - bound dynamically in renderDomains

            // Settings form
            $('#sc-settings-form').on('submit', this.handleSaveSettings.bind(this));

            // Roles form
            $('#sc-roles-form').on('submit', this.handleSaveRoles.bind(this));

            // Sync settings
            $('#sc-save-sync').on('click', this.handleSaveSyncSettings.bind(this));

            // Sync Now button
            $('#sc-sync-now').on('click', this.handleSyncNow.bind(this));

            // Team members
            $('#sc-add-member').on('click', this.handleAddMember.bind(this));

            // Clients
            $('#sc-add-client').on('click', this.handleAddClient.bind(this));

            // Admin roles form
            $('#sc-admin-roles-form').on('submit', this.handleSaveAdminRoles.bind(this));

            // Access settings
            $('#sc-save-access').on('click', this.handleSaveAccessSettings.bind(this));

            // Toggle Open registration option based on Authorized Users setting
            $('#tokenAccess').on('change', this.updateClientAccessOptions);
            this.updateClientAccessOptions();

            // Change project
            $('#sc-change-project').on('click', this.handleChangeProject.bind(this));

            // Login method tabs (password/code)
            $('.sc-login-method').on('click', this.handleLoginMethodSwitch);

            // Code login forms
            $('#sc-code-email-form').on('submit', this.handleSendCode.bind(this));
            $('#sc-code-verify-form').on('submit', this.handleVerifyCode.bind(this));
            $('#sc-resend-code').on('click', this.handleResendCode.bind(this));

            // Register method tabs (password/code)
            $('.sc-register-method').on('click', this.handleRegisterMethodSwitch);

            // Code registration forms
            $('#sc-reg-code-email-form').on('submit', this.handleSendRegCode.bind(this));
            $('#sc-reg-code-verify-form').on('submit', this.handleVerifyRegCode.bind(this));
            $('#sc-reg-resend-code').on('click', this.handleResendRegCode.bind(this));

            // Check for Google OAuth callback
            this.checkGoogleCallback();
        },

        /**
         * Handle login method switch (password/code)
         */
        handleLoginMethodSwitch: function() {
            const method = $(this).data('method');

            $('.sc-login-method').removeClass('active');
            $(this).addClass('active');

            $('.sc-login-method-content').hide();
            $('#login-method-' + method).show();
        },

        /**
         * Handle tab switch (login/register)
         */
        handleTabSwitch: function() {
            const tab = $(this).data('tab');

            $('.sc-tab').removeClass('active');
            $(this).addClass('active');

            $('.sc-tab-content').hide();
            $('#tab-' + tab).show();
        },

        /**
         * Handle settings tab switch
         */
        handleSettingsTabSwitch: function() {
            const tab = $(this).data('settings-tab');

            $('.sc-settings-tab').removeClass('active');
            $(this).addClass('active');

            $('.sc-settings-tab-content').hide();
            $('#settings-tab-' + tab).show();
        },

        /**
         * Handle login form submission
         */
        handleLogin: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Connecting...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_connect',
                    nonce: scAdmin.nonce,
                    email: $('#login-email').val(),
                    password: $('#login-password').val()
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Connection failed. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle register form submission
         */
        handleRegister: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Creating Account...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_register',
                    nonce: scAdmin.nonce,
                    email: $('#reg-email').val(),
                    password: $('#reg-password').val(),
                    company: $('#reg-company').val()
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Registration failed. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle Google login
         */
        handleGoogleLogin: function() {
            if (scAdmin.googleAuthUrl) {
                window.location.href = scAdmin.googleAuthUrl;
            }
        },

        /**
         * Handle Google register
         */
        handleGoogleRegister: function() {
            if (scAdmin.googleRegisterUrl) {
                window.location.href = scAdmin.googleRegisterUrl;
            }
        },

        /**
         * Check for Google OAuth callback
         * The callback adds ?token=xxx&account=xxx to the URL
         */
        checkGoogleCallback: function() {
            const urlParams = new URLSearchParams(window.location.search);
            const token = urlParams.get('token');
            const accountJson = urlParams.get('account');

            if (token && accountJson) {
                // Google auth was successful, save the connection via AJAX
                try {
                    const account = JSON.parse(decodeURIComponent(accountJson));

                    $.ajax({
                        url: scAdmin.ajaxUrl,
                        method: 'POST',
                        data: {
                            action: 'sc_save_google_auth',
                            nonce: scAdmin.nonce,
                            token: token,
                            account: accountJson
                        },
                        success: function(response) {
                            // Clean URL and reload
                            window.history.replaceState({}, document.title, window.location.pathname + '?page=simple-commenter');
                            location.reload();
                        },
                        error: function() {
                            SC.showError('Failed to complete Google authentication.');
                        }
                    });
                } catch (e) {
                    SC.showError('Invalid authentication response.');
                }
            }
        },

        /**
         * Handle Open Dashboard (auto-login)
         */
        handleOpenDashboard: function() {
            const $button = $('#sc-open-dashboard');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Opening...');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_get_login_link',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.url) {
                        window.open(response.data.url, '_blank');
                        $button.prop('disabled', false).text(originalText);
                    } else {
                        SC.showError(response.data?.message || 'Failed to get login link.');
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to get login link.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle Sync Current User (Connect to SimpleCommenter)
         */
        handleSyncCurrentUser: function() {
            const $button = $('#sc-sync-me');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Connecting...');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_sync_current_user',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Reload page to show updated state
                        window.location.reload();
                    } else {
                        SC.showError(response.data?.message || 'Failed to connect to SimpleCommenter.');
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to connect to SimpleCommenter.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle disconnect
         */
        handleDisconnect: function(e) {
            if (!confirm('Are you sure you want to disconnect? The widget will stop working on your site.')) {
                return;
            }

            const $button = $(e.target);
            const originalText = $button.text();
            $button.prop('disabled', true).text('Disconnecting...');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_disconnect',
                    nonce: scAdmin.nonce
                },
                success: function() {
                    location.reload();
                },
                error: function() {
                    SC.showError('Disconnect failed. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle upgrade button click (auto-login and redirect to pricing)
         */
        handleUpgrade: function() {
            const $button = $('#sc-upgrade-btn');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Opening...');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_get_upgrade_link',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.url) {
                        window.open(response.data.url, '_blank');
                        $button.prop('disabled', false).text(originalText);
                    } else {
                        SC.showError(response.data?.message || 'Failed to get upgrade link.');
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to get upgrade link.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Load domains list
         */
        loadDomains: function() {
            const $container = $('#sc-domains-list');

            if (!$container.length) {
                return;
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_get_domains',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.domains && response.data.domains.length > 0) {
                        SC.renderDomains(response.data.domains);
                    } else if (response.success) {
                        // No projects - show "Add Domain" button
                        let html = '<p>No projects found. Create your first project:</p>';
                        html += '<button id="sc-add-domain" type="button" class="button button-primary">';
                        html += 'Add This Site as New Project';
                        html += '</button>';
                        $container.html(html);
                        // Bind add domain button
                        $('#sc-add-domain').on('click', SC.handleAddDomain.bind(SC));
                    } else {
                        // API error - show error at top and still show Add button
                        SC.showError(response.data?.message || 'Failed to load projects');
                        let html = '<p>Could not load existing projects. You can still create a new one:</p>';
                        html += '<button id="sc-add-domain" type="button" class="button button-primary">';
                        html += 'Add This Site as New Project';
                        html += '</button>';
                        $container.html(html);
                        $('#sc-add-domain').on('click', SC.handleAddDomain.bind(SC));
                    }
                },
                error: function() {
                    SC.showError('Failed to connect to SimpleCommenter API');
                    $container.html('<p>Connection failed. Please try again.</p>');
                }
            });
        },

        /**
         * Render domains list with matching domain first
         */
        renderDomains: function(domains) {
            const $container = $('#sc-domains-list');
            const currentDomain = $container.data('current-domain') || '';

            // Normalize current domain for comparison
            const normalizedCurrent = currentDomain.replace(/^www\./, '').toLowerCase();

            // Find matching domain and sort
            let matchingDomain = null;
            let otherDomains = [];

            domains.forEach(function(domain) {
                const domainHost = (domain.domain || '').replace(/^www\./, '').toLowerCase();
                if (domainHost && domainHost === normalizedCurrent) {
                    matchingDomain = domain;
                } else {
                    otherDomains.push(domain);
                }
            });

            // Build HTML - matching domain first as primary action
            let html = '';

            if (matchingDomain) {
                // Show matching domain as a highlighted button
                html += '<button type="button" class="sc-select-domain sc-domain-match" ';
                html += 'data-id="' + matchingDomain.id + '" ';
                html += 'data-project-id="' + matchingDomain.projectId + '" ';
                html += 'data-domain="' + (matchingDomain.domain || '') + '">';
                html += '<strong>' + (matchingDomain.projectName || matchingDomain.projectId);
                html += '<span class="sc-match-badge">Recommended</span></strong>';
                html += '<span>' + (matchingDomain.domain || 'No domain set') + '</span>';
                html += '</button>';

                // Add divider and "Add This Site" button
                html += '<div class="sc-or-divider"><span>or</span></div>';
                html += '<button id="sc-add-domain" type="button" class="button button-primary" style="margin-bottom: 16px;">';
                html += 'Add This Site as New Project';
                html += '</button>';

                // Show other domains if any
                if (otherDomains.length > 0) {
                    html += '<div class="sc-or-divider"><span>other projects</span></div>';
                    html += '<ul class="sc-domain-list">';
                    otherDomains.forEach(function(domain) {
                        html += '<li>';
                        html += '<button type="button" class="sc-select-domain" ';
                        html += 'data-id="' + domain.id + '" ';
                        html += 'data-project-id="' + domain.projectId + '" ';
                        html += 'data-domain="' + (domain.domain || '') + '">';
                        html += '<strong>' + (domain.projectName || domain.projectId) + '</strong>';
                        html += '<span>' + (domain.domain || 'No domain set') + '</span>';
                        html += '</button>';
                        html += '</li>';
                    });
                    html += '</ul>';
                }
            } else {
                // No matching domain - show "Add This Site" first
                html += '<button id="sc-add-domain" type="button" class="button button-primary" style="margin-bottom: 16px;">';
                html += 'Add This Site as New Project';
                html += '</button>';

                if (domains.length > 0) {
                    html += '<div class="sc-or-divider"><span>or select existing</span></div>';
                    html += '<ul class="sc-domain-list">';
                    domains.forEach(function(domain) {
                        html += '<li>';
                        html += '<button type="button" class="sc-select-domain" ';
                        html += 'data-id="' + domain.id + '" ';
                        html += 'data-project-id="' + domain.projectId + '" ';
                        html += 'data-domain="' + (domain.domain || '') + '">';
                        html += '<strong>' + (domain.projectName || domain.projectId) + '</strong>';
                        html += '<span>' + (domain.domain || 'No domain set') + '</span>';
                        html += '</button>';
                        html += '</li>';
                    });
                    html += '</ul>';
                }
            }

            $container.html(html);

            // Bind select events
            $('.sc-select-domain').on('click', function() {
                const $btn = $(this);
                const isMatch = $btn.hasClass('sc-domain-match');
                SC.selectDomain($btn.data('id'), $btn.data('project-id'), $btn.data('domain'), isMatch);
            });

            // Bind add domain button
            $('#sc-add-domain').on('click', SC.handleAddDomain.bind(SC));
        },

        /**
         * Load team members
         */
        loadMembers: function(showLoading) {
            const $container = $('#sc-members-list');

            if (!$container.length) {
                return;
            }

            // Show loading state (unless explicitly disabled)
            if (showLoading !== false) {
                $container.html('<div class="sc-loading"><span class="spinner is-active"></span> Loading team members...</div>');
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_get_members',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.members) {
                        SC.renderMembers(response.data.members);
                    } else {
                        $container.html('<p>No team members found.</p>');
                    }
                },
                error: function() {
                    $container.html('<p class="sc-error">Failed to load team members.</p>');
                }
            });
        },

        /**
         * Check if an email matches a WordPress user
         * @param {string} email - Email to check
         * @returns {object|null} - WordPress user info or null
         */
        getMatchingWpUser: function(email) {
            if (!scAdmin.wpUsers || !email) {
                return null;
            }
            const normalizedEmail = email.toLowerCase();
            return scAdmin.wpUsers.find(function(wpUser) {
                return wpUser.email === normalizedEmail;
            }) || null;
        },

        /**
         * Get WordPress badge HTML
         * @param {object} wpUser - WordPress user object
         * @returns {string} - Badge HTML
         */
        getWpBadgeHtml: function(wpUser) {
            if (!wpUser) return '';
            const roleName = wpUser.role.charAt(0).toUpperCase() + wpUser.role.slice(1).replace('_', ' ');
            return '<span class="sc-wp-badge" title="WordPress ' + roleName + '">WP</span>';
        },

        /**
         * Render members list
         */
        renderMembers: function(members) {
            const $container = $('#sc-members-list');

            if (!members.length) {
                $container.html('<p>No team members yet.</p>');
                return;
            }

            let html = '';

            members.forEach(function(member) {
                const isOwner = member.isOwner || member.role === 'owner';
                const wpUser = SC.getMatchingWpUser(member.email);
                html += '<div class="sc-member-item" data-id="' + member.id + '">';
                html += '<div class="sc-member-info">';
                html += '<span class="sc-member-name">' + (member.name || member.email);
                html += SC.getWpBadgeHtml(wpUser);
                html += '</span>';
                html += '<span class="sc-member-email">' + member.email + '</span>';
                html += '</div>';
                html += '<div class="sc-member-meta">';
                html += '<span class="sc-member-role ' + (isOwner ? 'owner' : '') + '">' + (isOwner ? 'Owner' : member.role) + '</span>';
                if (!isOwner) {
                    html += '<button type="button" class="sc-member-remove" data-id="' + member.id + '" title="Remove">';
                    html += '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" width="16" height="16"><path stroke-linecap="round" stroke-linejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>';
                    html += '</button>';
                }
                html += '</div>';
                html += '</div>';
            });

            $container.html(html);

            // Bind remove events
            $('.sc-member-remove').on('click', function() {
                SC.removeMember($(this).data('id'));
            });
        },

        /**
         * Load clients for current domain
         */
        loadClients: function(showLoading) {
            const $container = $('#sc-clients-list');

            if (!$container.length) {
                return;
            }

            // Show loading state (unless explicitly disabled)
            if (showLoading !== false) {
                $container.html('<div class="sc-loading"><span class="spinner is-active"></span> Loading clients...</div>');
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_get_clients',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success && response.data.clients) {
                        SC.renderClients(response.data.clients);
                    } else {
                        $container.html('<p>No clients found.</p>');
                    }
                },
                error: function() {
                    $container.html('<p class="sc-error">Failed to load clients.</p>');
                }
            });
        },

        /**
         * Render clients list
         */
        renderClients: function(clients) {
            const $container = $('#sc-clients-list');

            if (!clients.length) {
                $container.html('<p>No clients yet.</p>');
                return;
            }

            let html = '';

            clients.forEach(function(client) {
                const wpUser = SC.getMatchingWpUser(client.email);
                html += '<div class="sc-client-item" data-id="' + client.id + '">';
                html += '<div class="sc-client-info">';
                html += '<span class="sc-client-name">' + (client.name || client.email);
                html += SC.getWpBadgeHtml(wpUser);
                html += '</span>';
                html += '<span class="sc-client-email">' + client.email + '</span>';
                html += '</div>';
                html += '<div class="sc-client-meta">';
                if (client.approved) {
                    html += '<span class="sc-client-status approved">Approved</span>';
                } else {
                    html += '<span class="sc-client-status pending">Pending</span>';
                }
                html += '</div>';
                html += '</div>';
            });

            $container.html(html);
        },

        /**
         * Handle add client
         */
        handleAddClient: function() {
            const email = $('#client-email').val();
            const name = $('#client-name').val();

            if (!email) {
                SC.showError('Email is required');
                return;
            }

            const $button = $('#sc-add-client');
            const $container = $('#sc-clients-list');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Adding...');
            $container.addClass('sc-loading-overlay');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_add_client',
                    nonce: scAdmin.nonce,
                    email: email,
                    name: name
                },
                success: function(response) {
                    if (response.success) {
                        // Clear form
                        $('#client-email').val('');
                        $('#client-name').val('');
                        // Reload clients
                        SC.loadClients();
                        $button.prop('disabled', false).text(originalText);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                        $container.removeClass('sc-loading-overlay');
                    }
                },
                error: function() {
                    SC.showError('Failed to add client.');
                    $button.prop('disabled', false).text(originalText);
                    $container.removeClass('sc-loading-overlay');
                }
            });
        },

        /**
         * Handle add member
         */
        handleAddMember: function() {
            const email = $('#member-email').val();
            const name = $('#member-name').val();
            const role = $('#member-role').val();

            if (!email) {
                SC.showError('Email is required');
                return;
            }

            const $button = $('#sc-add-member');
            const $container = $('#sc-members-list');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Adding...');

            // Show loading on members list
            $container.addClass('sc-loading-overlay');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_add_member',
                    nonce: scAdmin.nonce,
                    email: email,
                    name: name,
                    role: role
                },
                success: function(response) {
                    if (response.success) {
                        // Clear form
                        $('#member-email').val('');
                        $('#member-name').val('');
                        $('#member-role').val('team');
                        // Reload members (will remove loading state)
                        SC.loadMembers();
                        $button.prop('disabled', false).text(originalText);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                        $container.removeClass('sc-loading-overlay');
                    }
                },
                error: function() {
                    SC.showError('Failed to add member.');
                    $button.prop('disabled', false).text(originalText);
                    $container.removeClass('sc-loading-overlay');
                }
            });
        },

        /**
         * Remove a member
         */
        removeMember: function(memberId) {
            if (!confirm('Are you sure you want to remove this team member?')) {
                return;
            }

            const $container = $('#sc-members-list');
            const $memberItem = $container.find('.sc-member-item[data-id="' + memberId + '"]');

            // Show loading state on the member being removed
            $memberItem.addClass('sc-removing');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_remove_member',
                    nonce: scAdmin.nonce,
                    memberId: memberId
                },
                success: function(response) {
                    if (response.success) {
                        SC.loadMembers();
                    } else {
                        SC.showError(response.data.message);
                        $memberItem.removeClass('sc-removing');
                    }
                },
                error: function() {
                    SC.showError('Failed to remove member.');
                    $memberItem.removeClass('sc-removing');
                }
            });
        },

        /**
         * Handle save sync settings
         */
        handleSaveSyncSettings: function() {
            const autoSync = $('#sc-auto-sync-team').is(':checked');

            const $button = $('#sc-save-sync');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Saving...');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_save_sync_settings',
                    nonce: scAdmin.nonce,
                    autoSync: autoSync.toString()
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to save sync settings.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle Sync Now - sync all WordPress users to SimpleCommenter
         */
        handleSyncNow: function() {
            const $button = $('#sc-sync-now');
            const $status = $('#sc-sync-status');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Syncing...');
            $status.text('').removeClass('success error');

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_sync_all_users',
                    nonce: scAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        const data = response.data;
                        let statusMsg = data.synced + ' synced';
                        if (data.skipped > 0) {
                            statusMsg += ', ' + data.skipped + ' skipped';
                        }
                        if (data.errors > 0) {
                            statusMsg += ', ' + data.errors + ' errors';
                            // Log error details to console for debugging
                            if (data.errorDetails && data.errorDetails.length > 0) {
                                console.group('SimpleCommenter Sync Errors');
                                data.errorDetails.forEach(function(err) {
                                    console.error(err.email + ': ' + err.error);
                                });
                                console.groupEnd();

                                // Show all errors in an alert below the status
                                let errorList = '<div class="sc-sync-errors">';
                                errorList += '<strong>Sync errors:</strong><ul>';
                                data.errorDetails.forEach(function(err) {
                                    errorList += '<li>' + SC.escapeHtml(err.email) + ': ' + SC.escapeHtml(err.error) + '</li>';
                                });
                                errorList += '</ul></div>';

                                // Show errors after the sync box
                                $('.sc-sync-errors').remove(); // Remove any existing error list
                                $('.sc-sync-box').after(errorList);
                            }
                        } else {
                            // Clear any previous error list on success
                            $('.sc-sync-errors').remove();
                        }
                        $status.text(statusMsg).addClass(data.errors > 0 ? 'error' : 'success');

                        // Reload members and clients lists
                        SC.loadMembers();
                        SC.loadClients();
                    } else {
                        $status.text(response.data.message || 'Sync failed').addClass('error');
                    }
                    $button.prop('disabled', false).text(originalText);
                },
                error: function() {
                    $status.text('Sync failed').addClass('error');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle save admin roles
         */
        handleSaveAdminRoles: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Saving...');

            // Get all checked roles (excluding disabled administrator which is always included)
            const roles = [];
            $form.find('input[name="admin_roles[]"]:checked').each(function() {
                roles.push($(this).val());
            });

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_save_admin_roles',
                    nonce: scAdmin.nonce,
                    roles: JSON.stringify(roles)
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to save admin roles.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Toggle Open registration option based on Authorized Users setting
         */
        updateClientAccessOptions: function() {
            const tokenAccess = $('#tokenAccess').is(':checked');
            const $openOption = $('#clientAccessOpenLabel');
            const $openRadio = $('#clientAccessOpen');

            if (tokenAccess) {
                $openOption.addClass('sc-radio-option-disabled');
                $openRadio.prop('disabled', true);
                // If Open was selected, switch to Closed
                if ($openRadio.is(':checked')) {
                    $('input[name="clientAccess"][value="closed"]').prop('checked', true);
                }
            } else {
                $openOption.removeClass('sc-radio-option-disabled');
                $openRadio.prop('disabled', false);
            }
        },

        /**
         * Handle save access settings
         */
        handleSaveAccessSettings: function() {
            const $button = $('#sc-save-access');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Saving...');

            const settings = {
                conditionalLoading: $('#conditionalLoading').is(':checked'),
                tokenAccess: $('#tokenAccess').is(':checked'),
                clientAccess: $('input[name="clientAccess"]:checked').val() || 'open'
            };

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_save_settings',
                    nonce: scAdmin.nonce,
                    settings: JSON.stringify(settings)
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to save access settings.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Select a domain
         */
        selectDomain: function(domainId, projectId, projectDomain, skipWarning) {
            const $buttons = $('.sc-select-domain');
            const $container = $('#sc-domains-list');
            const currentDomain = ($container.data('current-domain') || '').toLowerCase().replace(/^www\./, '');
            const targetDomain = (projectDomain || '').toLowerCase().replace(/^www\./, '');

            // Check for domain mismatch and show warning (unless skipWarning is true)
            if (!skipWarning && targetDomain && currentDomain && targetDomain !== currentDomain) {
                const confirmMsg = 'Warning: This project\'s domain (' + projectDomain + ') doesn\'t match your WordPress site (' + $container.data('current-domain') + ').\n\n' +
                    'This is useful for staging sites that want to use the same project as production.\n\n' +
                    'Continue anyway?';
                if (!confirm(confirmMsg)) {
                    return;
                }
            }

            // Show loading state
            $buttons.prop('disabled', true);
            $container.addClass('sc-loading-overlay');

            // Add loading indicator
            if (!$container.find('.sc-selecting-loader').length) {
                $container.append('<div class="sc-selecting-loader"><span class="spinner is-active"></span> Selecting project...</div>');
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_select_domain',
                    nonce: scAdmin.nonce,
                    domainId: domainId,
                    projectId: projectId
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        SC.showError(response.data.message);
                        $buttons.prop('disabled', false);
                        $container.removeClass('sc-loading-overlay');
                        $container.find('.sc-selecting-loader').remove();
                    }
                },
                error: function() {
                    SC.showError('Failed to select project.');
                    $buttons.prop('disabled', false);
                    $container.removeClass('sc-loading-overlay');
                    $container.find('.sc-selecting-loader').remove();
                }
            });
        },

        /**
         * Handle add domain
         */
        handleAddDomain: function() {
            const $button = $('#sc-add-domain');
            const $container = $('#sc-domains-list');
            const originalText = $button.text();
            const $errorEl = $('#sc-add-domain-error');

            $button.prop('disabled', true).text('Adding...');
            $errorEl.hide();

            // Show loading overlay
            $container.addClass('sc-loading-overlay');
            if (!$container.find('.sc-selecting-loader').length) {
                $container.append('<div class="sc-selecting-loader"><span class="spinner is-active"></span> Creating project...</div>');
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_add_domain',
                    nonce: scAdmin.nonce,
                    projectName: scAdmin.siteName || 'My WordPress Site',
                    domain: scAdmin.siteUrl
                },
                success: function(response) {
                    if (response.success) {
                        // Select the newly created domain
                        SC.selectDomain(response.data.id, response.data.projectId);
                    } else {
                        // Show error at the top of the page
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                        $container.removeClass('sc-loading-overlay');
                        $container.find('.sc-selecting-loader').remove();
                    }
                },
                error: function() {
                    SC.showError('Failed to add project. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                    $container.removeClass('sc-loading-overlay');
                    $container.find('.sc-selecting-loader').remove();
                }
            });
        },

        /**
         * Handle save settings
         */
        handleSaveSettings: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Saving...');

            // Build settings object
            const settings = {
                projectName: $form.find('[name="projectName"]').val(),
                domain: $form.find('[name="domain"]').val(),
                active: $form.find('[name="active"]').is(':checked'),
                theme: {
                    primary: $form.find('[name="theme[primary]"]').val(),
                    overlay: $form.find('[name="theme[overlay]"]').val()
                },
                modalAlign: parseInt($form.find('[name="modalAlign"]').val(), 10),
                emailNotifications: $form.find('[name="emailNotifications"]').is(':checked'),
                emailFrequency: $form.find('[name="emailFrequency"]').val(),
                drawing: $form.find('[name="drawing"]').is(':checked'),
                screenshots: $form.find('[name="screenshots"]').is(':checked'),
                uploads: $form.find('[name="uploads"]').is(':checked'),
                minimized: $form.find('[name="minimized"]').is(':checked')
            };

            // Store original domain to check if it changed
            const originalDomain = scAdmin.projectDomain || '';
            const newDomain = settings.domain || '';

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_save_settings',
                    nonce: scAdmin.nonce,
                    settings: JSON.stringify(settings)
                },
                success: function(response) {
                    if (response.success) {
                        // Reload page if domain or project name changed to update header
                        if (originalDomain !== newDomain || response.data?.reloadRequired) {
                            location.reload();
                            return;
                        }
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to save settings.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle save roles
         */
        handleSaveRoles: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Saving...');

            // Get all checked roles
            const roles = [];
            $form.find('input[name="roles[]"]:checked').each(function() {
                roles.push($(this).val());
            });

            // Get total number of role checkboxes
            const totalRoles = $form.find('input[name="roles[]"]').length;

            // If all roles are selected, send empty array (means "everyone")
            const rolesToSave = (roles.length === totalRoles) ? [] : roles;

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_save_roles',
                    nonce: scAdmin.nonce,
                    roles: JSON.stringify(rolesToSave)
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to save role settings.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle change project
         */
        handleChangeProject: function() {
            // Clear project selection and reload
            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_select_domain',
                    nonce: scAdmin.nonce,
                    domainId: '',
                    projectId: ''
                },
                complete: function() {
                    location.reload();
                }
            });
        },

        /**
         * Store email for code verification
         */
        codeLoginEmail: '',

        /**
         * Handle send code form submission
         */
        handleSendCode: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();
            const email = $('#code-email').val();

            $button.prop('disabled', true).text('Sending...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_send_login_code',
                    nonce: scAdmin.nonce,
                    email: email
                },
                success: function(response) {
                    if (response.success) {
                        // Store email and show code entry step
                        SC.codeLoginEmail = email;
                        $('#code-sent-email').text(email);
                        $('#code-step-email').hide();
                        $('#code-step-verify').show();
                        $('#login-code').val('').focus();
                    } else {
                        SC.showError(response.data.message);
                    }
                    $button.prop('disabled', false).text(originalText);
                },
                error: function() {
                    SC.showError('Failed to send code. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle verify code form submission
         */
        handleVerifyCode: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();
            const code = $('#login-code').val();

            if (!SC.codeLoginEmail) {
                SC.showError('Please enter your email first.');
                return;
            }

            $button.prop('disabled', true).text('Verifying...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_verify_login_code',
                    nonce: scAdmin.nonce,
                    email: SC.codeLoginEmail,
                    code: code
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Verification failed. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle resend code
         */
        handleResendCode: function() {
            if (!SC.codeLoginEmail) {
                // Go back to email step
                $('#code-step-verify').hide();
                $('#code-step-email').show();
                return;
            }

            const $button = $('#sc-resend-code');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Sending...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_send_login_code',
                    nonce: scAdmin.nonce,
                    email: SC.codeLoginEmail
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Sent!');
                        $('#login-code').val('').focus();
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to resend code.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle register method switch (password/code)
         */
        handleRegisterMethodSwitch: function() {
            const method = $(this).data('method');

            $('.sc-register-method').removeClass('active');
            $(this).addClass('active');

            $('.sc-register-method-content').hide();
            $('#register-method-' + method).show();
        },

        /**
         * Store registration data for code verification
         */
        codeRegEmail: '',
        codeRegCompany: '',

        /**
         * Handle send registration code form submission
         */
        handleSendRegCode: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();
            const email = $('#reg-code-email').val();
            const company = $('#reg-code-company').val();

            if (!email || !company) {
                SC.showError('Email and company name are required');
                return;
            }

            $button.prop('disabled', true).text('Sending...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_send_register_code',
                    nonce: scAdmin.nonce,
                    email: email
                },
                success: function(response) {
                    if (response.success) {
                        // Store email and company for verification step
                        SC.codeRegEmail = email;
                        SC.codeRegCompany = company;
                        $('#reg-code-sent-email').text(email);
                        $('#reg-code-step-email').hide();
                        $('#reg-code-step-verify').show();
                        $('#reg-verify-code').val('').focus();
                    } else {
                        SC.showError(response.data.message);
                    }
                    $button.prop('disabled', false).text(originalText);
                },
                error: function() {
                    SC.showError('Failed to send code. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle verify registration code form submission
         */
        handleVerifyRegCode: function(e) {
            e.preventDefault();

            const $form = $(e.target);
            const $button = $form.find('button[type="submit"]');
            const originalText = $button.text();
            const code = $('#reg-verify-code').val();

            if (!SC.codeRegEmail || !SC.codeRegCompany) {
                SC.showError('Please enter your email first.');
                return;
            }

            $button.prop('disabled', true).text('Creating Account...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_verify_register_code',
                    nonce: scAdmin.nonce,
                    email: SC.codeRegEmail,
                    code: code,
                    company: SC.codeRegCompany
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Registration failed. Please try again.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle resend registration code
         */
        handleResendRegCode: function() {
            if (!SC.codeRegEmail) {
                // Go back to email step
                $('#reg-code-step-verify').hide();
                $('#reg-code-step-email').show();
                return;
            }

            const $button = $('#sc-reg-resend-code');
            const originalText = $button.text();

            $button.prop('disabled', true).text('Sending...');
            this.hideErrors();

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_send_register_code',
                    nonce: scAdmin.nonce,
                    email: SC.codeRegEmail
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Sent!');
                        $('#reg-verify-code').val('').focus();
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        SC.showError(response.data.message);
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function() {
                    SC.showError('Failed to resend code.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Show error message (as alert at top if connected, or inline if not)
         */
        showError: function(message) {
            // Try to show in global alert first (for connected state)
            const $globalAlert = $('#sc-global-alert');
            if ($globalAlert.length) {
                $globalAlert.text(message).show();
                // Scroll to top
                $('html, body').animate({
                    scrollTop: $globalAlert.offset().top - 20
                }, 300);
                return;
            }

            // Fallback to connect error (for login/register state)
            const $error = $('#sc-connect-error, #sc-global-error').first();
            $error.text(message).show();

            // Scroll to error
            $('html, body').animate({
                scrollTop: $error.offset().top - 50
            }, 300);
        },

        /**
         * Hide all error messages
         */
        hideErrors: function() {
            $('#sc-connect-error, #sc-global-error, #sc-global-alert').hide();
        },

        // ========================================
        // COMMENTS FUNCTIONALITY
        // ========================================

        /**
         * Comments state
         */
        commentsData: {
            comments: [],
            statusCounts: {},
            enabledStatuses: [],
            slugs: [],
            currentStatus: 'todo',
            currentPriority: '',
            currentSlug: '',
            offset: 0,
            limit: 20,
            total: 0,
            filtered: 0
        },

        /**
         * Initialize comments
         */
        initComments: function() {
            // Only init if comments tab exists
            if (!$('#settings-tab-comments').length) {
                return;
            }

            // Bind comment-related events
            $('#sc-priority-filter').on('change', this.handlePriorityFilter.bind(this));
            $('#sc-slug-filter').on('change', this.handleSlugFilter.bind(this));
            $('#sc-load-more').on('click', this.loadMoreComments.bind(this));

            // Load initial comments
            this.loadComments();
        },

        /**
         * Load comments from API
         */
        loadComments: function(append) {
            const self = this;

            if (!append) {
                this.commentsData.offset = 0;
                $('#sc-comments-list').html('<div class="sc-comments-loading"><span class="spinner is-active"></span> Loading comments...</div>');
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                timeout: 30000,
                data: {
                    action: 'sc_get_comments',
                    nonce: scAdmin.nonce,
                    status: this.commentsData.currentStatus,
                    limit: this.commentsData.limit,
                    offset: this.commentsData.offset
                },
                success: function(response) {
                    if (response.success) {
                        self.commentsData.statusCounts = response.data.statusCounts || {};
                        self.commentsData.enabledStatuses = response.data.enabledStatuses || [];
                        self.commentsData.slugs = response.data.slugs || [];
                        self.commentsData.total = response.data.total || 0;
                        self.commentsData.filtered = response.data.filtered || 0;

                        if (append) {
                            self.commentsData.comments = self.commentsData.comments.concat(response.data.comments || []);
                        } else {
                            self.commentsData.comments = response.data.comments || [];
                        }

                        self.renderStatusTabs();
                        self.renderSlugFilter();
                        self.renderComments();
                        self.updatePagination();

                        // Mark comments as seen (update last seen count)
                        self.markCommentsAsSeen();
                    } else {
                        $('#sc-comments-list').html('<div class="sc-comments-empty"><p>' + (response.data?.message || 'Failed to load comments') + '</p></div>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Comments load error:', status, error);
                    $('#sc-comments-list').html('<div class="sc-comments-empty"><p>Failed to load comments. Please try again.</p></div>');
                }
            });
        },

        /**
         * Mark comments as seen (update last seen todo count)
         */
        markCommentsAsSeen: function() {
            const todoCount = this.commentsData.statusCounts['todo'] || 0;

            // Silently update the last seen count
            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'sc_mark_comments_seen',
                    nonce: scAdmin.nonce,
                    todoCount: todoCount
                },
                success: function() {
                    // Update the menu badge in the sidebar (remove it)
                    const $menuItem = $('#adminmenu .toplevel_page_simple-commenter .awaiting-mod');
                    if ($menuItem.length) {
                        $menuItem.remove();
                    }
                }
            });
        },

        /**
         * Render status tabs with counts
         */
        renderStatusTabs: function() {
            const statuses = this.commentsData.enabledStatuses.length > 0
                ? this.commentsData.enabledStatuses
                : ['todo', 'review', 'done'];  // Match schema default

            const statusLabels = {
                'todo': 'To Do',
                'in_progress': 'In Progress',
                'review': 'Review',
                'rework': 'Rework',
                'on_hold': 'On Hold',
                'blocked': 'Blocked',
                'done': 'Done',
                'cancelled': 'Cancelled'
            };

            let html = '';
            statuses.forEach(function(status) {
                const count = SC.commentsData.statusCounts[status] || 0;
                const isActive = SC.commentsData.currentStatus === status;
                html += '<button type="button" class="sc-status-tab' + (isActive ? ' active' : '') + '" data-status="' + status + '">';
                html += statusLabels[status] || status;
                html += '<span class="sc-status-count">' + count + '</span>';
                html += '</button>';
            });

            $('#sc-status-tabs').html(html);

            // Bind click events
            $('.sc-status-tab').on('click', function() {
                SC.commentsData.currentStatus = $(this).data('status');
                $('.sc-status-tab').removeClass('active');
                $(this).addClass('active');
                SC.loadComments();
            });
        },

        /**
         * Render slug filter dropdown
         */
        renderSlugFilter: function() {
            const $select = $('#sc-slug-filter');
            const currentVal = $select.val();

            $select.find('option:not(:first)').remove();

            this.commentsData.slugs.forEach(function(slug) {
                $select.append('<option value="' + slug + '">' + slug + '</option>');
            });

            if (currentVal && this.commentsData.slugs.includes(currentVal)) {
                $select.val(currentVal);
            }
        },

        /**
         * Render comments list
         */
        renderComments: function() {
            const comments = this.getFilteredComments();

            if (comments.length === 0) {
                $('#sc-comments-list').html('<div class="sc-comments-empty"><p>No comments found.</p></div>');
                return;
            }

            let html = '';
            comments.forEach(function(comment) {
                html += SC.renderCommentCard(comment);
            });

            $('#sc-comments-list').html(html);

            // Bind events
            this.bindCommentEvents();
        },

        /**
         * Get comments filtered by frontend filters (priority, slug)
         */
        getFilteredComments: function() {
            let comments = this.commentsData.comments;

            // Filter by priority
            if (this.commentsData.currentPriority) {
                comments = comments.filter(function(c) {
                    return c.priority === SC.commentsData.currentPriority;
                });
            }

            // Filter by slug
            if (this.commentsData.currentSlug) {
                comments = comments.filter(function(c) {
                    return c.slug === SC.commentsData.currentSlug;
                });
            }

            return comments;
        },

        /**
         * Render a single comment card
         */
        renderCommentCard: function(comment) {
            const statusLabels = {
                'todo': 'To Do',
                'in_progress': 'In Progress',
                'review': 'Review',
                'rework': 'Rework',
                'on_hold': 'On Hold',
                'blocked': 'Blocked',
                'done': 'Done',
                'cancelled': 'Cancelled'
            };

            const date = new Date(comment.date);
            const dateStr = date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            const priority = comment.priority || 'normal';

            let html = '<div class="sc-comment-card" data-id="' + comment.id + '">';

            // Header - comment number and priority badge left-justified
            const priorityLabels = {
                'low': 'Low',
                'normal': 'Normal',
                'high': 'High'
            };

            html += '<div class="sc-comment-header">';
            html += '<span class="sc-comment-number">#' + comment.commentNumber + '</span>';
            html += '<select class="sc-priority-select" data-comment-id="' + comment.id + '">';
            Object.keys(priorityLabels).forEach(function(p) {
                const selected = priority === p ? ' selected' : '';
                html += '<option value="' + p + '"' + selected + '>' + priorityLabels[p] + '</option>';
            });
            html += '</select>';
            if (comment.slug) {
                html += '<span class="sc-comment-slug">' + comment.slug + '</span>';
            }
            html += '</div>';

            // Content
            html += '<div class="sc-comment-content">';
            if (comment.title) {
                html += '<h4 class="sc-comment-title">' + this.escapeHtml(comment.title) + '</h4>';
            }
            html += '<p class="sc-comment-text">' + this.escapeHtml(comment.text || '') + '</p>';

            // Screenshot inside content
            if (comment.screenshot && comment.screenshot.url) {
                html += '<div class="sc-comment-screenshot">';
                html += '<a href="' + comment.screenshot.url + '" target="_blank">';
                html += '<img src="' + (comment.screenshot.thumbnailUrl || comment.screenshot.url) + '" alt="Screenshot">';
                html += '</a>';
                html += '</div>';
            }

            // Attachments inside content
            if (comment.attachments && comment.attachments.length > 0) {
                html += '<div class="sc-comment-attachments">';
                comment.attachments.forEach(function(att) {
                    html += '<a href="' + att.url + '" target="_blank" class="sc-attachment">';
                    html += '<span class="sc-attachment-icon">📎</span>';
                    html += '<span class="sc-attachment-name">' + SC.escapeHtml(att.name) + '</span>';
                    html += '</a>';
                });
                html += '</div>';
            }

            // Replies section - inside content
            html += '<div class="sc-comment-replies" data-comment-id="' + comment.id + '">';
            html += '<button type="button" class="sc-toggle-replies">';
            html += 'Replies (' + (comment.replies?.length || 0) + ')';
            html += '</button>';
            html += '<div class="sc-replies-list" style="display:none;">';

            if (comment.replies && comment.replies.length > 0) {
                comment.replies.forEach(function(reply) {
                    html += SC.renderReply(reply);
                });
            }

            // Reply form
            html += '<div class="sc-reply-form">';
            html += '<textarea class="sc-reply-text" placeholder="Write a reply..."></textarea>';
            html += '<div class="sc-reply-actions">';
            html += '<label class="sc-file-upload">';
            html += '<input type="file" class="sc-reply-file" accept="image/*,.pdf,.doc,.docx">';
            html += '<span class="sc-attach-label">📎 Attach file</span>';
            html += '</label>';
            html += '<div class="sc-file-indicator" style="display:none;">';
            html += '<span class="sc-file-indicator-icon">📄</span>';
            html += '<span class="sc-file-indicator-name"></span>';
            html += '<button type="button" class="sc-file-remove" title="Remove file">&times;</button>';
            html += '</div>';
            html += '<button type="button" class="sc-submit-reply">Reply</button>';
            html += '</div>';
            html += '</div>';

            html += '</div>'; // .sc-replies-list
            html += '</div>'; // .sc-comment-replies

            html += '</div>'; // .sc-comment-content

            // Footer - meta and actions combined
            html += '<div class="sc-comment-footer">';
            html += '<div class="sc-comment-meta">';
            html += '<span class="sc-comment-author">' + this.escapeHtml(comment.author?.name || 'Anonymous') + '</span>';
            html += '<span class="sc-comment-date">' + dateStr + '</span>';
            if (comment.metadata?.browser) {
                html += '<span class="sc-comment-browser">' + comment.metadata.browser + '</span>';
            }
            html += '</div>';
            html += '<div class="sc-comment-actions">';
            html += '<select class="sc-status-select" data-comment-id="' + comment.id + '">';
            Object.keys(statusLabels).forEach(function(status) {
                const selected = comment.status === status ? ' selected' : '';
                html += '<option value="' + status + '"' + selected + '>' + statusLabels[status] + '</option>';
            });
            html += '</select>';
            html += '</div>';
            html += '</div>';

            html += '</div>'; // .sc-comment-card

            return html;
        },

        /**
         * Render a reply
         */
        renderReply: function(reply) {
            const date = new Date(reply.date);
            const dateStr = date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});

            let html = '<div class="sc-reply">';
            html += '<div class="sc-reply-header">';
            html += '<span class="sc-reply-author">' + this.escapeHtml(reply.author?.name || 'Unknown') + '</span>';
            html += '<span class="sc-reply-date">' + dateStr + '</span>';
            html += '</div>';
            html += '<p class="sc-reply-text">' + this.escapeHtml(reply.text || '') + '</p>';

            if (reply.attachments && reply.attachments.length > 0) {
                html += '<div class="sc-reply-attachments">';
                reply.attachments.forEach(function(att) {
                    html += '<a href="' + att.url + '" target="_blank" class="sc-attachment">';
                    html += '<span class="sc-attachment-name">' + SC.escapeHtml(att.name) + '</span>';
                    html += '</a>';
                });
                html += '</div>';
            }

            html += '</div>';
            return html;
        },

        /**
         * Bind comment-related events
         */
        bindCommentEvents: function() {
            // Status change
            $('.sc-status-select').off('change').on('change', function() {
                const commentId = $(this).data('comment-id');
                const status = $(this).val();
                SC.updateComment(commentId, status, null);
            });

            // Priority change
            $('.sc-priority-select').off('change').on('change', function() {
                const commentId = $(this).data('comment-id');
                const priority = $(this).val();
                SC.updateComment(commentId, null, priority);
            });

            // Toggle replies
            $('.sc-toggle-replies').off('click').on('click', function() {
                $(this).siblings('.sc-replies-list').slideToggle(200);
            });

            // Submit reply
            $('.sc-submit-reply').off('click').on('click', function() {
                const $card = $(this).closest('.sc-comment-card');
                const commentId = $card.data('id');
                // Use textarea selector to avoid matching <p class="sc-reply-text"> from rendered replies
                const $textarea = $card.find('.sc-reply-form textarea.sc-reply-text');
                const $fileInput = $card.find('.sc-reply-form .sc-reply-file');
                const text = $textarea.val();

                if (!text.trim()) {
                    alert('Please enter a reply');
                    return;
                }

                SC.submitReply(commentId, text, $fileInput[0].files[0], $card);
            });

            // File selection feedback
            $('.sc-reply-file').off('change').on('change', function() {
                const $actions = $(this).closest('.sc-reply-actions');
                const $indicator = $actions.find('.sc-file-indicator');
                const $label = $actions.find('.sc-file-upload');
                if (this.files && this.files.length > 0) {
                    const fileName = this.files[0].name;
                    // Truncate long filenames
                    const displayName = fileName.length > 25 ? fileName.substring(0, 22) + '...' : fileName;
                    $indicator.find('.sc-file-indicator-name').text(displayName);
                    $indicator.show();
                    $label.hide();
                } else {
                    $indicator.hide();
                    $label.show();
                }
            });

            // Remove file button
            $('.sc-file-remove').off('click').on('click', function(e) {
                e.preventDefault();
                const $actions = $(this).closest('.sc-reply-actions');
                const $fileInput = $actions.find('.sc-reply-file');
                const $indicator = $actions.find('.sc-file-indicator');
                const $label = $actions.find('.sc-file-upload');

                // Clear the file input
                $fileInput.val('');
                $indicator.hide();
                $label.show();
            });
        },

        /**
         * Update comment status or priority
         */
        updateComment: function(commentId, status, priority) {
            const data = {
                action: 'sc_update_comment',
                nonce: scAdmin.nonce,
                commentId: commentId
            };

            if (status) data.status = status;
            if (priority) data.priority = priority;

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: data,
                success: function(response) {
                    if (response.success) {
                        // Reload comments to reflect changes
                        if (status) {
                            SC.loadComments();
                        }
                    } else {
                        alert(response.data?.message || 'Failed to update comment');
                    }
                },
                error: function() {
                    alert('Failed to update comment');
                }
            });
        },

        /**
         * Submit a reply
         */
        submitReply: function(commentId, text, file, $card) {
            const $button = $card.find('.sc-submit-reply');
            const originalText = $button.text();
            $button.prop('disabled', true).text('Sending...');

            const formData = new FormData();
            formData.append('action', 'sc_add_reply');
            formData.append('nonce', scAdmin.nonce);
            formData.append('commentId', commentId);
            formData.append('text', text);

            if (file) {
                formData.append('file', file);
            }

            $.ajax({
                url: scAdmin.ajaxUrl,
                method: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        // Add reply to UI
                        const $repliesList = $card.find('.sc-replies-list');
                        const replyHtml = SC.renderReply(response.data.reply);
                        $repliesList.find('.sc-reply-form').before(replyHtml);

                        // Clear form
                        $card.find('textarea.sc-reply-text').val('');
                        $card.find('.sc-reply-file').val('');
                        $card.find('.sc-file-indicator').hide();
                        $card.find('.sc-file-upload').show();

                        // Update reply count
                        const $toggle = $card.find('.sc-toggle-replies');
                        const currentCount = parseInt($toggle.text().match(/\d+/)[0]) || 0;
                        $toggle.text('Replies (' + (currentCount + 1) + ')');

                        // Make sure replies are visible
                        $repliesList.show();
                    } else {
                        alert(response.data?.message || 'Failed to add reply');
                    }
                    $button.prop('disabled', false).text(originalText);
                },
                error: function() {
                    alert('Failed to add reply');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        },

        /**
         * Handle priority filter change
         */
        handlePriorityFilter: function() {
            this.commentsData.currentPriority = $('#sc-priority-filter').val();
            this.renderComments();
        },

        /**
         * Handle slug filter change
         */
        handleSlugFilter: function() {
            this.commentsData.currentSlug = $('#sc-slug-filter').val();
            this.renderComments();
        },

        /**
         * Load more comments
         */
        loadMoreComments: function() {
            this.commentsData.offset += this.commentsData.limit;
            this.loadComments(true);
        },

        /**
         * Update pagination UI
         */
        updatePagination: function() {
            const showing = this.commentsData.comments.length;
            const total = this.commentsData.filtered;

            if (showing < total) {
                $('#sc-comments-pagination').show();
                $('#sc-comments-count').text('Showing ' + showing + ' of ' + total);
            } else {
                $('#sc-comments-pagination').hide();
            }
        },

        /**
         * Escape HTML special characters
         */
        escapeHtml: function(text) {
            if (!text) return '';
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        SC.init();
    });

})(jQuery);
