<?php
/**
 * SimpleCommenter Admin Controller
 *
 * Handles the WordPress admin interface for SimpleCommenter
 */

if (!defined('ABSPATH')) {
    exit;
}

class SC_Admin {

    /**
     * API client instance
     */
    private $api;

    /**
     * Settings manager instance
     */
    private $settings;

    /**
     * Constructor
     *
     * @param SC_API $api API client
     * @param SC_Settings $settings Settings manager
     */
    public function __construct($api, $settings) {
        $this->api = $api;
        $this->settings = $settings;
    }

    /**
     * Initialize admin
     */
    public function init() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));

        // AJAX handlers
        add_action('wp_ajax_sc_connect', array($this, 'ajax_connect'));
        add_action('wp_ajax_sc_register', array($this, 'ajax_register'));
        add_action('wp_ajax_sc_get_domains', array($this, 'ajax_get_domains'));
        add_action('wp_ajax_sc_select_domain', array($this, 'ajax_select_domain'));
        add_action('wp_ajax_sc_add_domain', array($this, 'ajax_add_domain'));
        add_action('wp_ajax_sc_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_sc_disconnect', array($this, 'ajax_disconnect'));
        add_action('wp_ajax_sc_refresh_account', array($this, 'ajax_refresh_account'));
        add_action('wp_ajax_sc_get_login_link', array($this, 'ajax_get_login_link'));
        add_action('wp_ajax_sc_save_google_auth', array($this, 'ajax_save_google_auth'));
        add_action('wp_ajax_sc_save_roles', array($this, 'ajax_save_roles'));
        add_action('wp_ajax_sc_get_members', array($this, 'ajax_get_members'));
        add_action('wp_ajax_sc_add_member', array($this, 'ajax_add_member'));
        add_action('wp_ajax_sc_remove_member', array($this, 'ajax_remove_member'));
        add_action('wp_ajax_sc_get_clients', array($this, 'ajax_get_clients'));
        add_action('wp_ajax_sc_add_client', array($this, 'ajax_add_client'));
        add_action('wp_ajax_sc_save_sync_settings', array($this, 'ajax_save_sync_settings'));
        add_action('wp_ajax_sc_save_admin_roles', array($this, 'ajax_save_admin_roles'));
        add_action('wp_ajax_sc_send_login_code', array($this, 'ajax_send_login_code'));
        add_action('wp_ajax_sc_verify_login_code', array($this, 'ajax_verify_login_code'));
        add_action('wp_ajax_sc_send_register_code', array($this, 'ajax_send_register_code'));
        add_action('wp_ajax_sc_verify_register_code', array($this, 'ajax_verify_register_code'));
        add_action('wp_ajax_sc_get_comments', array($this, 'ajax_get_comments'));
        add_action('wp_ajax_sc_update_comment', array($this, 'ajax_update_comment'));
        add_action('wp_ajax_sc_add_reply', array($this, 'ajax_add_reply'));
        add_action('wp_ajax_sc_mark_comments_seen', array($this, 'ajax_mark_comments_seen'));
        add_action('wp_ajax_sc_sync_all_users', array($this, 'ajax_sync_all_users'));
        add_action('wp_ajax_sc_get_upgrade_link', array($this, 'ajax_get_upgrade_link'));
        add_action('wp_ajax_sc_sync_current_user', array($this, 'ajax_sync_current_user'));
    }

    /**
     * Add admin menu
     */
    public function add_menu() {
        // Custom SVG icon (SimpleCommenter mono logo - uses currentColor for WordPress menu theming)
        $icon_svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 82 72" fill="currentColor"><path d="M70.3,2.6c-19.2-2.8-39.4-2.8-58.6,0C5.3,3.5.5,9.2.5,16v25.2c0,6.8,4.7,12.4,11.3,13.4,4.1.6,8.4,1,12.7,1.4.3,0,.6,0,.9,0v12.7c0,1.1.7,2.1,1.7,2.5.3.1.7.2,1.1.2.7,0,1.4-.3,1.9-.8l12.4-12.4c.6-.6,1.3-1,1.9-1.3.9-.3,1.6-.5,2.3-.5,7.6-.2,15.6-.9,23.6-2,6.4-.9,11.3-6.7,11.3-13.4V16c0-6.8-4.8-12.5-11.2-13.4ZM76,18.9c-1.5-2.8-4.3-5-7.8-5.4-7.7-1.1-15.5-1.6-23.4-1.6s-15.8.5-23.5,1.6c-5.3.7-9,5.1-9,10.1v18.9c0,2.7,1.1,5.2,2.9,7-.9-.1-1.8-.2-2.6-.3-3.8-.6-6.6-3.9-6.6-8V16c0-4,2.8-7.5,6.5-8,9.3-1.4,18.9-2.1,28.5-2.1s19.2.7,28.5,2.1c3.7.5,6.5,3.9,6.5,8v2.9Z"/></svg>';
        $icon_base64 = 'data:image/svg+xml;base64,' . base64_encode($icon_svg);

        // Get new comments count for badge
        $badge_count = $this->get_new_comments_count();
        $menu_title = __('Commenter', 'simple-commenter');
        if ($badge_count > 0) {
            $menu_title .= ' <span class="awaiting-mod">' . $badge_count . '</span>';
        }

        add_menu_page(
            __('Simple Commenter', 'simple-commenter'),
            $menu_title,
            'read',  // Allow all logged-in users to see the menu
            'simple-commenter',
            array($this, 'render_page'),
            $icon_base64,
            80
        );
    }

    /**
     * Get count of new comments since last seen
     *
     * @return int New comments count
     */
    private function get_new_comments_count() {
        // Only check if connected and has project
        if (!$this->settings->is_connected() || empty($this->settings->get_project_id())) {
            return 0;
        }

        // Use transient to cache the count (check every 5 minutes)
        $cache_key = 'sc_new_comments_count';
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return (int) $cached;
        }

        // Fetch current todo count from API
        $domain_id = $this->settings->get_domain_id();
        $result = $this->api->get_comments($domain_id, 'todo', 1, 0);

        if (isset($result['error'])) {
            return 0;
        }

        $current_todo_count = isset($result['statusCounts']['todo']) ? (int) $result['statusCounts']['todo'] : 0;
        $last_seen_count = $this->settings->get_last_seen_todo_count();

        // Calculate new comments (difference between current and last seen)
        $new_count = max(0, $current_todo_count - $last_seen_count);

        // Cache for 5 minutes
        set_transient($cache_key, $new_count, 5 * MINUTE_IN_SECONDS);

        return $new_count;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_assets($hook) {
        if ($hook !== 'toplevel_page_simple-commenter') {
            return;
        }

        wp_enqueue_style(
            'sc-admin',
            SC_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SC_VERSION
        );

        wp_enqueue_script(
            'sc-admin',
            SC_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            SC_VERSION,
            true
        );

        // Get current admin page URL for Google OAuth callback
        $current_page_url = admin_url('admin.php?page=simple-commenter');

        // Get domain settings for JS
        $domain_settings = $this->settings->get_domain_settings();
        $project_domain = isset($domain_settings['domain']) ? $domain_settings['domain'] : '';

        // Get WordPress users for matching with SC members/clients
        $wp_users = $this->get_wordpress_users();

        wp_localize_script('sc-admin', 'scAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sc_admin_nonce'),
            'siteUrl' => get_site_url(),
            'siteName' => get_bloginfo('name'),
            'projectDomain' => $project_domain,
            'googleAuthUrl' => $this->api->get_google_auth_url($current_page_url, 'login'),
            'googleRegisterUrl' => $this->api->get_google_auth_url($current_page_url, 'register'),
            'wpUsers' => $wp_users,
        ));
    }

    /**
     * Get WordPress users for matching with SC members/clients
     *
     * @return array Array of user emails and their roles
     */
    private function get_wordpress_users() {
        $users = get_users(array(
            'fields' => array('ID', 'user_email', 'display_name'),
        ));

        $wp_users = array();
        foreach ($users as $user) {
            $user_obj = get_userdata($user->ID);
            $wp_users[] = array(
                'email' => strtolower($user->user_email),
                'name' => $user->display_name,
                'role' => !empty($user_obj->roles) ? $user_obj->roles[0] : 'subscriber',
            );
        }

        return $wp_users;
    }

    /**
     * Render admin page
     */
    public function render_page() {
        $connected = $this->settings->is_connected();
        $account = $this->settings->get_account_info();
        $trial = $this->settings->get_trial_info();
        $project_id = $this->settings->get_project_id();
        $domain_id = $this->settings->get_domain_id();
        $domain_settings = $this->settings->get_domain_settings();
        $enabled_roles = $this->settings->get_enabled_roles();
        $auto_sync_clients = $this->settings->get_auto_sync_clients();
        $admin_roles = $this->settings->get_admin_roles();
        $can_manage = $this->settings->current_user_can_manage();

        // Get current WordPress user info
        $wp_user = wp_get_current_user();
        $wp_user_email = $wp_user->user_email;
        $wp_user_name = $wp_user->display_name;
        $wp_user_role = !empty($wp_user->roles) ? $wp_user->roles[0] : 'subscriber';
        $wp_user_is_admin = current_user_can('administrator');

        // Check if current user is synced to SimpleCommenter
        $wp_user_is_synced = (bool) get_transient('sc_synced_user_' . $wp_user->ID);

        include SC_PLUGIN_DIR . 'templates/admin-page.php';
    }

    /**
     * Verify AJAX request - basic authentication for all logged-in users
     */
    private function verify_ajax_read() {
        if (!check_ajax_referer('sc_admin_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => 'Invalid security token'));
            exit;
        }

        if (!is_user_logged_in()) {
            wp_send_json_error(array('message' => 'You must be logged in'));
            exit;
        }
    }

    /**
     * Verify AJAX request - requires management permission for admin actions
     * Uses the sc_admin_roles setting configured in Management Permissions
     */
    private function verify_ajax_admin() {
        $this->verify_ajax_read();

        if (!$this->settings->current_user_can_manage()) {
            wp_send_json_error(array('message' => 'Permission denied. You need management permissions.'));
            exit;
        }
    }

    /**
     * Verify AJAX request (legacy - uses admin verification)
     */
    private function verify_ajax() {
        $this->verify_ajax_admin();
    }

    /**
     * AJAX: Connect existing account
     */
    public function ajax_connect() {
        $this->verify_ajax();

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Password must not be sanitized
        $password = isset( $_POST['password'] ) ? wp_unslash( $_POST['password'] ) : '';

        if (empty($email) || empty($password)) {
            wp_send_json_error(array('message' => 'Email and password are required'));
        }

        $result = $this->api->connect($email, $password);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        // Save connection
        $this->settings->save_connection($result['token'], $result['account']);

        wp_send_json_success(array(
            'account' => $result['account'],
            'domains' => isset($result['domains']) ? $result['domains'] : array(),
        ));
    }

    /**
     * AJAX: Register new account
     */
    public function ajax_register() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Password must not be sanitized
        $password = isset( $_POST['password'] ) ? wp_unslash( $_POST['password'] ) : '';
        $company = isset( $_POST['company'] ) ? sanitize_text_field( wp_unslash( $_POST['company'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($email) || empty($password) || empty($company)) {
            wp_send_json_error(array('message' => 'All fields are required'));
        }

        $result = $this->api->register($email, $password, $company, get_site_url());

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        // Save connection
        $this->settings->save_connection($result['token'], $result['account']);

        // If domain was auto-created, save it
        if (isset($result['domain']) && $result['domain']) {
            $domain = $result['domain'];
            $this->settings->save_domain($domain['id'], $domain['projectId'], array(
                'projectName' => $domain['projectName'],
                'domain' => $domain['domain'],
                'active' => true,
            ));
        }

        wp_send_json_success(array(
            'account' => $result['account'],
            'domain' => isset($result['domain']) ? $result['domain'] : null,
        ));
    }

    /**
     * AJAX: Get domains list
     */
    public function ajax_get_domains() {
        $this->verify_ajax();

        $result = $this->api->get_domains();

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success(array(
            'domains' => isset($result['domains']) ? $result['domains'] : array(),
        ));
    }

    /**
     * AJAX: Select domain
     */
    public function ajax_select_domain() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $domain_id = isset( $_POST['domainId'] ) ? sanitize_text_field( wp_unslash( $_POST['domainId'] ) ) : '';
        $project_id = isset( $_POST['projectId'] ) ? sanitize_text_field( wp_unslash( $_POST['projectId'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        // If both are empty, clear the domain selection (used by "Change Project")
        if (empty($domain_id) && empty($project_id)) {
            $this->settings->clear_domain();
            wp_send_json_success(array('cleared' => true));
            return;
        }

        if (empty($domain_id) || empty($project_id)) {
            wp_send_json_error(array('message' => 'Domain ID and Project ID are required'));
        }

        // Fetch domain settings
        $settings = $this->api->get_domain_settings($domain_id);

        if (isset($settings['error'])) {
            wp_send_json_error(array('message' => $settings['error']));
        }

        // Save domain selection
        $this->settings->save_domain($domain_id, $project_id, $settings);

        wp_send_json_success(array('settings' => $settings));
    }

    /**
     * AJAX: Add new domain
     */
    public function ajax_add_domain() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $project_name = isset( $_POST['projectName'] ) ? sanitize_text_field( wp_unslash( $_POST['projectName'] ) ) : '';
        $domain = isset( $_POST['domain'] ) ? esc_url_raw( wp_unslash( $_POST['domain'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($project_name)) {
            wp_send_json_error(array('message' => 'Project name is required'));
        }

        $result = $this->api->add_domain($project_name, $domain);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Save settings
     */
    public function ajax_save_settings() {
        $this->verify_ajax();

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No domain selected'));
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified in verify_ajax(), JSON decoded below
        $settings_json = isset( $_POST['settings'] ) ? wp_unslash( $_POST['settings'] ) : '{}';
        $settings = json_decode( $settings_json, true );

        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error(array('message' => 'Invalid settings format'));
        }

        $result = $this->api->update_domain_settings($domain_id, $settings);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        // Update local settings cache
        if (isset($result['settings'])) {
            $this->settings->save_domain(
                $domain_id,
                $this->settings->get_project_id(),
                $result['settings']
            );
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Disconnect
     */
    public function ajax_disconnect() {
        $this->verify_ajax();

        // Check if user has permission to manage
        if (!$this->settings->current_user_can_manage()) {
            wp_send_json_error(array('message' => 'You do not have permission to disconnect Simple Commenter.'));
            return;
        }

        $this->api->disconnect();
        $this->settings->clear_all();

        wp_send_json_success();
    }

    /**
     * AJAX: Refresh account info
     */
    public function ajax_refresh_account() {
        $this->verify_ajax();

        $result = $this->api->get_account_info();

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        update_option('sc_account_info', $result);

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get login link for dashboard auto-login
     *
     * Role mapping:
     * - WordPress Administrator -> SimpleCommenter member (role: "user")
     * - WordPress user with SC management permission -> SimpleCommenter member (role: "team")
     * - Everyone else -> SimpleCommenter client
     */
    public function ajax_get_login_link() {
        $this->verify_ajax_read();  // All logged-in users can get a login link (members and clients)

        $current_user = wp_get_current_user();
        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        // Get user's primary role
        $wp_role = !empty($current_user->roles) ? $current_user->roles[0] : 'subscriber';

        // Check if user is WP administrator
        $is_wp_admin = current_user_can('administrator');

        // Check if user has SC management permission (based on admin_roles setting)
        $has_manage_permission = $this->settings->current_user_can_manage();

        // Use the new wp-login endpoint with proper role mapping
        $result = $this->api->get_wp_login_link(
            $current_user->user_email,
            $current_user->display_name,
            $wp_role,
            $is_wp_admin,
            $has_manage_permission,
            $domain_id
        );

        if (isset($result['error'])) {
            // Provide user-friendly error messages
            $message = $result['error'];
            if (isset($result['code']) && $result['code'] === 'MEMBER_LIMIT_REACHED') {
                $message = 'Team member limit reached. Please ask the account owner to upgrade or remove an existing member.';
            }
            wp_send_json_error(array('message' => $message));
            return;
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Save Google authentication result
     */
    public function ajax_save_google_auth() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $token = isset( $_POST['token'] ) ? sanitize_text_field( wp_unslash( $_POST['token'] ) ) : '';
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- JSON decoded below
        $account_json = isset( $_POST['account'] ) ? wp_unslash( $_POST['account'] ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($token)) {
            wp_send_json_error(array('message' => 'Token is required'));
        }

        $account = json_decode($account_json, true);

        if (json_last_error() !== JSON_ERROR_NONE || !$account) {
            wp_send_json_error(array('message' => 'Invalid account data'));
        }

        // Save connection
        $this->settings->save_connection($token, $account);

        // If domain was included, save it
        if (isset($account['domain']) && $account['domain']) {
            $domain = $account['domain'];
            $this->settings->save_domain($domain['id'], $domain['projectId'], array(
                'projectName' => isset($domain['projectName']) ? $domain['projectName'] : '',
                'domain' => isset($domain['domain']) ? $domain['domain'] : '',
                'active' => true,
            ));
        }

        wp_send_json_success(array('account' => $account));
    }

    /**
     * AJAX: Save enabled WordPress roles
     */
    public function ajax_save_roles() {
        $this->verify_ajax();

        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified in verify_ajax(), JSON decoded below
        $roles_json = isset( $_POST['roles'] ) ? wp_unslash( $_POST['roles'] ) : '[]';
        $roles = json_decode( $roles_json, true );

        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error(array('message' => 'Invalid roles format'));
        }

        // Sanitize role names
        $sanitized_roles = array();
        foreach ($roles as $role) {
            $sanitized_roles[] = sanitize_text_field($role);
        }

        $this->settings->save_enabled_roles($sanitized_roles);

        wp_send_json_success(array('roles' => $sanitized_roles));
    }

    /**
     * AJAX: Get team members
     */
    public function ajax_get_members() {
        $this->verify_ajax();

        $result = $this->api->get_members();

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Add team member
     */
    public function ajax_add_member() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
        $role = isset( $_POST['role'] ) ? sanitize_text_field( wp_unslash( $_POST['role'] ) ) : 'team';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($email)) {
            wp_send_json_error(array('message' => 'Email is required'));
        }

        $result = $this->api->add_member($email, $name, $role);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Remove team member
     */
    public function ajax_remove_member() {
        $this->verify_ajax();

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $member_id = isset( $_POST['memberId'] ) ? sanitize_text_field( wp_unslash( $_POST['memberId'] ) ) : '';

        if (empty($member_id)) {
            wp_send_json_error(array('message' => 'Member ID is required'));
        }

        $result = $this->api->remove_member($member_id);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Get clients for current domain
     */
    public function ajax_get_clients() {
        $this->verify_ajax();

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        $result = $this->api->get_clients($domain_id);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Add client to current domain
     */
    public function ajax_add_client() {
        $this->verify_ajax();

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($email)) {
            wp_send_json_error(array('message' => 'Email is required'));
            return;
        }

        $result = $this->api->sync_client($domain_id, $email, $name, true);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Save sync settings (auto-sync clients)
     */
    public function ajax_save_sync_settings() {
        $this->verify_ajax();

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $auto_sync = isset( $_POST['autoSync'] ) && 'true' === $_POST['autoSync'];

        $this->settings->save_auto_sync_clients($auto_sync);

        wp_send_json_success(array('autoSync' => $auto_sync));
    }

    /**
     * AJAX: Save admin roles (who can manage/disconnect)
     */
    public function ajax_save_admin_roles() {
        $this->verify_ajax();

        // Only administrators can change admin roles
        if (!current_user_can('administrator')) {
            wp_send_json_error(array('message' => 'Only administrators can change management roles.'));
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified in verify_ajax(), JSON decoded below
        $roles_json = isset( $_POST['roles'] ) ? wp_unslash( $_POST['roles'] ) : '[]';
        $roles = json_decode( $roles_json, true );

        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error(array('message' => 'Invalid roles format'));
            return;
        }

        // Sanitize role names
        $sanitized_roles = array();
        foreach ($roles as $role) {
            $sanitized_roles[] = sanitize_text_field($role);
        }

        $this->settings->save_admin_roles($sanitized_roles);

        wp_send_json_success(array('roles' => $sanitized_roles));
    }

    /**
     * AJAX: Send login code to email
     */
    public function ajax_send_login_code() {
        $this->verify_ajax();

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';

        if (empty($email)) {
            wp_send_json_error(array('message' => 'Email is required'));
        }

        $result = $this->api->send_login_code($email);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Verify login code
     */
    public function ajax_verify_login_code() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $code = isset( $_POST['code'] ) ? sanitize_text_field( wp_unslash( $_POST['code'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($email) || empty($code)) {
            wp_send_json_error(array('message' => 'Email and code are required'));
        }

        $result = $this->api->verify_login_code($email, $code);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        // Save connection
        $this->settings->save_connection($result['token'], $result['account']);

        wp_send_json_success(array(
            'account' => $result['account'],
            'domains' => isset($result['domains']) ? $result['domains'] : array(),
        ));
    }

    /**
     * AJAX: Send registration code to email
     */
    public function ajax_send_register_code() {
        $this->verify_ajax();

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';

        if (empty($email)) {
            wp_send_json_error(array('message' => 'Email is required'));
        }

        $result = $this->api->send_register_code($email);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Verify registration code and create account
     */
    public function ajax_verify_register_code() {
        $this->verify_ajax();

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax()
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
        $code = isset( $_POST['code'] ) ? sanitize_text_field( wp_unslash( $_POST['code'] ) ) : '';
        $company = isset( $_POST['company'] ) ? sanitize_text_field( wp_unslash( $_POST['company'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($email) || empty($code)) {
            wp_send_json_error(array('message' => 'Email and code are required'));
        }

        if (empty($company)) {
            wp_send_json_error(array('message' => 'Company name is required'));
        }

        $result = $this->api->verify_register_code($email, $code, $company, get_site_url());

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        }

        // Save connection
        $this->settings->save_connection($result['token'], $result['account']);

        // If domain was auto-created, save it
        if (isset($result['domain']) && $result['domain']) {
            $domain = $result['domain'];
            $this->settings->save_domain($domain['id'], $domain['projectId'], array(
                'projectName' => $domain['projectName'],
                'domain' => $domain['domain'],
                'active' => true,
            ));
        }

        wp_send_json_success(array(
            'account' => $result['account'],
            'domain' => isset($result['domain']) ? $result['domain'] : null,
        ));
    }

    /**
     * AJAX: Get comments for domain
     */
    public function ajax_get_comments() {
        $this->verify_ajax_read();  // All logged-in users can view comments

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_read()
        $status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : '';
        $limit = isset( $_POST['limit'] ) ? intval( $_POST['limit'] ) : 50;
        $offset = isset( $_POST['offset'] ) ? intval( $_POST['offset'] ) : 0;
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        $result = $this->api->get_comments($domain_id, $status, $limit, $offset);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
            return;
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Update comment status or priority
     */
    public function ajax_update_comment() {
        $this->verify_ajax_read();  // All logged-in users can update status/priority

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_read()
        $comment_id = isset( $_POST['commentId'] ) ? sanitize_text_field( wp_unslash( $_POST['commentId'] ) ) : '';
        $status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : null;
        $priority = isset( $_POST['priority'] ) ? sanitize_text_field( wp_unslash( $_POST['priority'] ) ) : null;
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($comment_id)) {
            wp_send_json_error(array('message' => 'Comment ID is required'));
            return;
        }

        if (empty($status) && empty($priority)) {
            wp_send_json_error(array('message' => 'Status or priority is required'));
            return;
        }

        $result = $this->api->update_comment($domain_id, $comment_id, $status, $priority);

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
            return;
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Add reply to comment
     */
    public function ajax_add_reply() {
        $this->verify_ajax_read();  // All logged-in users can add replies

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_read()
        $comment_id = isset( $_POST['commentId'] ) ? sanitize_text_field( wp_unslash( $_POST['commentId'] ) ) : '';
        $text = isset( $_POST['text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['text'] ) ) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        if (empty($comment_id)) {
            wp_send_json_error(array('message' => 'Comment ID is required'));
            return;
        }

        if (empty($text)) {
            wp_send_json_error(array('message' => 'Reply text is required'));
            return;
        }

        // Get current user info
        $current_user = wp_get_current_user();
        $name = $current_user->display_name;
        $email = $current_user->user_email;

        // Check if file was uploaded
        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified above, file handled by wp_handle_upload
        if ( ! empty( $_FILES['file'] ) && ! empty( $_FILES['file']['tmp_name'] ) ) {
            $result = $this->api->add_reply_with_file(
                $domain_id,
                $comment_id,
                $text,
                $name,
                $email,
                $_FILES['file'] // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.NonceVerification.Missing -- File handled by API, nonce verified above
            );
        } else {
            $result = $this->api->add_reply($domain_id, $comment_id, $text, $name, $email);
        }

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
            return;
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Mark comments as seen (update last seen count)
     */
    public function ajax_mark_comments_seen() {
        $this->verify_ajax_read();  // All logged-in users can mark as seen

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_read()
        $todo_count = isset( $_POST['todoCount'] ) ? intval( $_POST['todoCount'] ) : 0;

        // Save the current todo count as last seen
        $this->settings->save_last_seen_todo_count($todo_count);

        // Clear the cached new comments count
        delete_transient('sc_new_comments_count');

        wp_send_json_success(array('lastSeenCount' => $todo_count));
    }

    /**
     * AJAX: Sync all WordPress users to SimpleCommenter
     */
    public function ajax_sync_all_users() {
        $this->verify_ajax();

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        // Get all WordPress users
        $wp_users = $this->get_wordpress_users();

        // Get admin roles setting for permission check
        $admin_roles = $this->settings->get_admin_roles();

        $synced = 0;
        $skipped = 0;
        $errors = 0;
        $error_details = array();

        foreach ($wp_users as $wp_user) {
            $email = $wp_user['email'];
            $name = $wp_user['name'];
            $role = $wp_user['role'];

            // Check if WP administrator
            $is_wp_admin = ($role === 'administrator');

            // Check if user has SC management permission
            $has_manage_permission = in_array($role, $admin_roles);

            // Sync the user
            $result = $this->api->sync_wp_user(
                $domain_id,
                $email,
                $name,
                $role,
                $is_wp_admin,
                $has_manage_permission
            );

            if (isset($result['error'])) {
                // Log the error for debugging (only in debug mode)
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Debug logging
                    error_log( 'SimpleCommenter sync error for ' . $email . ': ' . print_r( $result, true ) );
                }

                // Check if it's a "member limit reached" error - skip silently
                if (isset($result['code']) && $result['code'] === 'MEMBER_LIMIT_REACHED') {
                    $skipped++;
                } else {
                    $errors++;
                    // Include HTTP code in error message if available
                    $error_msg = $result['error'];
                    if (isset($result['code']) && $result['code'] > 0) {
                        $error_msg .= ' (HTTP ' . $result['code'] . ')';
                    }
                    $error_details[] = array(
                        'email' => $email,
                        'error' => $error_msg,
                    );
                }
            } elseif (!isset($result['success']) && !isset($result['email'])) {
                // Unexpected response format
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Debug logging
                    error_log( 'SimpleCommenter sync unexpected response for ' . $email . ': ' . print_r( $result, true ) );
                }
                $errors++;
                $error_details[] = array(
                    'email' => $email,
                    'error' => 'Unexpected API response',
                );
            } else {
                $synced++;
            }
        }

        wp_send_json_success(array(
            'synced' => $synced,
            'skipped' => $skipped,
            'errors' => $errors,
            'total' => count($wp_users),
            'errorDetails' => $error_details,
        ));
    }

    /**
     * AJAX: Get upgrade link (login link with redirect to pricing page)
     */
    public function ajax_get_upgrade_link() {
        $this->verify_ajax();

        $current_user = wp_get_current_user();
        $domain_id = $this->settings->get_domain_id();

        // Get user's primary role
        $wp_role = !empty($current_user->roles) ? $current_user->roles[0] : 'subscriber';

        // Check if user is WP administrator
        $is_wp_admin = current_user_can('administrator');

        // Check if user has SC management permission (based on admin_roles setting)
        $has_manage_permission = $this->settings->current_user_can_manage();

        // Use the wp-login endpoint with redirect to pricing page
        $result = $this->api->get_wp_login_link(
            $current_user->user_email,
            $current_user->display_name,
            $wp_role,
            $is_wp_admin,
            $has_manage_permission,
            $domain_id ?: '',
            '/app/pricing'  // Redirect to pricing page after login
        );

        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
            return;
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Sync current WordPress user to SimpleCommenter
     * Used when user clicks "Connect to SimpleCommenter" button
     */
    public function ajax_sync_current_user() {
        $this->verify_ajax_read();  // Any logged-in user can sync themselves

        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            wp_send_json_error(array('message' => 'No project selected'));
            return;
        }

        $current_user = wp_get_current_user();

        // Get user's primary role
        $wp_role = !empty($current_user->roles) ? $current_user->roles[0] : 'subscriber';

        // Check if user is WP administrator
        $is_wp_admin = current_user_can('administrator');

        // Check if user has SC management permission (based on admin_roles setting)
        $has_manage_permission = $this->settings->current_user_can_manage();

        // Sync the user
        $result = $this->api->sync_wp_user(
            $domain_id,
            $current_user->user_email,
            $current_user->display_name,
            $wp_role,
            $is_wp_admin,
            $has_manage_permission
        );

        if (isset($result['error'])) {
            // Provide user-friendly error messages
            $message = $result['error'];
            if (isset($result['code']) && $result['code'] === 'MEMBER_LIMIT_REACHED') {
                $message = 'Team member limit reached. Please ask the account owner to upgrade or remove an existing member.';
            }
            wp_send_json_error(array('message' => $message));
            return;
        }

        // Set transient to mark user as synced
        set_transient('sc_synced_user_' . $current_user->ID, true, DAY_IN_SECONDS);

        wp_send_json_success(array(
            'synced' => true,
            'email' => $current_user->user_email,
            'role' => $is_wp_admin ? 'user' : ($has_manage_permission ? 'team' : 'client'),
        ));
    }
}
