<?php
/**
 * SimpleCommenter API Client
 *
 * Handles all communication with the SimpleCommenter API
 */

if (!defined('ABSPATH')) {
    exit;
}

class SC_API {

    /**
     * API base URL
     */
    private $base_url;

    /**
     * Request timeout in seconds
     */
    private $timeout = 30;

    /**
     * Constructor
     */
    public function __construct() {
        $this->base_url = SC_API_BASE;
    }

    /**
     * Get integration token from options
     */
    private function get_token() {
        return get_option('sc_integration_token', '');
    }

    /**
     * Make API request
     *
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array|null $body Request body
     * @param bool $use_token Whether to include integration token
     * @return array Response data or error
     */
    private function request($endpoint, $method = 'GET', $body = null, $use_token = true) {
        $url = $this->base_url . $endpoint;

        $args = array(
            'method' => $method,
            'timeout' => $this->timeout,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
        );

        // Add integration token if available and requested
        if ($use_token && $this->get_token()) {
            $args['headers']['X-Integration-Token'] = $this->get_token();
        }

        // Add body for POST/PUT requests
        if ($body) {
            $args['body'] = wp_json_encode($body);
        }

        // Make request
        $response = wp_remote_request($url, $args);

        // Check for WP error
        if (is_wp_error($response)) {
            return array(
                'error' => $response->get_error_message(),
                'code' => 0,
            );
        }

        // Parse response
        $response_body = wp_remote_retrieve_body($response);
        $response_code = wp_remote_retrieve_response_code($response);
        $data = json_decode($response_body, true);

        // Handle error responses
        if ($response_code >= 400) {
            return array(
                'error' => isset($data['error']) ? $data['error'] : (isset($data['message']) ? $data['message'] : 'API request failed'),
                'code' => $response_code,
            );
        }

        return $data;
    }

    /**
     * Register new account
     *
     * @param string $email User email
     * @param string $password User password
     * @param string $company Company name
     * @param string $site_url WordPress site URL
     * @return array Response data
     */
    public function register($email, $password, $company, $site_url) {
        return $this->request('/external/register', 'POST', array(
            'email' => $email,
            'password' => $password,
            'company' => $company,
            'siteUrl' => $site_url,
            'source' => 'wordpress',
        ), false);
    }

    /**
     * Connect existing account
     *
     * @param string $email User email
     * @param string $password User password
     * @return array Response data
     */
    public function connect($email, $password) {
        return $this->request('/external/connect', 'POST', array(
            'email' => $email,
            'password' => $password,
            'source' => 'wordpress',
        ), false);
    }

    /**
     * Get account information
     *
     * @return array Account info
     */
    public function get_account_info() {
        return $this->request('/external/account', 'GET');
    }

    /**
     * Get list of domains
     *
     * @return array Domains list
     */
    public function get_domains() {
        return $this->request('/external/domains', 'GET');
    }

    /**
     * Add new domain
     *
     * @param string $project_name Project name
     * @param string $domain Domain URL
     * @return array New domain data
     */
    public function add_domain($project_name, $domain) {
        return $this->request('/external/domains', 'POST', array(
            'projectName' => $project_name,
            'domain' => $domain,
        ));
    }

    /**
     * Get domain settings
     *
     * @param string $domain_id Domain ID
     * @return array Domain settings
     */
    public function get_domain_settings($domain_id) {
        return $this->request('/external/domains/' . $domain_id . '/settings', 'GET');
    }

    /**
     * Update domain settings
     *
     * @param string $domain_id Domain ID
     * @param array $settings Settings to update
     * @return array Updated settings
     */
    public function update_domain_settings($domain_id, $settings) {
        return $this->request('/external/domains/' . $domain_id . '/settings', 'PUT', $settings);
    }

    /**
     * Disconnect (revoke token)
     *
     * @return array Response
     */
    public function disconnect() {
        $result = $this->request('/external/disconnect', 'POST');

        // Clear local token regardless of API response
        delete_option('sc_integration_token');

        return $result;
    }

    /**
     * Get login link for auto-login to dashboard
     *
     * @return array Login link data
     * @deprecated Use get_wp_login_link() for properly scoped access
     */
    public function get_login_link() {
        return $this->request('/external/login-link', 'GET');
    }

    /**
     * Get WordPress user login link for dashboard access
     *
     * Role mapping:
     * - WordPress Administrator -> SimpleCommenter member (role: "user")
     * - WordPress user with SC management permission -> SimpleCommenter member (role: "team")
     * - Everyone else -> SimpleCommenter client
     *
     * @param string $email WordPress user email
     * @param string $name WordPress user display name
     * @param string $wp_role WordPress role slug
     * @param bool $is_wp_admin Whether user is WP administrator
     * @param bool $has_manage_permission Whether user has SC management permission
     * @param string $domain_id The domain ID for client scoping
     * @param string $redirect Optional redirect path after login (e.g., '/app/pricing')
     * @return array Response with login URL and token
     */
    public function get_wp_login_link($email, $name, $wp_role, $is_wp_admin, $has_manage_permission, $domain_id, $redirect = null) {
        $data = array(
            'email' => $email,
            'name' => $name,
            'wpRole' => $wp_role,
            'isWpAdmin' => $is_wp_admin,
            'hasManagePermission' => $has_manage_permission,
            'domainId' => $domain_id,
        );

        if ($redirect) {
            $data['redirect'] = $redirect;
        }

        return $this->request('/external/wp-login', 'POST', $data);
    }

    /**
     * Get Google OAuth URL for WordPress redirect
     *
     * @param string $return_url URL to return to after auth
     * @param string $action "login" or "register"
     * @return string Google OAuth URL
     */
    public function get_google_auth_url($return_url, $action = 'login') {
        $base = str_replace('/api', '', $this->base_url);
        return $base . '/api/external/google-auth?' . http_build_query(array(
            'returnUrl' => $return_url,
            'action' => $action,
        ));
    }

    /**
     * Get clients for a domain
     *
     * @param string $domain_id Domain ID
     * @return array Clients list
     */
    public function get_clients($domain_id) {
        return $this->request('/external/clients?domainId=' . urlencode($domain_id), 'GET');
    }

    /**
     * Sync/add a client (WordPress user)
     *
     * @param string $domain_id Domain ID
     * @param string $email Client email
     * @param string $name Client name
     * @param bool $auto_approve Auto-approve the client
     * @return array Response
     */
    public function sync_client($domain_id, $email, $name, $auto_approve = true) {
        return $this->request('/external/clients', 'POST', array(
            'domainId' => $domain_id,
            'email' => $email,
            'name' => $name,
            'autoApprove' => $auto_approve,
        ));
    }

    /**
     * Sync WordPress user based on their role
     *
     * Role mapping:
     * - WordPress Administrator -> SimpleCommenter member (role: "user")
     * - WordPress user with SC management permission -> SimpleCommenter member (role: "team")
     * - Everyone else -> SimpleCommenter client
     *
     * @param string $domain_id Domain ID
     * @param string $email User email
     * @param string $name User display name
     * @param string $wp_role WordPress role slug
     * @param bool $is_wp_admin Whether user is WP administrator
     * @param bool $has_manage_permission Whether user has SC management permission
     * @return array Response
     */
    public function sync_wp_user($domain_id, $email, $name, $wp_role, $is_wp_admin, $has_manage_permission) {
        return $this->request('/external/wp-login', 'POST', array(
            'email' => $email,
            'name' => $name,
            'wpRole' => $wp_role,
            'isWpAdmin' => $is_wp_admin,
            'hasManagePermission' => $has_manage_permission,
            'domainId' => $domain_id,
            'syncOnly' => true,  // Flag to indicate this is just a sync, not a login request
        ));
    }

    /**
     * Get team members
     *
     * @return array Members list
     */
    public function get_members() {
        return $this->request('/external/members', 'GET');
    }

    /**
     * Add a team member
     *
     * @param string $email Member email
     * @param string $name Member name
     * @param string $role Member role (admin, user, team)
     * @return array Response
     */
    public function add_member($email, $name, $role = 'team') {
        return $this->request('/external/members', 'POST', array(
            'email' => $email,
            'name' => $name,
            'role' => $role,
        ));
    }

    /**
     * Remove a team member
     *
     * @param string $member_id Member ID
     * @return array Response
     */
    public function remove_member($member_id) {
        return $this->request('/external/members?memberId=' . urlencode($member_id), 'DELETE');
    }

    /**
     * Get client token for auto-login
     *
     * @param string $domain_id Domain ID
     * @param string $email Client email
     * @return array Response with token
     */
    public function get_client_token($domain_id, $email) {
        return $this->request('/external/client-token', 'POST', array(
            'domainId' => $domain_id,
            'email' => $email,
        ));
    }

    /**
     * Check if API is configured and connected
     *
     * @return bool
     */
    public function is_connected() {
        return !empty($this->get_token());
    }

    /**
     * Verify token is still valid
     *
     * @return bool
     */
    public function verify_connection() {
        if (!$this->is_connected()) {
            return false;
        }

        $result = $this->get_account_info();
        return !isset($result['error']);
    }

    /**
     * Send login code to email
     *
     * @param string $email User email
     * @return array Response data
     */
    public function send_login_code($email) {
        return $this->request('/external/login-code', 'POST', array(
            'email' => $email,
            'source' => 'wordpress',
        ), false);
    }

    /**
     * Verify login code and get integration token
     *
     * @param string $email User email
     * @param string $code 6-digit login code
     * @return array Response with token and account data
     */
    public function verify_login_code($email, $code) {
        return $this->request('/external/login-code/verify', 'POST', array(
            'email' => $email,
            'code' => $code,
            'source' => 'wordpress',
        ), false);
    }

    /**
     * Send registration code to email
     *
     * @param string $email User email
     * @return array Response data
     */
    public function send_register_code($email) {
        return $this->request('/external/register-code', 'POST', array(
            'email' => $email,
            'source' => 'wordpress',
        ), false);
    }

    /**
     * Verify registration code and create account
     *
     * @param string $email User email
     * @param string $code 6-digit registration code
     * @param string $company Company name
     * @param string $site_url WordPress site URL
     * @return array Response with token and account data
     */
    public function verify_register_code($email, $code, $company, $site_url) {
        return $this->request('/external/register-code/verify', 'POST', array(
            'email' => $email,
            'code' => $code,
            'company' => $company,
            'siteUrl' => $site_url,
            'source' => 'wordpress',
        ), false);
    }

    /**
     * Get comments for a domain
     *
     * @param string $domain_id Domain ID
     * @param string $status Optional status filter (comma-separated)
     * @param int $limit Max comments to return
     * @param int $offset Pagination offset
     * @return array Comments data
     */
    public function get_comments($domain_id, $status = '', $limit = 50, $offset = 0) {
        $query = http_build_query(array_filter(array(
            'domainId' => $domain_id,
            'status' => $status,
            'limit' => $limit,
            'offset' => $offset,
        )));
        return $this->request('/external/comments?' . $query, 'GET');
    }

    /**
     * Update comment status or priority
     *
     * @param string $domain_id Domain ID
     * @param string $comment_id Comment ID
     * @param string $status New status (optional)
     * @param string $priority New priority (optional)
     * @return array Response
     */
    public function update_comment($domain_id, $comment_id, $status = null, $priority = null) {
        $data = array(
            'domainId' => $domain_id,
            'commentId' => $comment_id,
        );
        if ($status) {
            $data['status'] = $status;
        }
        if ($priority) {
            $data['priority'] = $priority;
        }
        return $this->request('/external/comments/update', 'POST', $data);
    }

    /**
     * Add reply to a comment
     *
     * @param string $domain_id Domain ID
     * @param string $comment_id Comment ID
     * @param string $text Reply text
     * @param string $name Author name
     * @param string $email Author email
     * @return array Response
     */
    public function add_reply($domain_id, $comment_id, $text, $name = '', $email = '') {
        return $this->request('/external/comments/reply', 'POST', array(
            'domainId' => $domain_id,
            'commentId' => $comment_id,
            'text' => $text,
            'name' => $name,
            'email' => $email,
        ));
    }

    /**
     * Add reply with file attachment
     *
     * @param string $domain_id Domain ID
     * @param string $comment_id Comment ID
     * @param string $text Reply text
     * @param string $name Author name
     * @param string $email Author email
     * @param array $file File array from $_FILES
     * @return array Response
     */
    public function add_reply_with_file($domain_id, $comment_id, $text, $name, $email, $file) {
        $url = $this->base_url . '/external/comments/reply';

        $boundary = wp_generate_password(24, false);

        $body = '';

        // Add text fields
        $fields = array(
            'domainId' => $domain_id,
            'commentId' => $comment_id,
            'text' => $text,
            'name' => $name,
            'email' => $email,
        );

        foreach ($fields as $field_name => $value) {
            $body .= "--{$boundary}\r\n";
            $body .= "Content-Disposition: form-data; name=\"{$field_name}\"\r\n\r\n";
            $body .= "{$value}\r\n";
        }

        // Add file
        if ($file && isset($file['tmp_name']) && file_exists($file['tmp_name'])) {
            $file_content = file_get_contents($file['tmp_name']);
            $file_name = $file['name'];
            $file_type = $file['type'];

            $body .= "--{$boundary}\r\n";
            $body .= "Content-Disposition: form-data; name=\"file\"; filename=\"{$file_name}\"\r\n";
            $body .= "Content-Type: {$file_type}\r\n\r\n";
            $body .= $file_content . "\r\n";
        }

        $body .= "--{$boundary}--\r\n";

        $args = array(
            'method' => 'POST',
            'timeout' => $this->timeout,
            'headers' => array(
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
                'X-Integration-Token' => $this->get_token(),
            ),
            'body' => $body,
        );

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return array(
                'error' => $response->get_error_message(),
                'code' => 0,
            );
        }

        $response_body = wp_remote_retrieve_body($response);
        $response_code = wp_remote_retrieve_response_code($response);
        $data = json_decode($response_body, true);

        if ($response_code >= 400) {
            return array(
                'error' => isset($data['error']) ? $data['error'] : 'Failed to add reply',
                'code' => $response_code,
            );
        }

        return $data;
    }
}
