<?php
/**
 * SimpleCommenter Script Injector
 *
 * Handles injection of the SimpleCommenter widget script into WordPress pages
 */

if (!defined('ABSPATH')) {
    exit;
}

class SC_Script_Injector {

    /**
     * Settings manager instance
     */
    private $settings;

    /**
     * API client instance
     */
    private $api;

    /**
     * Constructor
     *
     * @param SC_Settings $settings Settings manager
     * @param SC_API $api API client (optional)
     */
    public function __construct($settings, $api = null) {
        $this->settings = $settings;
        $this->api = $api;
    }

    /**
     * Initialize script injection
     */
    public function init() {
        // Only inject on frontend
        if (is_admin()) {
            return;
        }

        // Check if widget should be active
        if (!$this->settings->is_widget_active()) {
            return;
        }

        // Check conditional loading setting
        if ($this->should_skip_conditional_loading()) {
            return;
        }

        // Hook into wp_enqueue_scripts for proper script loading
        add_action('wp_enqueue_scripts', array($this, 'enqueue_widget_script'));

        // Hook into wp_head to inject role-based access control
        add_action('wp_head', array($this, 'inject_role_access_control'));

        // Hook into wp_head to inject user info for auto-login
        add_action('wp_head', array($this, 'inject_user_info'));

        // Auto-sync WordPress users as clients (runs once per session)
        if ($this->settings->get_auto_sync_clients()) {
            add_action('wp_footer', array($this, 'sync_current_user'));
        }
    }

    /**
     * Inject role-based access control script
     * Sets window.simpleCommenterDisabled based on user role
     */
    public function inject_role_access_control() {
        $enabled_roles = $this->settings->get_enabled_roles();

        // If no roles configured, enable for everyone
        if (empty($enabled_roles)) {
            return;
        }

        $enable = false;

        // Always enable for admin pages (though we don't inject there anyway)
        if (is_admin()) {
            $enable = true;
        } elseif (is_user_logged_in()) {
            // Check if current user has an enabled role
            $user = wp_get_current_user();
            foreach ($user->roles as $role) {
                if (in_array($role, $enabled_roles)) {
                    $enable = true;
                    break;
                }
            }
        } else {
            // Guest users - check if 'guest' is in enabled roles
            if (in_array('guest', $enabled_roles)) {
                $enable = true;
            }
        }

        // Output the control script
        if ($enable) {
            echo '<script>window.simpleCommenterDisabled = false;</script>';
        } else {
            echo '<script>window.simpleCommenterDisabled = true;</script>';
        }
    }

    /**
     * Enqueue the widget script
     */
    public function enqueue_widget_script() {
        // Use domain hostname for the script URL (the actual domain, not projectId)
        $domain = $this->settings->get_domain_hostname();

        if (empty($domain)) {
            return;
        }

        // Build script URL
        $script_url = 'https://simplecommenter.com/js/comments.min.js';
        $script_url = add_query_arg('domain', $domain, $script_url);

        // Check WordPress version for script loading strategy support (6.3+)
        global $wp_version;
        $supports_strategy = version_compare($wp_version, '6.3', '>=');

        if ($supports_strategy) {
            // WordPress 6.3+ - use strategy parameter
            wp_register_script(
                'simple-commenter-widget',
                $script_url,
                array(),
                SC_VERSION,
                array(
                    'strategy' => 'defer',
                    'in_footer' => true,
                )
            );
        } else {
            // Older WordPress - use traditional in_footer
            wp_register_script(
                'simple-commenter-widget',
                $script_url,
                array(),
                SC_VERSION,
                true // in_footer
            );
            // Add defer attribute via filter
            add_filter('script_loader_tag', array($this, 'add_defer_attribute'), 10, 2);
        }

        wp_enqueue_script('simple-commenter-widget');
    }

    /**
     * Add defer attribute to script tag (for older WordPress)
     */
    public function add_defer_attribute($tag, $handle) {
        if ($handle === 'simple-commenter-widget') {
            return str_replace(' src', ' defer src', $tag);
        }
        return $tag;
    }

    /**
     * Inject current user info and token for widget auto-login
     * Sets window.simpleCommenterUser with user details
     * Also injects token for automatic login
     */
    public function inject_user_info() {
        if (!is_user_logged_in() || !$this->api) {
            return;
        }

        $user = wp_get_current_user();
        $domain_id = $this->settings->get_domain_id();

        // Build user info for the widget
        $user_info = array(
            'email' => $user->user_email,
            'name' => $user->display_name,
        );

        echo '<script>window.simpleCommenterUser = ' . wp_json_encode($user_info) . ';</script>';

        // Get client token for auto-login (if auto-sync is enabled)
        if ($this->settings->get_auto_sync_clients() && !empty($domain_id)) {
            $token_data = $this->get_cached_client_token($user->user_email, $domain_id);

            if ($token_data && isset($token_data['token'])) {
                // Inject token into localStorage before comments.js loads
                // This allows automatic login without modifying the URL
                $script = sprintf(
                    '<script>
                    (function() {
                        try {
                            var stored = localStorage.getItem("simpleCommenterUserData");
                            var userData = stored ? JSON.parse(stored) : {};
                            if (!userData.token || userData.email !== %s) {
                                userData.token = %s;
                                userData.email = %s;
                                userData.name = %s;
                                userData.role = "client";
                                userData.login = true;
                                localStorage.setItem("simpleCommenterUserData", JSON.stringify(userData));
                            }
                        } catch(e) { console.error("SC auto-login error:", e); }
                    })();
                    </script>',
                    wp_json_encode($user->user_email),
                    wp_json_encode($token_data['token']),
                    wp_json_encode($user->user_email),
                    wp_json_encode($token_data['name'] ?: $user->display_name)
                );
                // Script contains only JSON-encoded values which are safe
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Script uses wp_json_encode for all dynamic values
                echo $script;
            }
        }
    }

    /**
     * Get cached client token or fetch new one
     *
     * @param string $email User email
     * @param string $domain_id Domain ID
     * @return array|null Token data or null
     */
    private function get_cached_client_token($email, $domain_id) {
        $cache_key = 'sc_client_token_' . md5($email . $domain_id);
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return $cached;
        }

        // Fetch token from API
        $result = $this->api->get_client_token($domain_id, $email);

        if (!isset($result['error']) && isset($result['token'])) {
            // Cache for 1 day (tokens are valid for 30 days)
            set_transient($cache_key, $result, DAY_IN_SECONDS);
            return $result;
        }

        return null;
    }

    /**
     * Sync current WordPress user to SimpleCommenter based on their role
     *
     * Role mapping:
     * - WordPress Administrator -> SimpleCommenter member (role: "user")
     * - WordPress user with SC management permission -> SimpleCommenter member (role: "team")
     * - Everyone else -> SimpleCommenter client
     *
     * Uses transient to avoid syncing on every page load
     */
    public function sync_current_user() {
        if (!is_user_logged_in() || !$this->api) {
            return;
        }

        $user = wp_get_current_user();
        $domain_id = $this->settings->get_domain_id();

        if (empty($domain_id)) {
            return;
        }

        // Use transient to only sync once per day per user
        $transient_key = 'sc_synced_user_' . $user->ID;
        if (get_transient($transient_key)) {
            return;
        }

        // Get user's primary role
        $wp_role = !empty($user->roles) ? $user->roles[0] : 'subscriber';

        // Check if user is WP administrator
        $is_wp_admin = user_can($user, 'administrator');

        // Check if user has SC management permission (based on admin_roles setting)
        $admin_roles = $this->settings->get_admin_roles();
        $has_manage_permission = false;
        foreach ($user->roles as $role) {
            if (in_array($role, $admin_roles)) {
                $has_manage_permission = true;
                break;
            }
        }

        // Sync user with proper role mapping
        $result = $this->api->sync_wp_user(
            $domain_id,
            $user->user_email,
            $user->display_name,
            $wp_role,
            $is_wp_admin,
            $has_manage_permission
        );

        // Set transient to avoid repeated syncs (1 day)
        if (!isset($result['error'])) {
            set_transient($transient_key, true, DAY_IN_SECONDS);
        }
    }

    /**
     * Get the script tag HTML (for display in admin)
     *
     * @return string Script tag HTML
     */
    public function get_script_tag() {
        $domain = $this->settings->get_domain_hostname();

        if (empty($domain)) {
            return '';
        }

        return sprintf(
            '<script src="https://simplecommenter.com/js/comments.min.js?domain=%s" defer></script>', // phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedScript -- Returns HTML string for display purposes, not actual script injection
            esc_attr( $domain )
        );
    }

    /**
     * Check if we should skip loading due to conditional loading setting
     *
     * When conditionalLoading is enabled, only load when ?simple-commenter=true is in the URL
     *
     * @return bool True if we should skip loading
     */
    private function should_skip_conditional_loading() {
        $domain_settings = $this->settings->get_domain_settings();

        // Check if conditional loading is enabled
        if (empty($domain_settings['conditionalLoading'])) {
            return false;
        }

        // Check if simple-commenter=true is in the URL
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only boolean check, no data processing
        if ( isset( $_GET['simple-commenter'] ) && 'true' === $_GET['simple-commenter'] ) {
            return false; // Don't skip - load the script
        }

        // Conditional loading enabled but param not present - skip loading
        return true;
    }
}
