<?php
/**
 * SimpleCommenter Settings Manager
 *
 * Handles plugin settings storage and retrieval
 */

if (!defined('ABSPATH')) {
    exit;
}

class SC_Settings {

    /**
     * Check if plugin is connected to SimpleCommenter
     *
     * @return bool
     */
    public function is_connected() {
        return !empty(get_option('sc_integration_token', ''));
    }

    /**
     * Get integration token
     *
     * @return string
     */
    public function get_token() {
        return get_option('sc_integration_token', '');
    }

    /**
     * Get project ID
     *
     * @return string
     */
    public function get_project_id() {
        return get_option('sc_project_id', '');
    }

    /**
     * Get domain hostname (for widget script URL)
     * Returns the actual domain hostname, falling back to projectId
     *
     * @return string
     */
    public function get_domain_hostname() {
        $settings = $this->get_domain_settings();
        // Use actual domain hostname if available, otherwise fall back to projectId
        if (!empty($settings['domain'])) {
            return $settings['domain'];
        }
        return $this->get_project_id();
    }

    /**
     * Get domain ID
     *
     * @return string
     */
    public function get_domain_id() {
        return get_option('sc_domain_id', '');
    }

    /**
     * Get domain settings
     *
     * @return array
     */
    public function get_domain_settings() {
        return get_option('sc_settings', array());
    }

    /**
     * Get account info
     *
     * @return array
     */
    public function get_account_info() {
        return get_option('sc_account_info', array());
    }

    /**
     * Save connection data
     *
     * @param string $token Integration token
     * @param array $account_info Account information
     */
    public function save_connection($token, $account_info) {
        update_option('sc_integration_token', $token);
        update_option('sc_account_info', $account_info);

        // Clear caches
        delete_transient('sc_account_cache');
        delete_transient('sc_domains_cache');
    }

    /**
     * Save domain selection
     *
     * @param string $domain_id Domain ID
     * @param string $project_id Project ID
     * @param array $settings Domain settings
     */
    public function save_domain($domain_id, $project_id, $settings) {
        update_option('sc_domain_id', $domain_id);
        update_option('sc_project_id', $project_id);
        update_option('sc_settings', $settings);
    }

    /**
     * Clear domain selection only (keeps account connected)
     */
    public function clear_domain() {
        delete_option('sc_domain_id');
        delete_option('sc_project_id');
        delete_option('sc_settings');
    }

    /**
     * Clear all settings
     */
    public function clear_all() {
        delete_option('sc_integration_token');
        delete_option('sc_domain_id');
        delete_option('sc_project_id');
        delete_option('sc_settings');
        delete_option('sc_account_info');
        delete_option('sc_enabled_roles');

        // Clear caches
        delete_transient('sc_account_cache');
        delete_transient('sc_domains_cache');
    }

    /**
     * Get trial information
     *
     * @return array|null Trial info or null if no account
     */
    public function get_trial_info() {
        $account = $this->get_account_info();

        if (empty($account)) {
            return null;
        }

        // Check if trial info is available from API response
        if (isset($account['trial'])) {
            return array(
                'is_trial' => $account['trial']['isTrial'],
                'is_paid' => $account['trial']['isPaid'],
                'tier' => $account['trial']['tier'],
                'days_remaining' => $account['trial']['daysRemaining'],
                'expired' => $account['trial']['expired'],
            );
        }

        // Calculate from account info
        $tier = isset($account['tier']) ? strtolower($account['tier']) : 'pro';
        $created_at = isset($account['createdAt']) ? $account['createdAt'] : null;

        $is_trial = ($tier === 'pro' || $tier === 'free');
        $days_remaining = 0;

        if ($is_trial && $created_at) {
            $created = new DateTime($created_at);
            $now = new DateTime();
            $diff = $now->diff($created)->days;
            $days_remaining = max(0, 14 - $diff);
        }

        return array(
            'is_trial' => $is_trial,
            'is_paid' => in_array($tier, array('pro_paid', 'business', 'unlimited', 'enterprise')),
            'tier' => $tier,
            'days_remaining' => $days_remaining,
            'expired' => $is_trial && $days_remaining <= 0,
        );
    }

    /**
     * Get enabled WordPress user roles
     *
     * @return array Array of role slugs that can use SimpleCommenter
     */
    public function get_enabled_roles() {
        return get_option('sc_enabled_roles', array());
    }

    /**
     * Save enabled WordPress user roles
     *
     * @param array $roles Array of role slugs
     */
    public function save_enabled_roles($roles) {
        update_option('sc_enabled_roles', $roles);
    }

    /**
     * Get auto-sync clients setting
     *
     * @return bool Whether to auto-sync WordPress users as clients
     */
    public function get_auto_sync_clients() {
        return (bool) get_option('sc_auto_sync_clients', true);
    }

    /**
     * Save auto-sync clients setting
     *
     * @param bool $enabled Whether to enable auto-sync
     */
    public function save_auto_sync_clients($enabled) {
        update_option('sc_auto_sync_clients', (bool) $enabled);
    }

    /**
     * Get admin roles (roles that can manage/disconnect SimpleCommenter)
     *
     * @return array Array of role slugs
     */
    public function get_admin_roles() {
        $roles = get_option('sc_admin_roles', array('administrator'));
        // Always include administrator
        if (!in_array('administrator', $roles)) {
            $roles[] = 'administrator';
        }
        return $roles;
    }

    /**
     * Save admin roles
     *
     * @param array $roles Array of role slugs
     */
    public function save_admin_roles($roles) {
        // Always include administrator
        if (!in_array('administrator', $roles)) {
            $roles[] = 'administrator';
        }
        update_option('sc_admin_roles', $roles);
    }

    /**
     * Check if current user can manage SimpleCommenter
     * Uses the sc_admin_roles setting configured in Management Permissions
     *
     * @return bool
     */
    public function current_user_can_manage() {
        if (!is_user_logged_in()) {
            return false;
        }

        // Administrators always have permission
        if (current_user_can('administrator')) {
            return true;
        }

        // Check if user's role is in the admin_roles setting
        $admin_roles = $this->get_admin_roles();
        $user = wp_get_current_user();

        foreach ($user->roles as $role) {
            if (in_array($role, $admin_roles)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get last seen comment count (for "todo" status)
     *
     * @return int
     */
    public function get_last_seen_todo_count() {
        return (int) get_option('sc_last_seen_todo_count', 0);
    }

    /**
     * Save last seen comment count (for "todo" status)
     *
     * @param int $count Comment count
     */
    public function save_last_seen_todo_count($count) {
        update_option('sc_last_seen_todo_count', (int) $count);
    }

    /**
     * Check if widget should be active
     *
     * @return bool
     */
    public function is_widget_active() {
        // Must be connected
        if (!$this->is_connected()) {
            return false;
        }

        // Must have project selected
        if (empty($this->get_project_id())) {
            return false;
        }

        // Check domain settings
        $settings = $this->get_domain_settings();
        if (isset($settings['active']) && !$settings['active']) {
            return false;
        }

        // Check trial status
        $trial = $this->get_trial_info();
        if ($trial && $trial['expired']) {
            return false;
        }

        return true;
    }
}
