<?php
/**
 * Plugin Name: Simple Commenter
 * Plugin URI: https://simplecommenter.com
 * Description: The website feedback tool your clients will actually use. Collect visual feedback with screenshots, annotations, and file uploads—directly on your WordPress site.
 * Version: 1.0.0
 * Author: Simple Commenter
 * Author URI: https://ander.ee
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: simple-commenter
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('SC_VERSION', '1.0.0');
define('SC_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SC_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SC_API_BASE', 'https://simplecommenter.com/api');

// Autoload classes
require_once SC_PLUGIN_DIR . 'includes/class-sc-api.php';
require_once SC_PLUGIN_DIR . 'includes/class-sc-settings.php';
require_once SC_PLUGIN_DIR . 'includes/class-sc-admin.php';
require_once SC_PLUGIN_DIR . 'includes/class-sc-script-injector.php';

/**
 * Main SimpleCommenter class
 */
class SimpleCommenter {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * API client instance
     */
    private $api;

    /**
     * Settings manager instance
     */
    private $settings;

    /**
     * Admin controller instance
     */
    private $admin;

    /**
     * Script injector instance
     */
    private $injector;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->api = new SC_API();
        $this->settings = new SC_Settings();
        $this->admin = new SC_Admin($this->api, $this->settings);
        $this->injector = new SC_Script_Injector($this->settings, $this->api);

        add_action('init', array($this, 'init'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Note: load_plugin_textdomain() is not needed for plugins hosted on WordPress.org
        // since WordPress 4.6+. WordPress automatically loads translations.

        // Initialize admin
        if (is_admin()) {
            $this->admin->init();
        }

        // Initialize script injector
        $this->injector->init();
    }

    /**
     * Get API client
     */
    public function get_api() {
        return $this->api;
    }

    /**
     * Get settings manager
     */
    public function get_settings() {
        return $this->settings;
    }
}

/**
 * Initialize plugin
 */
add_action('plugins_loaded', function() {
    SimpleCommenter::get_instance();
});

/**
 * Activation hook
 */
register_activation_hook(__FILE__, function() {
    // Create default options
    add_option('sc_integration_token', '');
    add_option('sc_domain_id', '');
    add_option('sc_project_id', '');
    add_option('sc_settings', array());
    add_option('sc_account_info', array());

    // Flush rewrite rules
    flush_rewrite_rules();
});

/**
 * Deactivation hook
 */
register_deactivation_hook(__FILE__, function() {
    // Optionally disconnect on deactivation
    // This is commented out to preserve connection on deactivation
    // $api = new SC_API();
    // $api->disconnect();

    // Clear cached data
    delete_transient('sc_account_cache');
    delete_transient('sc_domains_cache');
});

/**
 * Uninstall hook (only runs when plugin is deleted)
 */
register_uninstall_hook(__FILE__, 'sc_uninstall');

function sc_uninstall() {
    // Disconnect from SimpleCommenter
    $api = new SC_API();
    $api->disconnect();

    // Remove all options
    delete_option('sc_integration_token');
    delete_option('sc_domain_id');
    delete_option('sc_project_id');
    delete_option('sc_settings');
    delete_option('sc_account_info');

    // Clear transients
    delete_transient('sc_account_cache');
    delete_transient('sc_domains_cache');
}
