<?php
/**
 * Admin page template
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get allowed SVG tags for wp_kses
 *
 * @return array Allowed SVG tags and attributes
 */
function sc_get_allowed_svg_tags() {
    return array(
        'svg'    => array(
            'width'       => true,
            'height'      => true,
            'viewbox'     => true,  // Must be lowercase for wp_kses
            'fill'        => true,
            'xmlns'       => true,
            'class'       => true,
            'stroke'      => true,
            'stroke-width' => true,
        ),
        'path'   => array(
            'd'              => true,
            'fill'           => true,
            'stroke'         => true,
            'stroke-width'   => true,
            'stroke-linejoin' => true,
            'stroke-linecap' => true,
            'clip-rule'      => true,
            'fill-rule'      => true,
        ),
        'circle' => array(
            'cx'   => true,
            'cy'   => true,
            'r'    => true,
            'fill' => true,
        ),
        'rect'   => array(
            'x'      => true,
            'y'      => true,
            'width'  => true,
            'height' => true,
            'fill'   => true,
            'rx'     => true,
            'ry'     => true,
        ),
    );
}

// SVG Icons
// SimpleCommenter logo with yellow accent for page header
$icon_logo = '<svg width="32" height="28" viewBox="-3 -3 88 78" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M11.85 42.0499C11.85 47.0499 15.55 51.4499 20.85 52.1499C24.45 52.6499 22.55 56.55 26.25 56.75L24.65 66.45L41.15 53.75C42.05 52.85 47.55 53.75 48.85 53.75C55.15 53.55 61.55 53.0499 67.75 52.1499C73.05 51.4499 76.75 47.0499 76.75 42.0499V23.15C76.75 18.15 73.05 13.75 67.75 13.05C60.05 11.95 52.25 11.45 44.35 11.45C36.35 11.45 28.55 11.95 20.85 13.05C15.55 13.75 11.85 18.15 11.85 23.15V42.0499Z" fill="#F7D070"/><path d="M78.35 15.55V40.65C78.35 46.05 74.55 50.65 69.45 51.35C61.75 52.45 53.95 53.15 46.15 53.35C44.95 53.35 43.85 53.65 42.85 54.05C41.85 54.45 40.85 55.15 40.05 55.95L27.65 68.35V55.65C27.65 54.25 26.55 53.05 25.15 52.95C20.55 52.55 16.05 52.05 11.65 51.45C6.55 50.65 2.75 46.15 2.75 40.75V15.55C2.75 10.15 6.55 5.55 11.65 4.85C21.15 3.45 30.75 2.75 40.55 2.75C50.25 2.75 59.95 3.45 69.45 4.85C74.55 5.55 78.35 10.15 78.35 15.55Z" stroke="#222222" stroke-width="5.5" stroke-linejoin="round"/></svg>';
$icon_chat = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="M7.5 8.25h9m-9 3H12m-9.75 1.51c0 1.6 1.123 2.994 2.707 3.227 1.129.166 2.27.293 3.423.379.35.026.67.21.865.501L12 21l2.755-4.133a1.14 1.14 0 0 1 .865-.501 48.172 48.172 0 0 0 3.423-.379c1.584-.233 2.707-1.626 2.707-3.228V6.741c0-1.602-1.123-2.995-2.707-3.228A48.394 48.394 0 0 0 12 3c-2.392 0-4.744.175-7.043.513C3.373 3.746 2.25 5.14 2.25 6.741v6.018Z" /></svg>';
$icon_check = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="m4.5 12.75 6 6 9-13.5" /></svg>';
$icon_warning = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="M12 9v3.75m-9.303 3.376c-.866 1.5.217 3.374 1.948 3.374h14.71c1.73 0 2.813-1.874 1.948-3.374L13.949 3.378c-.866-1.5-3.032-1.5-3.898 0L2.697 16.126ZM12 15.75h.007v.008H12v-.008Z" /></svg>';
$icon_clock = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="M12 6v6h4.5m4.5 0a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z" /></svg>';
$icon_external = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="M13.5 6H5.25A2.25 2.25 0 0 0 3 8.25v10.5A2.25 2.25 0 0 0 5.25 21h10.5A2.25 2.25 0 0 0 18 18.75V10.5m-10.5 6L21 3m0 0h-5.25M21 3v5.25" /></svg>';
$icon_info = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="m11.25 11.25.041-.02a.75.75 0 0 1 1.063.852l-.708 2.836a.75.75 0 0 0 1.063.853l.041-.021M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Zm-9-3.75h.008v.008H12V8.25Z" /></svg>';
$icon_star = '<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="0 0 24 24" width="24" height="24"><path fill-rule="evenodd" d="M10.788 3.21c.448-1.077 1.976-1.077 2.424 0l2.082 5.006 5.404.434c1.164.093 1.636 1.545.749 2.305l-4.117 3.527 1.257 5.273c.271 1.136-.964 2.033-1.96 1.425L12 18.354 7.373 21.18c-.996.608-2.231-.29-1.96-1.425l1.257-5.273-4.117-3.527c-.887-.76-.415-2.212.749-2.305l5.404-.434 2.082-5.005Z" clip-rule="evenodd" /></svg>';
$icon_x = '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" width="24" height="24"><path stroke-linecap="round" stroke-linejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>';

// Google icon SVG
$icon_google = '<svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z" fill="#4285F4"/><path d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z" fill="#34A853"/><path d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z" fill="#FBBC05"/><path d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z" fill="#EA4335"/></svg>';

// Get current site domain for matching
$current_site_domain = wp_parse_url( get_site_url(), PHP_URL_HOST );
$current_site_domain = preg_replace('/^www\./', '', strtolower($current_site_domain));

// Check if user is on trial (free/pro tiers are trial tiers)
// Paid tiers are: pro_paid, business, unlimited, premium, enterprise, etc.
function sc_is_user_on_trial($tier) {
    if (!$tier) return true;
    $tier = strtolower($tier);
    return in_array($tier, array('free', 'pro'));
}

// Helper to get tier badge class and label
function sc_get_tier_info($trial) {
    if (!$trial) {
        return array('class' => 'sc-tier-trial', 'label' => 'Trial', 'is_trial' => true, 'tier_label' => 'Trial');
    }

    $tier = strtolower($trial['tier'] ?? 'pro');
    $is_trial = sc_is_user_on_trial($tier);

    if ($is_trial && $trial['expired']) {
        return array('class' => 'sc-tier-expired', 'label' => 'Expired', 'is_trial' => true, 'tier_label' => 'Trial');
    }

    if ($is_trial) {
        $days = $trial['days_remaining'] ?? 0;
        return array('class' => 'sc-tier-trial', 'label' => $days . ' days left', 'is_trial' => true, 'tier_label' => 'Trial');
    }

    // Paid tiers
    if (in_array($tier, array('business'))) {
        return array('class' => 'sc-tier-business', 'label' => 'Business', 'is_trial' => false, 'tier_label' => 'Business');
    }

    if (in_array($tier, array('enterprise'))) {
        return array('class' => 'sc-tier-business', 'label' => 'Enterprise', 'is_trial' => false, 'tier_label' => 'Enterprise');
    }

    if (in_array($tier, array('unlimited'))) {
        return array('class' => 'sc-tier-business', 'label' => 'Unlimited', 'is_trial' => false, 'tier_label' => 'Unlimited');
    }

    // pro_paid or other paid tiers
    return array('class' => 'sc-tier-pro', 'label' => 'Pro', 'is_trial' => false, 'tier_label' => 'Pro');
}

$tier_info = sc_get_tier_info($trial);
// Use project name for display, fall back to account email
$display_name = isset($domain_settings['projectName']) && !empty($domain_settings['projectName'])
    ? $domain_settings['projectName']
    : (isset($account['email']) ? $account['email'] : 'N/A');
$account_initial = strtoupper(substr($display_name, 0, 1));
?>

<div class="wrap sc-admin">
    <?php if (!$connected): ?>
        <!-- Not Connected State -->
        <div class="sc-card sc-connect-card">
            <!-- Header with logo -->
            <div class="sc-connect-header">
                <?php echo wp_kses( $icon_logo, sc_get_allowed_svg_tags() ); ?>
                <div class="sc-connect-header-text">
                    <h1><?php esc_html_e('Simple Commenter', 'simple-commenter'); ?></h1>
                    <p><?php esc_html_e('Visual website feedback tool', 'simple-commenter'); ?></p>
                </div>
            </div>

            <div class="sc-connect-status">
                <span class="sc-connect-status-dot"></span>
                <?php esc_html_e('Not Connected', 'simple-commenter'); ?>
            </div>

            <div class="sc-connect-features">
                <div class="sc-connect-feature">
                    <?php echo wp_kses( $icon_check, sc_get_allowed_svg_tags() ); ?>
                    <span><?php esc_html_e('Ditch confusing email threads — feedback lands exactly where it belongs', 'simple-commenter'); ?></span>
                </div>
                <div class="sc-connect-feature">
                    <?php echo wp_kses( $icon_check, sc_get_allowed_svg_tags() ); ?>
                    <span><?php esc_html_e('Clients leave feedback directly on the page — no login required', 'simple-commenter'); ?></span>
                </div>
                <div class="sc-connect-feature">
                    <?php echo wp_kses( $icon_check, sc_get_allowed_svg_tags() ); ?>
                    <span><?php esc_html_e('Get notified instantly and reply in organized threads', 'simple-commenter'); ?></span>
                </div>
                <div class="sc-connect-feature sc-wordpress-feature">
                    <?php echo wp_kses( $icon_check, sc_get_allowed_svg_tags() ); ?>
                    <span><?php esc_html_e('WordPress exclusive features', 'simple-commenter'); ?></span>
                </div>
            </div>

            <div class="sc-connect-note">
                <?php echo wp_kses( $icon_info, sc_get_allowed_svg_tags() ); ?>
                <span><?php esc_html_e('Sign in as the account owner. Team members sync automatically from WordPress.', 'simple-commenter'); ?></span>
            </div>

            <div class="sc-connect-separator"></div>

            <h2><?php esc_html_e('Connect Your Account', 'simple-commenter'); ?></h2>

            <div class="sc-tabs">
                <button class="sc-tab active" data-tab="login">
                    <?php esc_html_e('Login', 'simple-commenter'); ?>
                </button>
                <button class="sc-tab" data-tab="register">
                    <?php esc_html_e('Create Account', 'simple-commenter'); ?>
                </button>
            </div>

            <!-- Login Form -->
            <div class="sc-tab-content" id="tab-login">
                <!-- Google Login Button -->
                <a href="#" id="sc-google-login" class="sc-google-btn">
                    <?php echo wp_kses( $icon_google, sc_get_allowed_svg_tags() ); ?>
                    <?php esc_html_e('Continue with Google', 'simple-commenter'); ?>
                </a>

                <div class="sc-divider">
                    <span><?php esc_html_e('or', 'simple-commenter'); ?></span>
                </div>

                <!-- Login Method Tabs -->
                <div class="sc-login-methods">
                    <button type="button" class="sc-login-method active" data-method="code">
                        <?php esc_html_e('Email Code', 'simple-commenter'); ?>
                    </button>
                    <button type="button" class="sc-login-method" data-method="password">
                        <?php esc_html_e('Password', 'simple-commenter'); ?>
                    </button>
                </div>

                <!-- Code Login -->
                <div class="sc-login-method-content" id="login-method-code">
                    <!-- Step 1: Email -->
                    <div id="code-step-email">
                        <form id="sc-code-email-form">
                            <div class="sc-field">
                                <label for="code-email"><?php esc_html_e('Email', 'simple-commenter'); ?></label>
                                <input type="email" id="code-email" name="email" required>
                            </div>
                            <p class="description"><?php esc_html_e('We\'ll send a 6-digit code to your email.', 'simple-commenter'); ?></p>
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Send Code', 'simple-commenter'); ?>
                            </button>
                        </form>
                    </div>

                    <!-- Step 2: Code Entry -->
                    <div id="code-step-verify" style="display:none;">
                        <form id="sc-code-verify-form">
                            <p class="sc-code-sent-message">
                                <?php esc_html_e('Enter the 6-digit code sent to', 'simple-commenter'); ?>
                                <strong id="code-sent-email"></strong>
                            </p>
                            <div class="sc-field">
                                <label for="login-code"><?php esc_html_e('Code', 'simple-commenter'); ?></label>
                                <input type="text" id="login-code" name="code" maxlength="6" pattern="[0-9]{6}"
                                       placeholder="000000" autocomplete="one-time-code" required class="sc-code-input">
                            </div>
                            <div class="sc-code-actions">
                                <button type="submit" class="button button-primary">
                                    <?php esc_html_e('Verify Code', 'simple-commenter'); ?>
                                </button>
                                <button type="button" id="sc-resend-code" class="button">
                                    <?php esc_html_e('Resend Code', 'simple-commenter'); ?>
                                </button>
                            </div>
                            <p class="description sc-code-expiry">
                                <?php esc_html_e('Code expires in 10 minutes.', 'simple-commenter'); ?>
                            </p>
                        </form>
                    </div>
                </div>

                <!-- Password Login -->
                <div class="sc-login-method-content" id="login-method-password" style="display:none;">
                    <form id="sc-login-form">
                        <div class="sc-field">
                            <label for="login-email"><?php esc_html_e('Email', 'simple-commenter'); ?></label>
                            <input type="email" id="login-email" name="email" required>
                        </div>
                        <div class="sc-field">
                            <label for="login-password"><?php esc_html_e('Password', 'simple-commenter'); ?></label>
                            <input type="password" id="login-password" name="password" required>
                        </div>
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e('Connect Account', 'simple-commenter'); ?>
                        </button>
                    </form>
                </div>
            </div>

            <!-- Register Form -->
            <div class="sc-tab-content" id="tab-register" style="display:none;">
                <!-- Google Register Button -->
                <a href="#" id="sc-google-register" class="sc-google-btn">
                    <?php echo wp_kses( $icon_google, sc_get_allowed_svg_tags() ); ?>
                    <?php esc_html_e('Sign up with Google', 'simple-commenter'); ?>
                </a>

                <div class="sc-divider">
                    <span><?php esc_html_e('or', 'simple-commenter'); ?></span>
                </div>

                <!-- Register Method Tabs -->
                <div class="sc-login-methods">
                    <button type="button" class="sc-register-method active" data-method="code">
                        <?php esc_html_e('Email Code', 'simple-commenter'); ?>
                    </button>
                    <button type="button" class="sc-register-method" data-method="password">
                        <?php esc_html_e('Password', 'simple-commenter'); ?>
                    </button>
                </div>

                <!-- Code Registration -->
                <div class="sc-register-method-content" id="register-method-code">
                    <!-- Step 1: Email + Company -->
                    <div id="reg-code-step-email">
                        <form id="sc-reg-code-email-form">
                            <div class="sc-field">
                                <label for="reg-code-email"><?php esc_html_e('Email', 'simple-commenter'); ?></label>
                                <input type="email" id="reg-code-email" name="email" required>
                            </div>
                            <div class="sc-field">
                                <label for="reg-code-company"><?php esc_html_e('Company/Site Name', 'simple-commenter'); ?></label>
                                <input type="text" id="reg-code-company" name="company"
                                       value="<?php echo esc_attr(get_bloginfo('name')); ?>" required>
                            </div>
                            <div class="sc-info">
                                <?php echo wp_kses( $icon_info, sc_get_allowed_svg_tags() ); ?>
                                <?php
                                /* translators: %s: site URL */
                                $site_url_message = sprintf( esc_html__( 'Your site URL (%s) will be automatically added as your first project.', 'simple-commenter' ), '<strong>' . esc_html( wp_parse_url( get_site_url(), PHP_URL_HOST ) ) . '</strong>' );
                                ?>
                                <span><?php echo $site_url_message; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Already escaped above ?></span>
                            </div>
                            <p class="description"><?php esc_html_e('We\'ll send a 6-digit code to your email.', 'simple-commenter'); ?></p>
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Send Code', 'simple-commenter'); ?>
                            </button>
                        </form>
                    </div>

                    <!-- Step 2: Code Entry -->
                    <div id="reg-code-step-verify" style="display:none;">
                        <form id="sc-reg-code-verify-form">
                            <p class="sc-code-sent-message">
                                <?php esc_html_e('Enter the 6-digit code sent to', 'simple-commenter'); ?>
                                <strong id="reg-code-sent-email"></strong>
                            </p>
                            <div class="sc-field">
                                <label for="reg-verify-code"><?php esc_html_e('Code', 'simple-commenter'); ?></label>
                                <input type="text" id="reg-verify-code" name="code" maxlength="6" pattern="[0-9]{6}"
                                       placeholder="000000" autocomplete="one-time-code" required class="sc-code-input">
                            </div>
                            <div class="sc-code-actions">
                                <button type="submit" class="button button-primary">
                                    <?php esc_html_e('Create Account', 'simple-commenter'); ?>
                                </button>
                                <button type="button" id="sc-reg-resend-code" class="button">
                                    <?php esc_html_e('Resend Code', 'simple-commenter'); ?>
                                </button>
                            </div>
                            <p class="description sc-code-expiry">
                                <?php esc_html_e('Code expires in 10 minutes.', 'simple-commenter'); ?>
                            </p>
                        </form>
                    </div>
                </div>

                <!-- Password Registration -->
                <div class="sc-register-method-content" id="register-method-password" style="display:none;">
                    <form id="sc-register-form">
                        <div class="sc-field">
                            <label for="reg-email"><?php esc_html_e('Email', 'simple-commenter'); ?></label>
                            <input type="email" id="reg-email" name="email" required>
                        </div>
                        <div class="sc-field">
                            <label for="reg-password"><?php esc_html_e('Password', 'simple-commenter'); ?></label>
                            <input type="password" id="reg-password" name="password" required minlength="6">
                            <p class="description"><?php esc_html_e('Minimum 6 characters', 'simple-commenter'); ?></p>
                        </div>
                        <div class="sc-field">
                            <label for="reg-company"><?php esc_html_e('Company/Site Name', 'simple-commenter'); ?></label>
                            <input type="text" id="reg-company" name="company"
                                   value="<?php echo esc_attr(get_bloginfo('name')); ?>" required>
                        </div>
                        <div class="sc-info">
                            <?php echo wp_kses( $icon_info, sc_get_allowed_svg_tags() ); ?>
                            <?php
                            /* translators: %s: site URL */
                            $site_url_message_2 = sprintf( esc_html__( 'Your site URL (%s) will be automatically added as your first project.', 'simple-commenter' ), '<strong>' . esc_html( wp_parse_url( get_site_url(), PHP_URL_HOST ) ) . '</strong>' );
                            ?>
                            <span><?php echo $site_url_message_2; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Already escaped above ?></span>
                        </div>
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e('Create Account', 'simple-commenter'); ?>
                        </button>
                    </form>
                </div>
            </div>

            <div class="sc-error" id="sc-connect-error" style="display:none;"></div>
        </div>

    <?php else: ?>
        <!-- Connected State -->

        <!-- Page Header -->
        <div class="sc-page-header">
            <h1>
                <?php echo wp_kses( $icon_logo, sc_get_allowed_svg_tags() ); ?>
                <?php esc_html_e('Simple Commenter', 'simple-commenter'); ?>
            </h1>
            <p><?php esc_html_e('Visual website feedback tool', 'simple-commenter'); ?></p>
        </div>

        <!-- Global Error Alert (shown at top) -->
        <div class="sc-alert sc-alert-error" id="sc-global-alert" style="display:none;"></div>

        <!-- Account Header Bar -->
        <div class="sc-account-bar">
            <!-- SC Account Info Row -->
            <div class="sc-account-bar-row">
                <div class="sc-account-info">
                    <div class="sc-account-avatar"><?php echo esc_html($account_initial); ?></div>
                    <div class="sc-account-details">
                        <div class="sc-account-name"><?php echo esc_html($display_name); ?></div>
                        <?php if (!($trial && $trial['expired'])): ?>
                            <span class="sc-tier-badge <?php echo esc_attr($tier_info['class']); ?>"><?php echo esc_html($tier_info['tier_label']); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
                <?php
                // Show status badge on right
                if (!empty($project_id)):
                    if ($trial && $trial['expired']): ?>
                        <span class="sc-status-badge sc-status-badge-error"><?php esc_html_e('Inactive', 'simple-commenter'); ?></span>
                    <?php elseif (isset($domain_settings['active']) && !$domain_settings['active']): ?>
                        <span class="sc-status-badge sc-status-badge-warning"><?php esc_html_e('Disabled', 'simple-commenter'); ?></span>
                    <?php else: ?>
                        <span class="sc-status-badge sc-status-badge-success"><?php esc_html_e('Active', 'simple-commenter'); ?></span>
                    <?php endif;
                endif; ?>
            </div>

            <!-- Current WordPress User Info -->
            <div class="sc-wp-user-bar">
                <div class="sc-wp-user-info">
                    <span class="sc-wp-user-label"><?php esc_html_e('Logged in as:', 'simple-commenter'); ?></span>
                    <strong class="sc-wp-user-email"><?php echo esc_html($wp_user_email); ?></strong>
                    <?php if ($wp_user_is_admin): ?>
                        <span class="sc-wp-role-badge sc-wp-role-admin"><?php esc_html_e('Admin', 'simple-commenter'); ?></span>
                    <?php elseif ($can_manage): ?>
                        <span class="sc-wp-role-badge sc-wp-role-team"><?php esc_html_e('Team', 'simple-commenter'); ?></span>
                    <?php else: ?>
                        <span class="sc-wp-role-badge sc-wp-role-client"><?php echo esc_html(ucfirst($wp_user_role)); ?></span>
                    <?php endif; ?>
                </div>
                <div class="sc-wp-user-actions">
                    <?php if ($wp_user_is_synced): ?>
                        <button id="sc-open-dashboard" class="button button-primary">
                            <?php esc_html_e('Open Dashboard', 'simple-commenter'); ?>
                            <?php echo wp_kses( $icon_external, sc_get_allowed_svg_tags() ); ?>
                        </button>
                    <?php else: ?>
                        <button id="sc-sync-me" class="button button-primary">
                            <?php esc_html_e('Connect to SimpleCommenter', 'simple-commenter'); ?>
                        </button>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Trial/Account Status Banner - Only show for trial users (free/pro tiers) -->
        <?php if ($tier_info['is_trial']): ?>
            <?php $is_expired = $trial && $trial['expired']; ?>
            <div class="sc-banner <?php echo $is_expired ? 'sc-banner-error' : 'sc-banner-warning'; ?>">
                <?php if ($is_expired): ?>
                    <?php echo wp_kses( $icon_x, sc_get_allowed_svg_tags() ); ?>
                    <span>
                        <strong><?php esc_html_e('Trial Expired', 'simple-commenter'); ?></strong> &mdash;
                        <?php esc_html_e('Widget is disabled.', 'simple-commenter'); ?>
                    </span>
                    <button type="button" id="sc-upgrade-btn" class="button button-primary">
                        <?php esc_html_e('Upgrade Now', 'simple-commenter'); ?>
                    </button>
                <?php else: ?>
                    <?php echo wp_kses( $icon_clock, sc_get_allowed_svg_tags() ); ?>
                    <span>
                        <strong><?php esc_html_e('Trial:', 'simple-commenter'); ?></strong>
                        <?php
                        /* translators: %d: number of days remaining in trial */
                        printf( esc_html__( '%d days remaining', 'simple-commenter' ), intval( $trial['days_remaining'] ?? 0 ) );
                        ?>
                    </span>
                    <button type="button" id="sc-upgrade-btn" class="button button-small">
                        <?php esc_html_e('Upgrade', 'simple-commenter'); ?>
                    </button>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <?php if (empty($project_id)): ?>
            <!-- Domain Selection -->
            <div class="sc-card">
                <h2><?php esc_html_e('Select Project', 'simple-commenter'); ?></h2>
                <p><?php esc_html_e('Choose which project to use for this WordPress site:', 'simple-commenter'); ?></p>

                <div id="sc-domains-list" data-current-domain="<?php echo esc_attr($current_site_domain); ?>">
                    <div class="sc-loading">
                        <span class="spinner is-active"></span>
                        <?php esc_html_e('Loading projects...', 'simple-commenter'); ?>
                    </div>
                </div>

                <div class="sc-error" id="sc-add-domain-error" style="display:none;"></div>
            </div>

            <?php if ($can_manage): ?>
            <!-- Disconnect Option -->
            <div class="sc-card sc-card-muted">
                <p class="sc-disconnect-hint"><?php esc_html_e('Want to use a different account?', 'simple-commenter'); ?></p>
                <button type="button" id="sc-disconnect-select" class="button button-link-delete">
                    <?php esc_html_e('Disconnect Account', 'simple-commenter'); ?>
                </button>
            </div>
            <?php endif; ?>
        <?php else: ?>
            <?php
            // Check if domain matches current site
            $project_domain = isset($domain_settings['domain']) ? strtolower($domain_settings['domain']) : '';
            $domain_matches = empty($project_domain) || $project_domain === $current_site_domain ||
                              str_replace('www.', '', $project_domain) === str_replace('www.', '', $current_site_domain);
            ?>

            <!-- Project Header -->
            <div class="sc-project-header">
                <div class="sc-project-info">
                    <h2>
                        <?php echo esc_html(isset($domain_settings['projectName']) ? $domain_settings['projectName'] : $project_id); ?>
                        <?php if (!$domain_matches): ?>
                            <span class="sc-domain-mismatch-badge"><?php esc_html_e('Domain Mismatch', 'simple-commenter'); ?></span>
                        <?php endif; ?>
                    </h2>
                    <code><?php echo esc_html( isset( $domain_settings['domain'] ) ? $domain_settings['domain'] : wp_parse_url( get_site_url(), PHP_URL_HOST ) ); ?></code>
                </div>
            </div>

            <?php if (!$domain_matches): ?>
            <!-- Domain Mismatch Warning -->
            <div class="sc-banner sc-banner-error">
                <?php echo wp_kses( $icon_x, sc_get_allowed_svg_tags() ); ?>
                <span>
                    <strong><?php esc_html_e('Widget will not load', 'simple-commenter'); ?></strong> &mdash;
                    <?php
                    /* translators: %1$s: project domain, %2$s: current site domain */
                    printf( esc_html__( 'This project is configured for %1$s but your site is %2$s.', 'simple-commenter' ), '<code>' . esc_html( $project_domain ) . '</code>', '<code>' . esc_html( $current_site_domain ) . '</code>' );
                    ?>
                </span>
                <button type="button" id="sc-change-project-btn" class="button button-small">
                    <?php esc_html_e('Change Project', 'simple-commenter'); ?>
                </button>
            </div>
            <?php endif; ?>

            <!-- Settings Tabs -->
            <div class="sc-card sc-settings-card">
                <div class="sc-settings-tabs">
                    <button class="sc-settings-tab active" data-settings-tab="comments">
                        <?php esc_html_e('Comments', 'simple-commenter'); ?>
                    </button>
                    <?php if ($can_manage): ?>
                    <button class="sc-settings-tab" data-settings-tab="general">
                        <?php esc_html_e('Settings', 'simple-commenter'); ?>
                    </button>
                    <button class="sc-settings-tab" data-settings-tab="access">
                        <?php esc_html_e('Access', 'simple-commenter'); ?>
                    </button>
                    <button class="sc-settings-tab" data-settings-tab="team">
                        <?php esc_html_e('Team & Clients', 'simple-commenter'); ?>
                    </button>
                    <?php endif; ?>
                </div>

                <!-- Comments Tab -->
                <div class="sc-settings-tab-content" id="settings-tab-comments">
                    <!-- Status Filter -->
                    <div class="sc-comments-filters">
                        <!-- Row 1: Status tabs -->
                        <div class="sc-filter-row sc-filter-row-tabs">
                            <div class="sc-status-tabs" id="sc-status-tabs">
                                <!-- Status tabs will be populated by JS -->
                            </div>
                        </div>
                        <!-- Row 2: Dropdowns -->
                        <div class="sc-filter-row sc-filter-row-dropdowns">
                            <select id="sc-priority-filter" class="sc-filter-select">
                                <option value=""><?php esc_html_e('All Priorities', 'simple-commenter'); ?></option>
                                <option value="urgent"><?php esc_html_e('Urgent', 'simple-commenter'); ?></option>
                                <option value="high"><?php esc_html_e('High', 'simple-commenter'); ?></option>
                                <option value="normal"><?php esc_html_e('Normal', 'simple-commenter'); ?></option>
                                <option value="low"><?php esc_html_e('Low', 'simple-commenter'); ?></option>
                            </select>
                            <select id="sc-slug-filter" class="sc-filter-select">
                                <option value=""><?php esc_html_e('All Pages', 'simple-commenter'); ?></option>
                                <!-- Slugs will be populated by JS -->
                            </select>
                        </div>
                    </div>

                    <!-- Separator -->
                    <div class="sc-filters-separator"></div>

                    <!-- Comments List -->
                    <div id="sc-comments-list">
                        <div class="sc-loading">
                            <span class="spinner is-active"></span>
                            <?php esc_html_e('Loading comments...', 'simple-commenter'); ?>
                        </div>
                    </div>

                    <!-- Load More -->
                    <div id="sc-comments-pagination" style="display:none;">
                        <button type="button" id="sc-load-more" class="button">
                            <?php esc_html_e('Load More', 'simple-commenter'); ?>
                        </button>
                        <span id="sc-comments-count"></span>
                    </div>
                </div>

                <?php if ($can_manage): ?>
                <!-- General Tab -->
                <div class="sc-settings-tab-content" id="settings-tab-general" style="display:none;">
                    <form id="sc-settings-form">
                        <!-- Project Info -->
                        <div class="sc-settings-section">
                            <h4><?php esc_html_e('Project Info', 'simple-commenter'); ?></h4>
                            <div class="sc-field-row">
                                <div class="sc-field">
                                    <label for="project-name"><?php esc_html_e('Project Name', 'simple-commenter'); ?></label>
                                    <input type="text" id="project-name" name="projectName"
                                           value="<?php echo esc_attr(isset($domain_settings['projectName']) ? $domain_settings['projectName'] : ''); ?>">
                                </div>
                                <div class="sc-field">
                                    <label for="project-domain"><?php esc_html_e('Domain', 'simple-commenter'); ?></label>
                                    <input type="text" id="project-domain" name="domain"
                                           value="<?php echo esc_attr(isset($domain_settings['domain']) ? $domain_settings['domain'] : ''); ?>"
                                           placeholder="example.com">
                                    <p class="description"><?php esc_html_e('Update if your site URL has changed', 'simple-commenter'); ?></p>
                                </div>
                            </div>
                            <?php
                            // Check if current site domain matches project domain
                            $project_domain = isset($domain_settings['domain']) ? strtolower(preg_replace('/^www\./', '', $domain_settings['domain'])) : '';
                            if ($project_domain && $project_domain !== $current_site_domain): ?>
                                <div class="sc-banner sc-banner-warning" style="margin-top: 12px;">
                                    <?php echo wp_kses( $icon_warning, sc_get_allowed_svg_tags() ); ?>
                                    <span>
                                        <?php
                                        /* translators: %1$s: project domain, %2$s: current site domain */
                                        printf( esc_html__( 'Domain mismatch: Project domain (%1$s) doesn\'t match this site (%2$s). Update the domain above if needed.', 'simple-commenter' ), '<strong>' . esc_html( $project_domain ) . '</strong>', '<strong>' . esc_html( $current_site_domain ) . '</strong>' );
                                        ?>
                                    </span>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Widget Active -->
                        <div class="sc-settings-section">
                            <div class="sc-field sc-toggle">
                                <label>
                                    <input type="checkbox" name="active"
                                           <?php checked(isset($domain_settings['active']) ? $domain_settings['active'] : true); ?>>
                                    <strong><?php esc_html_e('Widget Active', 'simple-commenter'); ?></strong>
                                </label>
                                <p class="description"><?php esc_html_e('Enable or disable the comment widget on your site', 'simple-commenter'); ?></p>
                            </div>
                        </div>

                        <!-- Theme -->
                        <div class="sc-settings-section">
                            <h4><?php esc_html_e('Theme', 'simple-commenter'); ?></h4>
                            <div class="sc-field-row">
                                <div class="sc-field">
                                    <label for="theme-primary"><?php esc_html_e('Primary Color', 'simple-commenter'); ?></label>
                                    <input type="color" id="theme-primary" name="theme[primary]"
                                           value="<?php echo esc_attr(isset($domain_settings['theme']['primary']) ? $domain_settings['theme']['primary'] : '#2563eb'); ?>">
                                </div>
                                <div class="sc-field">
                                    <label for="theme-overlay"><?php esc_html_e('Overlay Color', 'simple-commenter'); ?></label>
                                    <input type="color" id="theme-overlay" name="theme[overlay]"
                                           value="<?php echo esc_attr(isset($domain_settings['theme']['overlay']) ? $domain_settings['theme']['overlay'] : '#f9fafb'); ?>">
                                </div>
                                <div class="sc-field">
                                    <label for="modal-align"><?php esc_html_e('Widget Position', 'simple-commenter'); ?></label>
                                    <select id="modal-align" name="modalAlign">
                                        <option value="0" <?php selected(isset($domain_settings['modalAlign']) ? $domain_settings['modalAlign'] : 0, 0); ?>><?php esc_html_e('Right', 'simple-commenter'); ?></option>
                                        <option value="1" <?php selected(isset($domain_settings['modalAlign']) ? $domain_settings['modalAlign'] : 0, 1); ?>><?php esc_html_e('Center', 'simple-commenter'); ?></option>
                                        <option value="2" <?php selected(isset($domain_settings['modalAlign']) ? $domain_settings['modalAlign'] : 0, 2); ?>><?php esc_html_e('Left', 'simple-commenter'); ?></option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <!-- Notifications -->
                        <div class="sc-settings-section">
                            <h4><?php esc_html_e('Notifications', 'simple-commenter'); ?></h4>
                            <div class="sc-field sc-toggle">
                                <label>
                                    <input type="checkbox" name="emailNotifications"
                                           <?php checked(isset($domain_settings['emailNotifications']) ? $domain_settings['emailNotifications'] : false); ?>>
                                    <?php esc_html_e('Email Notifications', 'simple-commenter'); ?>
                                </label>
                            </div>
                            <div class="sc-field">
                                <label for="emailFrequency"><?php esc_html_e('Frequency', 'simple-commenter'); ?></label>
                                <select id="emailFrequency" name="emailFrequency">
                                    <option value="hourly" <?php selected(isset($domain_settings['emailFrequency']) ? $domain_settings['emailFrequency'] : 'daily', 'hourly'); ?>><?php esc_html_e('Hourly', 'simple-commenter'); ?></option>
                                    <option value="daily" <?php selected(isset($domain_settings['emailFrequency']) ? $domain_settings['emailFrequency'] : 'daily', 'daily'); ?>><?php esc_html_e('Daily', 'simple-commenter'); ?></option>
                                    <option value="weekly" <?php selected(isset($domain_settings['emailFrequency']) ? $domain_settings['emailFrequency'] : 'daily', 'weekly'); ?>><?php esc_html_e('Weekly', 'simple-commenter'); ?></option>
                                </select>
                            </div>
                        </div>

                        <!-- Features -->
                        <div class="sc-settings-section">
                            <h4><?php esc_html_e('Features', 'simple-commenter'); ?></h4>
                            <div class="sc-features-grid">
                                <label>
                                    <input type="checkbox" name="drawing"
                                           <?php checked(isset($domain_settings['drawing']) ? $domain_settings['drawing'] : true); ?>>
                                    <?php esc_html_e('Drawing Tools', 'simple-commenter'); ?>
                                </label>
                                <label>
                                    <input type="checkbox" name="screenshots"
                                           <?php checked(isset($domain_settings['screenshots']) ? $domain_settings['screenshots'] : true); ?>>
                                    <?php esc_html_e('Screenshots', 'simple-commenter'); ?>
                                </label>
                                <label>
                                    <input type="checkbox" name="uploads"
                                           <?php checked(isset($domain_settings['uploads']) ? $domain_settings['uploads'] : true); ?>>
                                    <?php esc_html_e('File Uploads', 'simple-commenter'); ?>
                                </label>
                                <label>
                                    <input type="checkbox" name="minimized"
                                           <?php checked(isset($domain_settings['minimized']) ? $domain_settings['minimized'] : false); ?>>
                                    <?php esc_html_e('Start Minimized', 'simple-commenter'); ?>
                                </label>
                            </div>
                        </div>

                        <div class="sc-settings-actions">
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Save Settings', 'simple-commenter'); ?>
                            </button>
                        </div>
                    </form>

                    <!-- Project Actions -->
                    <?php if ($can_manage): ?>
                    <div class="sc-settings-section sc-settings-section-actions">
                        <h4><?php esc_html_e('Project', 'simple-commenter'); ?></h4>
                        <div class="sc-settings-links">
                            <button type="button" id="sc-change-project" class="sc-settings-link">
                                <?php esc_html_e('Change Project', 'simple-commenter'); ?>
                                <span class="sc-settings-link-desc"><?php esc_html_e('Switch to a different project', 'simple-commenter'); ?></span>
                            </button>
                            <button type="button" id="sc-disconnect" class="sc-settings-link sc-settings-link-danger">
                                <?php esc_html_e('Disconnect', 'simple-commenter'); ?>
                                <span class="sc-settings-link-desc"><?php esc_html_e('Unlink this WordPress site from SimpleCommenter', 'simple-commenter'); ?></span>
                            </button>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Access Tab -->
                <div class="sc-settings-tab-content" id="settings-tab-access" style="display:none;">
                    <!-- Access Control -->
                    <div class="sc-settings-section">
                        <h4><?php esc_html_e('Access Control', 'simple-commenter'); ?></h4>

                        <div class="sc-access-option">
                            <div class="sc-access-option-header">
                                <label class="sc-toggle-label">
                                    <input type="checkbox" name="conditionalLoading" id="conditionalLoading"
                                           <?php checked(isset($domain_settings['conditionalLoading']) ? $domain_settings['conditionalLoading'] : false); ?>>
                                    <strong><?php esc_html_e('Conditional Script Loading', 'simple-commenter'); ?></strong>
                                </label>
                            </div>
                            <p class="sc-access-option-desc">
                                <?php esc_html_e('Only load widget when', 'simple-commenter'); ?>
                                <code class="sc-inline-code">?simple-commenter=true</code>
                                <?php esc_html_e('is in the URL.', 'simple-commenter'); ?>
                            </p>
                        </div>

                        <div class="sc-access-option">
                            <div class="sc-access-option-header">
                                <label class="sc-toggle-label">
                                    <input type="checkbox" name="tokenAccess" id="tokenAccess"
                                           <?php checked(isset($domain_settings['tokenAccess']) ? $domain_settings['tokenAccess'] : false); ?>>
                                    <strong><?php esc_html_e('Authorized Users Only', 'simple-commenter'); ?></strong>
                                </label>
                            </div>
                            <p class="sc-access-option-desc">
                                <?php esc_html_e('Require users to log in to Simple Commenter to access the widget. WordPress team members and invited clients are automatically logged in.', 'simple-commenter'); ?>
                            </p>
                        </div>

                        <div class="sc-access-option">
                            <div class="sc-access-option-header">
                                <strong><?php esc_html_e('Client Registration Mode', 'simple-commenter'); ?></strong>
                            </div>
                            <p class="sc-access-option-desc">
                                <?php esc_html_e('Control how new clients can register to provide feedback. This does not affect widget authorization.', 'simple-commenter'); ?>
                            </p>
                            <div class="sc-client-access-options">
                                <label class="sc-radio-option" id="clientAccessOpenLabel">
                                    <input type="radio" name="clientAccess" value="open" id="clientAccessOpen"
                                           <?php checked(isset($domain_settings['clientAccess']) ? $domain_settings['clientAccess'] : 'open', 'open'); ?>>
                                    <span class="sc-radio-content">
                                        <strong><?php esc_html_e('Open', 'simple-commenter'); ?></strong>
                                        <span><?php esc_html_e('Anyone can register and provide feedback', 'simple-commenter'); ?></span>
                                    </span>
                                </label>
                                <label class="sc-radio-option">
                                    <input type="radio" name="clientAccess" value="closed"
                                           <?php checked(isset($domain_settings['clientAccess']) ? $domain_settings['clientAccess'] : 'open', 'closed'); ?>>
                                    <span class="sc-radio-content">
                                        <strong><?php esc_html_e('Closed', 'simple-commenter'); ?></strong>
                                        <span><?php esc_html_e('Only invited clients can register', 'simple-commenter'); ?></span>
                                    </span>
                                </label>
                                <label class="sc-radio-option">
                                    <input type="radio" name="clientAccess" value="request"
                                           <?php checked(isset($domain_settings['clientAccess']) ? $domain_settings['clientAccess'] : 'open', 'request'); ?>>
                                    <span class="sc-radio-content">
                                        <strong><?php esc_html_e('Request Access', 'simple-commenter'); ?></strong>
                                        <span><?php esc_html_e('Clients can request access, you approve', 'simple-commenter'); ?></span>
                                    </span>
                                </label>
                            </div>
                        </div>

                        <button type="button" id="sc-save-access" class="button button-primary">
                            <?php esc_html_e('Save Access Settings', 'simple-commenter'); ?>
                        </button>
                    </div>

                    <!-- User Role Access -->
                    <div class="sc-settings-section">
                        <h4><?php esc_html_e('Widget Visibility', 'simple-commenter'); ?></h4>
                        <p class="description"><?php esc_html_e('Control which WordPress user roles can see the comment widget.', 'simple-commenter'); ?></p>

                        <form id="sc-roles-form">
                            <div class="sc-roles-grid">
                                <?php
                                global $wp_roles;
                                $all_roles = $wp_roles->roles;
                                $roles_with_guest = array_merge(
                                    array('guest' => array('name' => 'Guest (Not logged in)')),
                                    $all_roles
                                );

                                foreach ($roles_with_guest as $role_slug => $role_info):
                                    $role_name = is_array($role_info) && isset($role_info['name']) ? $role_info['name'] : $role_slug;
                                    $is_checked = empty($enabled_roles) || in_array($role_slug, $enabled_roles);
                                ?>
                                    <label>
                                        <input type="checkbox" name="roles[]" value="<?php echo esc_attr($role_slug); ?>"
                                               <?php checked($is_checked); ?>>
                                        <?php echo esc_html(translate_user_role($role_name)); ?>
                                    </label>
                                <?php endforeach; ?>
                            </div>

                            <div class="sc-info sc-info-small">
                                <?php echo wp_kses( $icon_info, sc_get_allowed_svg_tags() ); ?>
                                <span><?php esc_html_e('Select all to show widget to everyone. Select specific roles to restrict access.', 'simple-commenter'); ?></span>
                            </div>

                            <button type="submit" class="button button-secondary">
                                <?php esc_html_e('Save Visibility', 'simple-commenter'); ?>
                            </button>
                        </form>
                    </div>

                    <?php if (current_user_can('administrator')): ?>
                    <!-- Management Permissions -->
                    <div class="sc-settings-section">
                        <h4><?php esc_html_e('Management Permissions', 'simple-commenter'); ?></h4>
                        <p class="description"><?php esc_html_e('Which roles can manage plugin settings. These roles also sync as Team Members (counts toward your member limit).', 'simple-commenter'); ?></p>

                        <form id="sc-admin-roles-form">
                            <div class="sc-roles-grid">
                                <?php
                                foreach ($all_roles as $role_slug => $role_info):
                                    $role_name = isset($role_info['name']) ? $role_info['name'] : $role_slug;
                                    $is_checked = in_array($role_slug, $admin_roles);
                                    $is_admin = $role_slug === 'administrator';
                                ?>
                                    <label>
                                        <input type="checkbox" name="admin_roles[]" value="<?php echo esc_attr($role_slug); ?>"
                                               <?php checked($is_checked); ?>
                                               <?php disabled($is_admin); ?>>
                                        <?php echo esc_html(translate_user_role($role_name)); ?>
                                        <?php if ($is_admin): ?>
                                            <span class="sc-required">(<?php esc_html_e('always', 'simple-commenter'); ?>)</span>
                                        <?php endif; ?>
                                    </label>
                                <?php endforeach; ?>
                            </div>

                            <button type="submit" class="button button-secondary">
                                <?php esc_html_e('Save Permissions', 'simple-commenter'); ?>
                            </button>
                        </form>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Team & Clients Tab -->
                <div class="sc-settings-tab-content" id="settings-tab-team" style="display:none;">
                    <!-- WordPress User Sync (moved to top) -->
                    <div class="sc-sync-box">
                        <div class="sc-sync-header">
                            <div class="sc-field sc-toggle">
                                <label>
                                    <input type="checkbox" id="sc-auto-sync-team" name="autoSync"
                                           <?php checked($auto_sync_clients); ?>>
                                    <strong><?php esc_html_e('WordPress User Sync', 'simple-commenter'); ?></strong>
                                </label>
                            </div>
                            <button type="button" id="sc-save-sync" class="button button-small">
                                <?php esc_html_e('Save', 'simple-commenter'); ?>
                            </button>
                        </div>
                        <p class="description"><?php esc_html_e('Automatically sync WordPress users to SimpleCommenter when they visit your site.', 'simple-commenter'); ?></p>
                        <div class="sc-role-mapping-box">
                            <span class="sc-role-mapping-title"><?php esc_html_e('Role mapping:', 'simple-commenter'); ?></span>
                            <ul class="sc-role-mapping-list">
                                <li><strong><?php esc_html_e('WordPress Administrators', 'simple-commenter'); ?></strong> &rarr; <?php esc_html_e('Member (role: user)', 'simple-commenter'); ?></li>
                                <li><strong><?php esc_html_e('Roles with Management Permission', 'simple-commenter'); ?></strong> &rarr; <?php esc_html_e('Member (role: team)', 'simple-commenter'); ?></li>
                                <li><strong><?php esc_html_e('Everyone else', 'simple-commenter'); ?></strong> &rarr; <?php esc_html_e('Client (unlimited)', 'simple-commenter'); ?></li>
                            </ul>
                            <p class="description" style="margin-top: 8px;">
                                <?php esc_html_e('Configure which roles have Management Permission in the Access tab.', 'simple-commenter'); ?>
                            </p>
                        </div>
                        <div class="sc-sync-actions">
                            <button type="button" id="sc-sync-now" class="button button-primary">
                                <?php esc_html_e('Sync Now', 'simple-commenter'); ?>
                            </button>
                            <span id="sc-sync-status" class="sc-sync-status"></span>
                        </div>
                    </div>

                    <!-- Team Members Section -->
                    <div class="sc-settings-section">
                        <h4><?php esc_html_e('Team Members', 'simple-commenter'); ?></h4>
                        <p class="description"><?php esc_html_e('Team members can manage comments in the SimpleCommenter dashboard.', 'simple-commenter'); ?></p>

                        <div id="sc-members-list" class="sc-members-list">
                            <div class="sc-loading">
                                <span class="spinner is-active"></span>
                                <?php esc_html_e('Loading...', 'simple-commenter'); ?>
                            </div>
                        </div>

                        <div class="sc-add-user-form">
                            <h5><?php esc_html_e('Add Team Member', 'simple-commenter'); ?></h5>
                            <div class="sc-field-row sc-field-row-3">
                                <div class="sc-field">
                                    <label for="member-email"><?php esc_html_e('Email', 'simple-commenter'); ?></label>
                                    <input type="email" id="member-email" placeholder="team@example.com">
                                </div>
                                <div class="sc-field">
                                    <label for="member-name"><?php esc_html_e('Name', 'simple-commenter'); ?></label>
                                    <input type="text" id="member-name" placeholder="John Doe">
                                </div>
                                <div class="sc-field">
                                    <label for="member-role"><?php esc_html_e('Role', 'simple-commenter'); ?></label>
                                    <select id="member-role">
                                        <option value="team"><?php esc_html_e('Team', 'simple-commenter'); ?></option>
                                        <option value="user"><?php esc_html_e('Admin', 'simple-commenter'); ?></option>
                                    </select>
                                </div>
                            </div>
                            <button type="button" id="sc-add-member" class="button button-primary" style="margin-top: 12px;">
                                <?php esc_html_e('Add Member', 'simple-commenter'); ?>
                            </button>
                        </div>
                    </div>

                    <!-- Clients Section -->
                    <div class="sc-settings-section">
                        <h4><?php esc_html_e('Clients', 'simple-commenter'); ?></h4>
                        <p class="description"><?php esc_html_e('Clients can leave comments and view their own feedback.', 'simple-commenter'); ?></p>

                        <div id="sc-clients-list" class="sc-clients-list">
                            <div class="sc-loading">
                                <span class="spinner is-active"></span>
                                <?php esc_html_e('Loading...', 'simple-commenter'); ?>
                            </div>
                        </div>

                        <div class="sc-add-user-form">
                            <h5><?php esc_html_e('Add Client', 'simple-commenter'); ?></h5>
                            <div class="sc-field-row sc-field-row-2">
                                <div class="sc-field">
                                    <label for="client-email"><?php esc_html_e('Email', 'simple-commenter'); ?></label>
                                    <input type="email" id="client-email" placeholder="client@example.com">
                                </div>
                                <div class="sc-field">
                                    <label for="client-name"><?php esc_html_e('Name', 'simple-commenter'); ?></label>
                                    <input type="text" id="client-name" placeholder="Jane Doe">
                                </div>
                            </div>
                            <button type="button" id="sc-add-client" class="button button-primary" style="margin-top: 12px;">
                                <?php esc_html_e('Add Client', 'simple-commenter'); ?>
                            </button>
                        </div>
                    </div>

                    <div class="sc-info sc-info-small">
                        <?php echo wp_kses( $icon_info, sc_get_allowed_svg_tags() ); ?>
                        <span><?php esc_html_e('Users can log in at simplecommenter.com with their email address.', 'simple-commenter'); ?></span>
                    </div>
                </div>
                <?php endif; ?><!-- end $can_manage -->
            </div>
        <?php endif; ?>

    <?php endif; ?>

    <div class="sc-error" id="sc-global-error" style="display:none;"></div>
</div>
