initialize_script()
{
  echo "Checking directory structure"
  CATMdir=${ContentDir}
  [ ! -d ${CATMdir} ] && mkdir -p "${CATMdir}"
  if [ ! -d ${CAMFdir} ]; then
    echo "Directory where voxelized meshes are stored (${CATMdir}) could not be created, exiting script"
    exit 1
  fi
  YUVdir=${CATMdir}/texture_video_anchor
  [ ! -d ${YUVdir} ] && mkdir -p "${YUVdir}"
  if [ ! -d ${YUVdir} ]; then
    echo "Directory where input YUV/RGB files are stored (${YUVdir}) could not be created, exiting script"
    exit 1
  fi
  DRACOdir=${WorkDir}/DRACO
  [ ! -d ${DRACOdir} ] && mkdir -p "${DRACOdir}"
  if [ ! -d ${DRACOdir} ]; then
    echo "Directory where DRACO coding results will be stored (${DRACOdir}) could not be created, exiting script"
    exit 1
  fi
  HMdir=${WorkDir}/HM
  [ ! -d ${HMdir} ] && mkdir -p "${HMdir}"
  if [ ! -d ${HMdir} ]; then
    echo "Directory where HM coding results will be stored (${HMdir}) could not be created, exiting script"
    exit 1
  fi
  MMETRICdir=${WorkDir}/MMETRIC_VE41
  [ ! -d ${MMETRICdir} ] && mkdir -p "${MMETRICdir}"
  if [ ! -d ${MMETRICdir} ]; then
    echo "Directory where mmetric results will be stored (${MMETRICdir}) could not be created, exiting script"
    exit 1
  fi
  RESULTSdir=${WorkDir}/RESULTS
  [ ! -d ${RESULTSdir} ] && mkdir -p "${RESULTSdir}"
  if [ ! -d ${RESULTSdir} ]; then
    echo "Directory where simulation results will be stored (${RESULTSdir}) could not be created, exiting script"
    exit 1
  fi
  SCRIPTSdir=${ScriptsDir}
  if [ ! -d ${SCRIPTSdir} ]; then
    echo "Directory where configuration files will be stored (${SCRIPTSdir}) does not exist, please download from git"
    echo "git clone http://mpegx.int-evry.fr/software/MPEG/PCC/mpeg-vmesh-anchor.git"
    exit 1
  fi
  
  echo "Checking required software"
  dracoDecSW=${SWDir}/draco/build/draco_decoder
  if [ ! -f ${dracoEncSW} ]; then
    echo "DRACO encoder SW (${dracoEncSW}) does not exist, please download and compile using the following command:"
    echo "cd ${SWDir}"
    echo "git clone https://github.com/google/draco.git"
    echo "mkdir draco/build && cd draco/build"
    echo "cmake ../"
    echo "make"
    exit 1
  fi
  hmDecSW=${SWDir}/HM/bin/TAppDecoderStatic
  if [ ! -f ${hmEncSW} ]; then
    echo "HM encoder SW (${hmEncSW}) does not exist, please download and compile using the following command:"
    echo "cd ${SWDir}"
    echo "git clone --branch HM-16.21+SCM-8.8 https://vcgit.hhi.fraunhofer.de/jvet/HM.git"
    echo "mkdir HM/build_dir && cd HM/build_dir"
    echo "cmake ../"
    exit 1
  fi
  hdrToolsSW=${SWDir}/HDRTools/build/bin/HDRConvert
  if [ ! -f ${hdrToolsSW} ]; then
    echo "HDRTools SW (${hdrToolsSW}) does not exist, please download and compile using the following command:"
    echo "cd ${SWDir}"
    echo "git clone https://gitlab.com/standards/HDRTools.git"
    echo "cd HDRTools"
    echo "mkdir build && cd build"
    echo "cmake .. -DCMAKE_BUILD_TYPE=Release -DLIBPNG=ON"
    echo "make"
    exit 1
  fi
  meshlabserverSW=/usr/bin/meshlabserver
  if [ ! -f ${meshlabserverSW} ]; then
    echo "meshlabserver SW (${meshlabserverSW}) does not exist, please install meshlab"
    exit 1
  fi
  mmetricSW=${SWDir}/mpeg-pcc-mmetric/bin/mm 
  mmetricSW=${SWDir}/mpeg-pcc-mmetric_VE41/bin/mm 
  if [ ! -f ${mmetricSW} ]; then
    echo "mmetric SW (${mmetricSW}) does not exist, please download and compile using the following command:"
    echo "cd ${SWDir}"
    echo "git clone http://mpegx.int-evry.fr/software/MPEG/PCC/mpeg-pcc-mmetric.git --branch version_0_1_12 mpeg-pcc-mmetric_VE41"
    echo "cd mpeg-pcc-mmetric_VE41"
    echo "./build.sh release"
    exit 1
  fi
  ffmpegSW=/usr/bin/ffmpeg
  if [ ! -f ${ffmpegSW} ]; then
    echo "ffmpeg SW (${ffmpegSW}) does not exist, please install ffmpeg"
    exit 1
  fi
}

load_sequence_parameters()
{
  if [ ${seq} == 'longdress' ]; then
    #sequence parameters
    seqNum=01
    startFr=1051
    lastFr=1350
    textMapDim=2048
    numFrames=300
    qpMesh=10
    qtMesh=12
    #metrics parameters
    lengthTh=0.515014648
    gridRes=2048
    faceRes=2055
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'soldier' ]; then
    #sequence parameters
    seqNum=02
    startFr=536 
    lastFr=835  
    textMapDim=2048
    numFrames=300
    qpMesh=10
    qtMesh=12
    #metrics parameters
    lengthTh=0.599487305 
    gridRes=2048
    faceRes=1784
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'basketball_player' ]; then
    #sequence parameters
    seqNum=03
    startFr=1 
    lastFr=600
    textMapDim=2048
    numFrames=300
    qpMesh=12
    qtMesh=12
    #metrics parameters
    lengthTh=1.04980469    
    gridRes=2048
    faceRes=1941
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'dancer' ]; then
    #sequence parameters
    seqNum=04
    startFr=1 
    lastFr=600
    textMapDim=2048
    numFrames=300
    qpMesh=12
    qtMesh=12
    #metrics parameters
    lengthTh=0.985595703    
    gridRes=2048
    faceRes=2010
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'mitch' ]; then
    #sequence parameters
    seqNum=05
    startFr=1
    lastFr=475 
    textMapDim=4096
    numFrames=300
    qpMesh=12
    qtMesh=13
    #metrics parameters
    lengthTh=1.09912109  
    gridRes=2048
    faceRes=1787
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'thomas' ]; then
    #sequence parameters
    seqNum=06
    startFr=618
    lastFr=917 
    textMapDim=4096
    numFrames=300
    qpMesh=12
    qtMesh=13
    #metrics parameters
    lengthTh=1.03942871 
    gridRes=2048
    faceRes=1903
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'football' ]; then
    #sequence parameters
    seqNum=07
    startFr=445
    lastFr=744  
    textMapDim=4096
    numFrames=300
    qpMesh=12
    qtMesh=13
    #metrics parameters
    lengthTh=0.4425  
    gridRes=2048
    faceRes=1845
    targetPC=2000000
    camRes=2048
  fi
  if [ ${seq} == 'levi' ]; then
    #sequence parameters
    seqNum=08
    startFr=0
    lastFr=149 
    textMapDim=4096
    numFrames=150
    qpMesh=12
    qtMesh=13
    #metrics parameters
    lengthTh=0.00200637984 
    gridRes=2048
    faceRes=1614
    targetPC=2000000
    camRes=2048
  fi
  #coding parameters
  codingParameters=${SCRIPTSdir}/cfg/parameters/${seq}.csv

  #loading the sequence bounding box (will be used later for AFX lossless coding)
  source ${SCRIPTSdir}/statistics_anchor/${seq}_statistics_all.log
  echo "Using bbox :[(${globalMinPos}),(${globalMaxPos})]"

  echo "Checking if encoded bitstreams are available"
  #download the content from the MPEG repository
  [ ! -d ${RESULTSdir}/bitstreams ] && mkdir  ${RESULTSdir}/bitstreams
  #now reading parameters from configuration file and collecting simulation results from HM, DRACO, and mmetric
  numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`
  #generating video for each rate point
  for ((rate=1;rate<=$((numLines-1));rate++)); do
    #reading the coding parameters
    qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
    qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
    meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
    qm=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 5 -d","`
    textureResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 6 -d","`
    textMapSize=$((textMapDim / textureResolution))
    codingCode=0
    if [[ ${qp} == -1 ]]; then
      continue
    fi
    for codingMode in {AI,RA}; do
      codingCode=$((codingCode+1))
      bitstreamZIPFile=${RESULTSdir}/bitstreams/P00S${seqNum}C${codingCode}R${rate}.zip
      bitstreamMD5File=${RESULTSdir}/bitstreams/P00S${seqNum}C${codingCode}R${rate}.md5
      if [ ! -f ${bitstreamZIPFile} ]; then
        echo "ERROR: Bitstream is missing, please download the bitstream from the MPEG repository"
        echo "cd ${RESULTSdir}/bitstreams"
        echo "wget https://mpegfs.int-evry.fr/mpegcontent/ws-mpegcontent/CfP/Mesh/CONTENT/bitstreams/P00S${seqNum}C${codingCode}R${rate}.zip"
        echo "wget https://mpegfs.int-evry.fr/mpegcontent/ws-mpegcontent/CfP/Mesh/CONTENT/bitstreams/P00S${seqNum}C${codingCode}R${rate}.md5"
        exit 1
      else
        [ ! -d ${HMdir}/${seq} ] && mkdir ${HMdir}/${seq}
        [ ! -d ${HMdir}/${seq}/${codingMode} ] && mkdir ${HMdir}/${seq}/${codingMode}
        [ ! -d ${HMdir}/${seq}/${codingMode}/${textureResolution} ] && mkdir ${HMdir}/${seq}/${codingMode}/${textureResolution}
        [ ! -d ${DRACOdir}/${seq} ] && mkdir ${DRACOdir}/${seq} 
        [ ! -d ${DRACOdir}/${seq}/${meshResolution} ] && mkdir ${DRACOdir}/${seq}/${meshResolution}
        #cheking if all the files are available
        unzipFile=0
        compTexture=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}.bin
        if [ ! -f ${compTexture} ]; then
          echo "${compTexture} does not exist, unzipping ${bitstreamZIPFile}"
          unzipFile=1
        else
          echo -ne "${compTexture} is available\033[0K\r"
        fi
        for (( i=0; i<${numFrames}; i++ )); do
          f=$((i + startFr))
          fr=$(printf "%04d" $f)
          compMesh=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr${fr}.drc
          if [ ! -f ${compMesh} ]; then
            echo "${compMesh} does not exist, unzipping ${bitstreamZIPFile}"
            unzipFile=1
            break
          else
            echo -ne "${compMesh} is available\033[0K\r"
          fi
        done
        if [ ${unzipFile} == 1 ]; then
          echo "mkdir tmp_bitstream"
          mkdir tmp_bitstream
          echo "unzip  ${bitstreamZIPFile} -d  tmp_bitstream"
          unzip  ${bitstreamZIPFile} -d  tmp_bitstream
          #now check the texture md5sum and move the texture file to the correct directory 
          refTextMd5Sum=`cat ${bitstreamMD5File} | grep ${seq}_QP${qm}.bin | awk '{ print $1 }'`
          localTextMd5Sum=`md5sum tmp_bitstream/${seq}_QP${qm}.bin | awk '{ print \$1 }'`
          if [ ${localTextMd5Sum} != ${refTextMd5Sum} ]; then
            echo "local texture md5sum (${localTextMd5Sum}) does not match the reference texture ${refTextMd5Sum}"
            rm -Rf tmp_bitstream
            exit 1
          else
            mv tmp_bitstream/${seq}_QP${qm}.bin ${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}.bin
          fi
          #now checking if the mesh md5sum is correct and move the mesh files to the correct directory
          for (( i=0; i<${numFrames}; i++ )); do
            f=$((i + startFr))
            fr=$(printf "%04d" $f)
            refMeshMd5Sum=`cat ${bitstreamMD5File} | grep ${seq}_fr${fr}.drc | awk '{ print $1 }'`
            localMeshMd5Sum=`md5sum tmp_bitstream/${seq}_fr${fr}.drc | awk '{ print \$1 }'`
            if [ ${localMeshMd5Sum} != ${refMeshMd5Sum} ]; then
              echo "local mesh md5sum (${localMeshMd5Sum}) does not match the reference md5sum ${refMeshMd5Sum}"
              rm -Rf tmp_bitstream
              exit 1
            else
             mv tmp_bitstream/${seq}_fr${fr}.drc ${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr${fr}.drc
            fi
          done
          rm -Rf tmp_bitstream
        fi
      fi
    done
  done
  #clear the last line for readability
  echo -ne "\033[0K\r"
}

run_HM_lossy ()
{
  #running HM decoder for AI and RA conditions
  numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`
  for ((rate=1;rate<=$((numLines-1));rate++)); do
    #reading the coding parameters
    qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
    qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
    meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
    qm=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 5 -d","`
    textureResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 6 -d","`
    textMapSize=$((textMapDim / textureResolution))
	  if [[ ${qp} == -1 ]]; then
      continue
    fi
    for codingMode in {AI,RA}; do
      echo -ne "Lossy ${codingMode}\033[0K\r"
      outBin=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}.bin
      outRGB=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}_${textMapSize}x${textMapSize}_yuv420p.yuv
      outLog=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}_DEC.log
      if [ -f ${outBin} ]; then
        #check if decoding has been done
        if [ ! -f ${outRGB} ]; then
          echo ""
          echo Running HM decoder
          echo "${hmDecSW} --BitstreamFile=${outBin} --ReconFile=${outRGB} --OutputBitDepth=8 --OutputBitDepthC=8 > ${outLog}" 
                ${hmDecSW} --BitstreamFile=${outBin} --ReconFile=${outRGB} --OutputBitDepth=8 --OutputBitDepthC=8 > ${outLog} 
        fi
        #check if the video file has been converted to a sequence of PNG
        [ ! -d ${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate} ] && mkdir ${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate}
        pngDir=${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate}
        doConversion=0
        #now checking if all the files exist
        for (( i=0; i<${numFrames}; i++ )); do
          f=$((i + startFr))
          fr=$(printf "%04d" $f)
          pngFile=${pngDir}/${seq}_fr${fr}.png
          if [ ! -f  ${pngFile} ]; then
            echo "PNG file (${pngFile}) missing, doing conversion"
            doConversion=1
            break
          fi
        done
        if [ ${doConversion} == 1 ]; then
          #first transform the GBR PNG files with zero-based indices
          echo ${hdrToolsSW} -f ${SCRIPTSdir}/cfg/hdrconvert/yuv420togbr8.cfg -p SourceFile="${outRGB}" -p Outputfile="${pngDir}/tmp_fr%04d.png" -p SourceWidth=${textMapSize} -p SourceHeight=${textMapSize} -p NumberOfFrames=${numFrames} -p OutputWidth=${textMapDim} -p OutputHeight=${textMapDim}
               ${hdrToolsSW} -f ${SCRIPTSdir}/cfg/hdrconvert/yuv420togbr8.cfg -p SourceFile="${outRGB}" -p Outputfile="${pngDir}/tmp_fr%04d.png" -p SourceWidth=${textMapSize} -p SourceHeight=${textMapSize} -p NumberOfFrames=${numFrames} -p OutputWidth=${textMapDim} -p OutputHeight=${textMapDim}
          #now change the name to the appropriate naming convention
          for (( i=0; i<${numFrames}; i++ )); do
            frHDR=$(printf "%04d" $i)
            f=$((i + startFr))
            fr=$(printf "%04d" $f)
            mv ${pngDir}/tmp_fr${frHDR}.png ${pngDir}/${seq}_fr${fr}.png
          done
        fi
      else
        echo "${outBin} does not exit, check if the files are present"
        exit 1
      fi
    done
  done
  #clear the last line for readability
  echo -ne "\033[0K\r"
}

run_DRACO_lossy()
{
  #now reading QP parameters from configuration file and running HM encoder
  numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`
  for ((rate=1;rate<=$((numLines-1));rate++)); do
    #reading the coding parameters
    qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
    qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
    meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
  	if [[ ${qp} == -1 ]]; then
      continue
    fi
    echo "Running R${rate} with QP=${qp}, QT=${qt} and Mesh Resolution=${meshResolution}"
    #creating the output directory
    [ ! -d ${DRACOdir}/${seq}/${meshResolution} ] && mkdir ${DRACOdir}/${seq}/${meshResolution}
    [ ! -d ${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt} ] && mkdir ${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}
  	outputDir=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}
	
    #geometryBitrate=0
    for (( i=0; i<${numFrames}; i++ )); do
      f=$((i + startFr))
      fr=$(printf "%04d" $f)
      echo -ne "seq=${seq} fr=${fr} qp=${qp} qt=${qt} meshResolution=${meshResolution}\033[0K\r"
      compMesh=${outputDir}/${seq}_fr${fr}
      if  [ -f ${compMesh}.obj ]; then
        echo -ne "OBJ file exists\033[0K\r"
      else
        echo "${dracoDecSW} -i ${compMesh}.drc -o ${compMesh}.obj >> ${compMesh}_DEC.log"
              ${dracoDecSW} -i ${compMesh}.drc -o ${compMesh}.obj >> ${compMesh}_DEC.log
      fi
    done;
  done;
  #clear the last line for readability
  echo -ne "\033[0K\r"
}

cmd_ibsm()
{
  compressedMesh=$1
  compressedTexture=$2
  log=$3
  outputCsv=$4
  endFr=$((numFrames + startFr - 1))
  echo "${mmetricSW} \
        sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
        compare --mode ibsm --ibsmResolution ${camRes} --ibsmRenderer gl12_ibsm --ibsmCameraRotation \"0.0 0.0 0.0\" --inputModelA ${refMesh} --inputMapA ${refTexture} --inputModelB ${compressedMesh} --inputMapB ${compressedTexture} --outputCsv ${outputCsv} > ${log}"
        ${mmetricSW} \
        sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
        compare --mode ibsm --ibsmResolution ${camRes} --ibsmRenderer gl12_ibsm  --ibsmCameraRotation "0.0 0.0 0.0" --inputModelA ${refMesh} --inputMapA ${refTexture} --inputModelB ${compressedMesh} --inputMapB ${compressedTexture} --outputCsv ${outputCsv} > ${log}
}

cmd_ibsm_90degree()
{
  compressedMesh=$1
  compressedTexture=$2
  log=$3
  outputCsv=$4
  endFr=$((numFrames + startFr - 1))
  echo "${mmetricSW} \
        sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
        compare --mode ibsm --ibsmResolution ${camRes} --ibsmRenderer gl12_ibsm --ibsmCameraRotation \"0.0 0.0 90.0\" --inputModelA ${refMesh} --inputMapA ${refTexture} --inputModelB ${compressedMesh} --inputMapB ${compressedTexture} --outputCsv ${outputCsv} > ${log}"
        ${mmetricSW} \
        sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
        compare --mode ibsm --ibsmResolution ${camRes} --ibsmRenderer gl12_ibsm  --ibsmCameraRotation "0.0 0.0 90.0" --inputModelA ${refMesh} --inputMapA ${refTexture} --inputModelB ${compressedMesh} --inputMapB ${compressedTexture} --outputCsv ${outputCsv} > ${log}
}

cmd_ibsm_45degree()
{
  compressedMesh=$1
  compressedTexture=$2
  log=$3
  outputCsv=$4
  endFr=$((numFrames + startFr - 1))
  echo "${mmetricSW} \
        sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
        compare --mode ibsm --ibsmResolution ${camRes} --ibsmRenderer gl12_ibsm --ibsmCameraRotation \"45.0 45.0 45.0\" --inputModelA ${refMesh} --inputMapA ${refTexture} --inputModelB ${compressedMesh} --inputMapB ${compressedTexture} --outputCsv ${outputCsv} > ${log}"
        ${mmetricSW} \
        sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
        compare --mode ibsm --ibsmResolution ${camRes} --ibsmRenderer gl12_ibsm  --ibsmCameraRotation "45.0 45.0 45.0" --inputModelA ${refMesh} --inputMapA ${refTexture} --inputModelB ${compressedMesh} --inputMapB ${compressedTexture} --outputCsv ${outputCsv} > ${log}
}

cmd_prnd()
{
  compressedMesh=$1
  compressedTexture=$2
  log=$3
  outputCsv=$4
  endFr=$((numFrames + startFr - 1))
  echo "${mmetricSW} \
    sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
    reindex --sort oriented -i ${refMesh} -o ID:ref_reordered END \
    sample --mode prnd --bilinear --nbSamples ${targetPC} -i ID:ref_reordered -m ${refTexture} -o ID:pcA --hideProgress 1 END \
    reindex --sort oriented -i ${compressedMesh} -o ID:comp_reordered END \
    sample --mode prnd --bilinear --nbSamples ${targetPC} -i ID:comp_reordered -m ${compressedTexture} -o ID:pcB --hideProgress 1 END \
    compare --mode pcc --resolution 1024 --inputModelA ID:pcA --inputModelB ID:pcB END \
    compare --mode pcqm --inputModelA ID:pcA --inputModelB ID:pcB --outputCsv ${outputCsv} > ${log}"
  ${mmetricSW} \
    sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
    reindex --sort oriented -i ${refMesh} -o ID:ref_reordered END \
    sample --mode prnd --bilinear --nbSamples ${targetPC} -i ID:ref_reordered -m ${refTexture} -o ID:pcA --hideProgress 1 END \
    reindex --sort oriented -i ${compressedMesh} -o ID:comp_reordered END \
    sample --mode prnd --bilinear --nbSamples ${targetPC} -i ID:comp_reordered -m ${compressedTexture} -o ID:pcB --hideProgress 1 END \
    compare --mode pcc --resolution 1024 --inputModelA ID:pcA --inputModelB ID:pcB END \
    compare --mode pcqm --inputModelA ID:pcA --inputModelB ID:pcB --outputCsv ${outputCsv} > ${log} &
}

cmd_grid()
{
  compressedMesh=$1
  compressedTexture=$2
  log=$3
  outputCsv=$4
  endFr=$((numFrames + startFr - 1))
  echo "${mmetricSW} \
    sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
    reindex --sort oriented -i ${refMesh} -o ID:ref_reordered END \
    sample --mode grid --useNormal --useFixedPoint --minPos "${globalMinPos}" --maxPos "${globalMaxPos}" --bilinear --gridSize ${gridRes} -i ID:ref_reordered -m ${refTexture} -o ID:pcA --hideProgress 1 END \
    reindex --sort oriented -i ${compressedMesh} -o ID:comp_reordered END \
    sample --mode grid --useNormal --useFixedPoint --minPos "${globalMinPos}" --maxPos "${globalMaxPos}" --bilinear --gridSize ${gridRes} -i ID:comp_reordered -m ${compressedTexture} -o ID:pcB --hideProgress 1 END \
    compare --mode pcc --resolution 1024 --inputModelA ID:pcA --inputModelB ID:pcB END \
    compare --mode pcqm --inputModelA ID:pcA --inputModelB ID:pcB --outputCsv ${outputCsv} > ${log}"
  ${mmetricSW} \
    sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
    reindex --sort oriented -i ${refMesh} -o ID:ref_reordered END \
    sample --mode grid --useNormal --useFixedPoint --minPos "${globalMinPos}" --maxPos "${globalMaxPos}" --bilinear --gridSize ${gridRes} -i ID:ref_reordered -m ${refTexture} -o ID:pcA --hideProgress 1 END \
    reindex --sort oriented -i ${compressedMesh} -o ID:comp_reordered END \
    sample --mode grid --useNormal --useFixedPoint --minPos "${globalMinPos}" --maxPos "${globalMaxPos}" --bilinear --gridSize ${gridRes} -i ID:comp_reordered -m ${compressedTexture} -o ID:pcB --hideProgress 1 END \
    compare --mode pcc --resolution 1024 --inputModelA ID:pcA --inputModelB ID:pcB END \
    compare --mode pcqm --inputModelA ID:pcA --inputModelB ID:pcB --outputCsv ${outputCsv} > ${log} &
}

cmd_ediv()
{
  compressedMesh=$1
  compressedTexture=$2
  log=$3
  outputCsv=$4
  endFr=$((numFrames + startFr - 1))
  echo "${mmetricSW} \
    sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
    reindex --sort oriented -i ${refMesh} -o ID:ref_reordered END \
    sample --mode ediv --bilinear --lengthThreshold ${lengthTh} -i ID:ref_reordered -m ${refTexture} -o ID:pcA --hideProgress 1 END \
    reindex --sort oriented -i ${compressedMesh} -o ID:comp_reordered END \
    sample --mode ediv --bilinear --lengthThreshold ${lengthTh} -i ID:comp_reordered -m ${compressedTexture} -o ID:pcB --hideProgress 1 END \
    compare --mode pcc --resolution 1024 --inputModelA ID:pcA --inputModelB ID:pcB END \
    compare --mode pcqm --inputModelA ID:pcA --inputModelB ID:pcB --outputCsv ${outputCsv} > ${log}"
  ${mmetricSW} \
    sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
    reindex --sort oriented -i ${refMesh} -o ID:ref_reordered END \
    sample --mode ediv --bilinear --lengthThreshold ${lengthTh} -i ID:ref_reordered -m ${refTexture} -o ID:pcA --hideProgress 1 END \
    reindex --sort oriented -i ${compressedMesh} -o ID:comp_reordered END \
    sample --mode ediv --bilinear --lengthThreshold ${lengthTh} -i ID:comp_reordered -m ${compressedTexture} -o ID:pcB --hideProgress 1 END \
    compare --mode pcc --resolution 1024 --hausdorff --inputModelA ID:pcA --inputModelB ID:pcB END \
    compare --mode pcqm --inputModelA ID:pcA --inputModelB ID:pcB --outputCsv ${outputCsv} > ${log} &
}

run_mmetric_lossy()
{
  #creating directory structure for results
  [ ! -d ${MMETRICdir}/${seq} ] && mkdir ${MMETRICdir}/${seq}

  #reference meshes
  refMesh=${CATMdir}/${seq}/${seq}_fr%04d.obj
  refTexture=${CATMdir}/${seq}/${seq}_fr%04d.png

  #now reading QP parameters from configuration file and running mmetric software
  numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`
  for ((rate=1;rate<=$((numLines-1));rate++)); do
    #reading the coding parameters
    qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
    qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
    meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
    qm=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 5 -d","`
    textureResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 6 -d","`
    textMapSize=$((textMapDim / textureResolution))
  	if [[ ${qp} == -1 ]]; then
      continue
    fi
    for codingMode in {AI,RA}; do
	    #creating output log file
      logFile=${MMETRICdir}/${seq}/mmetric_$1_${numFrames}_frames_QP${qp}_QT${qt}_${codingMode}_${meshResolution}_QM${qm}_${textureResolution}.log
      csvFile=${MMETRICdir}/${seq}/mmetric_$1_${numFrames}_frames_QP${qp}_QT${qt}_${codingMode}_${meshResolution}_QM${qm}_${textureResolution}.csv
      echo -ne "Running $1 metric with ${numFrames} frames for rate R${rate} with QP=${qp}, QT=${qt}, Mesh Resolution=${meshResolution}, HM ${codingMode} Compression (QP=${qm} and Texture Resolution=1/${textureResolution})\033[0K\r"
	    #compressed meshes
	    compMesh=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr%04d.obj
      compTexture=${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate}/${seq}_fr%04d.png
      if  [ ! -f ${logFile} ]; then
        echo "mmetric for ${compMesh} and ${compTexture} texture."
        case $1 in
          ediv)
            cmd_ediv ${compMesh} ${compTexture} ${logFile} ${csvFile}
          ;;
          grid)
            cmd_grid ${compMesh} ${compTexture} ${logFile} ${csvFile}
          ;;
          prnd)
            cmd_prnd ${compMesh} ${compTexture} ${logFile} ${csvFile}
          ;;
          ibsm)
            cmd_ibsm ${compMesh} ${compTexture} ${logFile} ${csvFile}
          ;;
          ibsm_90degree)
            cmd_ibsm_90degree ${compMesh} ${compTexture} ${logFile} ${csvFile}
          ;;
          ibsm_45degree)
            cmd_ibsm_45degree ${compMesh} ${compTexture} ${logFile} ${csvFile}
          ;;
        esac
      else
        echo -ne "${logFile} already exist.\033[0K\r" 
        echo -ne "\033[0K\r" 
      fi
    done;
  done;
}

collect_draco_results ()
{
  #checking the encoded file
  dracoBin=$1
  #initializing the variables
  totalGeometryBytes_val=0
  #checking the total bitrate
  if  [ -f ${dracoBin} ]; then
    echo -ne "Reading the size of ${dracoBin}\033[0K\r"
    totalGeometryBytes_val=$(stat -c %s ${dracoBin})
  else
    echo "${dracoBin} does not exists"
  fi
  #checking the encoder log
  dracoEncLog=$2
  #initializing the variables
  connBitrate_val=0
  connTexBitrate_val=0
  vertexBitrate_val=0
  textCoordBitrate_val=0
  geoEncTime_val=0
  if  [ -f ${dracoEncLog} ]; then
    geoEncTime_val=$(cat ${dracoEncLog} | grep "to encode" | awk '{print $6}')
    geoEncTime_val=${geoEncTime_val:1}
  else
    echo "${dracoEncLog} does not exists"
  fi
  #checking the encoder log
  dracoDecLog=$3
  #initializing the variables
  geoDecTime_val=0
  if  [ -f ${dracoDecLog} ]; then
    geoDecTime_val=$(cat ${dracoDecLog} | grep "to decode" | awk '{print $6}')
    geoDecTime_val=${geoDecTime_val:1}
  else
    echo "${dracoDecLog} does not exists"
  fi
}

collect_encoder_results_all()
{
  binPattern=$1
  encPattern=$2
  decPattern=$3
  summaryLog=$4
  if [ -f ${summaryLog} ]; then
    #echo "Reading ${summaryLog}"
    connBitrate=$(cat ${summaryLog} | grep "ConnectivityVertexRate" | cut -f 2 -d",")
    connTexBitrate=$(cat ${summaryLog} | grep "ConnectivityTextureRate" | cut -f 2 -d",")
    connTotal=$(cat ${summaryLog} | grep "ConnectivityTotalRate" | cut -f 2 -d",")
    vertexBitrate=$(cat ${summaryLog} | grep "VertexPositionRate" | cut -f 2 -d",")
    textCoordBitrate=$(cat ${summaryLog} | grep "TextureCoordinateRate" | cut -f 2 -d",")
    totalGeometryBitrate=$(cat ${summaryLog} | grep "TotalGeometryRate" | cut -f 2 -d",")
    totalGeometryBytes=$(cat ${summaryLog} | grep "TotalGeometry(bytes)" | cut -f 2 -d",")
    geoEncTime=$(cat ${summaryLog} | grep "GeoEncTime" | cut -f 2 -d",")
    geoDecTime=$(cat ${summaryLog} | grep "GeoDecTime" | cut -f 2 -d",")
  else
    #variable reset
    connBitrate=0
    connTexBitrate=0
    connTotal=0
    vertexBitrate=0
    textCoordBitrate=0
    totalGeometryBitrate=0
    totalGeometryBytes=0
    geoEncTime=0
    geoDecTime=0
    for (( i=0; i<${numFrames}; i++ )); do
      f=$((i + startFr))
      fr=$(printf "%04d" $f)
      binFile=$(printf ${binPattern} $f)
      encLog=$(printf ${encPattern} $f)
      decLog=$(printf ${decPattern} $f)
      collect_draco_results ${binFile} ${encLog} ${decLog}
      echo -ne "${fr},${r},${qp},${qt},${connBitrate_val},${connTexBitrate_val},$((connBitrate_val+connTexBitrate_val)),${vertexBitrate_val},${textCoordBitrate_val},${totalGeometryBytes_val},${geoEncTime_val},${geoDecTime_val}\033[0K\r"
      #accumulating the frame results
      connBitrate=`echo "${connBitrate} + ${connBitrate_val}" | bc`
      connTexBitrate=`echo "${connTexBitrate} + ${connTexBitrate_val}" | bc`
      vertexBitrate=`echo "${vertexBitrate} + ${vertexBitrate_val}" | bc`
      textCoordBitrate=`echo "${textCoordBitrate} + ${textCoordBitrate_val}" | bc`
      totalGeometryBytes=`echo "${totalGeometryBytes} + ${totalGeometryBytes_val}" | bc`
      geoEncTime=`echo "${geoEncTime} + ${geoEncTime_val}" | bc`
      geoDecTime=`echo "${geoDecTime} + ${geoDecTime_val}" | bc`
    done;
    #now diving by the number of frames and saving it to the log file
    connBitrate=`echo "scale=10; ${connBitrate} * 8.0 * 30.0 / (${numFrames}*1000000.0)" | bc -l`
    connTexBitrate=`echo "scale=10; ${connTexBitrate} * 8.0 * 30.0 / (${numFrames}*1000000.0)" | bc -l`
    connTotal=`echo "scale=10; ${connTexBitrate} + ${connBitrate}" | bc -l`
    vertexBitrate=`echo "scale=10; ${vertexBitrate} * 8.0 * 30.0 / (${numFrames}*1000000.0)" | bc -l`
    textCoordBitrate=`echo "scale=10; ${textCoordBitrate} * 8.0 * 30.0 / (${numFrames}*1000000.0)" | bc -l`
    totalGeometryBitrate=`echo "scale=10; ${totalGeometryBytes} * 8.0 * 30.0 / (${numFrames}*1000000.0)" | bc -l` 
    echo -ne "\033[0K\r"
    echo "ConnectivityVertexRate(Mbps),${connBitrate}" > ${summaryLog}
    echo "ConnectivityTextureRate(Mbps),${connTexBitrate}" >> ${summaryLog}
    echo "ConnectivityTotalRate(Mbps),${connTotal}" >> ${summaryLog}
    echo "VertexPositionRate(Mbps),${vertexBitrate}" >> ${summaryLog}
    echo "TextureCoordinateRate(Mbps),${textCoordBitrate}" >> ${summaryLog}
    echo "TotalGeometryRate(Mbps),${totalGeometryBitrate}" >> ${summaryLog}
    echo "TotalGeometry(bytes),${totalGeometryBytes}" >> ${summaryLog}
    echo "GeoEncTime(s),${geoEncTime}" >> ${summaryLog}
    echo "GeoDecTime(s),${geoDecTime}" >> ${summaryLog}
  fi
}

collect_HM_results ()
{
  hmFile=$1
  textureBitrate=0
  textureBytes=0
  if [ -f ${hmFile} ]; then
    textureBytes=$(stat -c %s ${hmFile})
    #echo "TEXTURE BYTES = ${textureBytes} bytes"
    textureBitrate=`echo "scale=10; ${textureBytes} * 8.0 * 30.0 / (${numFrames}*1000000.0)" | bc -l`
    #echo "TEXTURE BITRATE = ${textureBitrate} Mbps"
  else
    echo "${hmFile} does not exists"
  fi
  hmEncLog=$2
  textMap_Y_PSNR=inf
  textMap_U_PSNR=inf
  textMap_V_PSNR=inf
  textureEncTime=0
  if [ -f ${hmEncLog} ]; then
    #echo "Reading ${hmEncLog}"
    #textMap_Y_PSNR=$(tail -26 ${hmLog} | awk '{print $5}')
    #textMap_U_PSNR=$(tail -26 ${hmLog} | awk '{print $6}')
    #textMap_V_PSNR=$(tail -26 ${hmLog} | awk '{print $7}')
    textureEncTime=$(cat ${hmEncLog} | grep "Total Time:" | awk '{print $3}')
    #echo "TEXTURE ENC TIME = ${textureEncTime} seconds"
  else
    echo "${hmEncLog} does not exists"
  fi
  hmDecLog=$3
  textureDecTime=0
  if [ -f ${hmDecLog} ]; then
    #echo "Reading ${hmDecLog}"
    textureDecTime=$(cat ${hmDecLog} | grep "Total Time:" | awk '{print $3}')
    #echo "TEXTURE DEC TIME = ${textureDecTime} seconds"
  else
    echo "${hmDecLog} does not exists"
  fi
}

collect_mmerror_results_all ()
{
  log=$1
  case $2 in
    ediv | grid | prnd)
      D1_PSNR=0
      D2_PSNR=0
      Y_PSNR=0
      U_PSNR=0
      V_PSNR=0
      PCQM_PSNR=0
      if  [ -f ${log} ]; then
        #echo -ne "Reading ${log}\033[0K\r"
    	  D1_PSNR=$(cat ${log} | grep -m${numFrames} "mseF,PSNR (p2point):" | awk 'NR == 1 { sum=0 } { sum+=$3; } END {printf "%f\n", sum/NR}')
    	  D2_PSNR=$(cat ${log} | grep -m${numFrames} "mseF,PSNR (p2plane):" | awk 'NR == 1 { sum=0 } { sum+=$3; } END {printf "%f\n", sum/NR}')
    	  Y_PSNR=$(cat ${log} | grep -m${numFrames} "c\[0\],PSNRF" | awk 'NR == 1 { sum=0 } { sum+=$3; } END {printf "%f\n", sum/NR}')
    	  U_PSNR=$(cat ${log} | grep -m${numFrames} "c\[1\],PSNRF" | awk 'NR == 1 { sum=0 } { sum+=$3; } END {printf "%f\n", sum/NR}')
    	  V_PSNR=$(cat ${log} | grep -m${numFrames} "c\[2\],PSNRF" | awk 'NR == 1 { sum=0 } { sum+=$3; } END {printf "%f\n", sum/NR}')
    	  PCQM_PSNR=$(cat ${log} | grep -m${numFrames} "PCQM-PSNR" | cut -d"=" -f2 | awk 'NR == 1 { sum=0 } { sum+=$1; } END {printf "%f\n", sum/NR}')
        #echo ${log}: ${D1_PSNR},${D2_PSNR},${Y_PSNR},${U_PSNR},${V_PSNR},${PCQM_PSNR}
      else
        echo "${log} does not exists"
      fi
      ;;
    ibsm)
      R_PSNR=0
      G_PSNR=0
      B_PSNR=0
      RGB_PSNR=0
      GEO_PSNR=0
      if  [ -f ${log} ]; then
        #echo -ne "Reading ${log}\033[0K\r"
    	  R_PSNR=$(cat ${log} | grep -m${numFrames} "R   PSNR =" | awk 'NR == 1 { sum=0 } { sum+=$4; } END {printf "%f\n", sum/NR}')
    	  G_PSNR=$(cat ${log} | grep -m${numFrames} "G   PSNR =" | awk 'NR == 1 { sum=0 } { sum+=$4; } END {printf "%f\n", sum/NR}')
    	  B_PSNR=$(cat ${log} | grep -m${numFrames} "B   PSNR =" | awk 'NR == 1 { sum=0 } { sum+=$4; } END {printf "%f\n", sum/NR}')
    	  RGB_PSNR=$(cat ${log} | grep -m${numFrames} "RGB PSNR =" | awk 'NR == 1 { sum=0 } { sum+=$4; } END {printf "%f\n", sum/NR}')
    	  GEO_PSNR=$(cat ${log} | grep -m${numFrames} "GEO PSNR =" | awk 'NR == 1 { sum=0 } { sum+=$4; } END {printf "%f\n", sum/NR}')
        #echo ${log}: ${R_PSNR},${G_PSNR},${B_PSNR},${RGB_PSNR},${GEO_PSNR}
      else
        echo "${log} does not exists"
      fi
      ;;
  esac
}

collect_results_lossy()
{
  #creating directory structure for results
  [ ! -d ${RESULTSdir}/results_${numFrames}frames ] && mkdir  ${RESULTSdir}/results_${numFrames}frames

  for codingMode in {AI,RA}; do
    #creating logfile
    statLog=${RESULTSdir}/results_${numFrames}frames/statistics_${seq}_${codingMode}.csv
    echo "R,QP,QT,Mesh Resolution,QM,Texture Resolution,ConnectivityVertexRate(Mbps),ConnectivityTextureRate(Mbps),ConnectivityTotalRate(Mbps),VertexPositionRate(Mbps),TextureCoordinateRate(Mbps),TotalGeometryRate(Mbps),TextureMapRate(Mbps),TotalRate(Mbps),D1(dB),D2(dB),Y-PSNR(dB),U-PSNR(dB),V-PSNR(dB),PCQM(dB),R-PSNR(dB),G-PSNR(dB),B-PSNR(dB),RGB-PSNR(dB),GEO-PSNR(dB),TextureEncTime(s),TextureDecTime(s),GeoEncTime(s),GeoDecTime(s),TotalGeometry(bytes),TextureMap(bytes)" > ${statLog}
  
    #now reading parameters from configuration file and collecting simulation results from HM, DRACO, and mmetric
    numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`
    for ((rate=1;rate<=$((numLines-1));rate++)); do
      #reading the coding parameters
      qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
      qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
      meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
      qm=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 5 -d","`
      textureResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 6 -d","`
      textMapSize=$((textMapDim / textureResolution))
	    if [[ ${qp} == -1 ]]; then
        echo ${rate},${qp},${qt},${qm},0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
  	    echo ${rate},${qp},${qt},${qm},0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 >> ${statLog}
        continue
      fi
     
      echo "Collecting results for rate R${rate} for $1 metric with ${numFrames} frames,  for DRACO (QP=${qp}, QT=${qt}, and Mesh Resolution=${meshResolution}), and for HM ${codingMode} Compression (QP=${qm} and Texture Resolution=1/${textureResolution})"
      #reading the mmerror file
      for sampleMethod in {ibsm,}; do
        mmetricLogFile=${MMETRICdir}/${seq}/mmetric_${sampleMethod}_${numFrames}_frames_QP${qp}_QT${qt}_${codingMode}_${meshResolution}_QM${qm}_${textureResolution}.log
        #echo collect_mmerror_results_all ${mmetricLogFile} ${sampleMethod}
        collect_mmerror_results_all ${mmetricLogFile} ${sampleMethod}
      done    
      #reading the encoding results
      summaryLog=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/summary_statistics_${numFrames}_frames.log
      binPattern=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr%04d.drc
      encLogPattern=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr%04d.log
      decLogPattern=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr%04d_DEC.log
      #echo collect_encoder_results_all ${binPattern} ${encLogPattern} ${decLogPattern} ${summaryLog}
      collect_encoder_results_all ${binPattern} ${encLogPattern} ${decLogPattern} ${summaryLog}
      #reading the HM file
      hmEncLog=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}.log
      hmDecLog=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}_DEC.log
      hmFile=${HMdir}/${seq}/${codingMode}/${textureResolution}/${seq}_QP${qm}.bin
      #echo collect_HM_results ${hmFile} ${hmEncLog} ${hmDecLog} 
      collect_HM_results ${hmFile} ${hmEncLog} ${hmDecLog} 
      rateTotal=`echo "scale=10; ${totalGeometryBitrate} + ${textureBitrate}" | bc -l`
  	  echo -ne "\033[0K\r"
      echo R${rate},${qp},${qt},${meshResolution},${qm},${textureResolution},${connBitrate},${connTexBitrate},${connTotal},${vertexBitrate},${textCoordBitrate},${totalGeometryBitrate},${textureBitrate},${rateTotal},${D1_PSNR},${D2_PSNR},${Y_PSNR},${U_PSNR},${V_PSNR},${PCQM_PSNR},${R_PSNR},${G_PSNR},${B_PSNR},${RGB_PSNR},${GEO_PSNR},${geoEncTime},${geoDecTime},${totalGeometryBytes},${textureBytes}
  	  echo R${rate},${qp},${qt},${meshResolution},${qm},${textureResolution},${connBitrate},${connTexBitrate},${connTotal},${vertexBitrate},${textCoordBitrate},${totalGeometryBitrate},${textureBitrate},${rateTotal},${D1_PSNR},${D2_PSNR},${Y_PSNR},${U_PSNR},${V_PSNR},${PCQM_PSNR},${R_PSNR},${G_PSNR},${B_PSNR},${RGB_PSNR},${GEO_PSNR},${geoEncTime},${geoDecTime},${totalGeometryBytes},${textureBytes} >> ${statLog}
    done
  done
}

plot_grid()
{
 #R,QP,QT,MeshResolution,QM,TextureResolution,ConnectivityVertexRate(Mbps),ConnectivityTextureRate(Mbps),ConnectivityTotalRate(Mbps),VertexPositionRate(Mbps),TextureCoordinateRate(Mbps),TotalGeometryRate(Mbps),TextureMapRate(Mbps),TotalRate(Mbps),D1(dB),D2(dB),Y-PSNR(dB),U-PSNR(dB),V-PSNR(dB),PCQM(dB),R-PSNR(dB),G-PSNR(dB),B-PSNR(dB),RGB-PSNR(dB),GEO-PSNR(dB)
 echo "" > $1
 echo "# Output W3C Scalable Vector Graphics " >> $1
 echo "set terminal pdf font \"Verdana,6\" " >> $1
 echo "# Read comma-delimited data from file " >> $1
 echo "set datafile separator comma " >> $1
 echo "#multiple plots arrangement " >> $1
 echo "set multiplot layout 2,3 rowsfirst " >> $1
 echo "#legend off " >> $1
 echo "set key autotitle columnhead" >> $1
 echo "set key off " >> $1
 echo "# Use a line graph " >> $1
 echo "set style data line " >> $1
 echo "set pointsize 0.2 " >> $1
 echo "#set grid " >> $1
 echo "set style line 102 lc rgb \"gray\" lw 0.25 dt \".\" " >> $1
 echo "set grid ytics mytics back ls 102 " >> $1
 echo " " >> $1
 echo "# Set graph title " >> $1
 echo "set title '' " >> $1
 echo "# Set axis label " >> $1
 echo "set ylabel 'D1 (dB)' " >> $1
 echo "set xlabel 'Geometry Bitrate (Mbps)' " >> $1
 echo "plot ARG1 using 12:15 with linespoint lw 2 pt 11 " >> $1
 echo " " >> $1
 echo "set ylabel 'D2 (dB)' " >> $1
 echo "plot ARG1 using 12:16 with linespoint lw 2 pt 11 " >> $1
 echo " " >> $1
 echo "set ylabel 'PCQM (dB)' " >> $1
 echo "set xlabel 'Total Bitrate (Mbps)' " >> $1
 echo "plot ARG1 using 14:20 with linespoint lw 2 pt 11 " >> $1
 echo " " >> $1
 echo "set ylabel 'Y-PSNR (dB)' " >> $1
 echo "plot ARG1 using 14:17 with linespoint lw 2 pt 11 " >> $1
 echo " " >> $1
 echo "set ylabel 'U-PSNR (dB)' " >> $1
 echo "plot ARG1 using 14:18 with linespoint lw 2 pt 11 " >> $1
 echo " " >> $1
 echo "set ylabel 'V-PSNR (dB)' " >> $1
 echo "plot ARG1 using 14:19 with linespoint lw 2 pt 11  " >> $1
}

plot_ibsm()
{
  #R,QP,QT,MeshResolution,QM,TextureResolution,ConnectivityVertexRate(Mbps),ConnectivityTextureRate(Mbps),ConnectivityTotalRate(Mbps),VertexPositionRate(Mbps),TextureCoordinateRate(Mbps),TotalGeometryRate(Mbps),TextureMapRate(Mbps),TotalRate(Mbps),D1(dB),D2(dB),Y-PSNR(dB),U-PSNR(dB),V-PSNR(dB),PCQM(dB),R-PSNR(dB),G-PSNR(dB),B-PSNR(dB),RGB-PSNR(dB),GEO-PSNR(dB)
  echo "" > $1
  echo "# Output W3C Scalable Vector Graphics " >> $1
  echo "set terminal pdf font \"Verdana,6\" " >> $1
  echo "# Read comma-delimited data from file " >> $1
  echo "set datafile separator comma " >> $1
  echo "#multiple plots arrangement " >> $1
  echo "set multiplot layout 2,3 rowsfirst " >> $1
  echo "#legend off " >> $1
  echo "set key autotitle columnhead" >> $1
  echo "# Use a line graph " >> $1
  echo "set style data line " >> $1
  echo "set pointsize 0.2 " >> $1
  echo "#set grid " >> $1
  echo "set style line 102 lc rgb \"gray\" lw 0.25 dt \".\" " >> $1
  echo "set grid ytics mytics back ls 102 " >> $1
  echo " " >> $1
  echo "# Set graph title " >> $1
  echo "set title '' " >> $1
  echo "# Set axis label " >> $1
  echo "set ylabel 'R (dB)' " >> $1
  echo "set xlabel 'Total Bitrate (Mbps)' " >> $1
  echo "plot ARG1 using 14:21 with linespoint lw 2 pt 11 notitle" >> $1
  echo " " >> $1
  echo "set ylabel 'G (dB)' " >> $1
  echo "plot ARG1 using 14:22 with linespoint lw 2 pt 11 notitle" >> $1
  echo " " >> $1
  echo "set ylabel 'B (dB)' " >> $1
  echo "plot ARG1 using 14:23 with linespoint lw 2 pt 11 notitle " >> $1
  echo " " >> $1
  echo "set ylabel 'RGB-PSNR (dB)' " >> $1
  echo "plot ARG1 using 14:24 with linespoint lw 2 pt 11 notitle" >> $1
  echo " " >> $1
  echo "set ylabel 'GEO-PSNR (dB)' " >> $1
  echo "set xlabel 'Geometry Bitrate (Mbps)' " >> $1
  echo "plot ARG1 using 12:25 with linespoint lw 2 pt 11 notitle" >> $1
  echo " " >> $1
  echo "set ylabel '(GEO-PSNR * 0.5 + RGB-PSNR * 0.5 dB)' " >> $1
  echo "set xlabel 'Total Bitrate (Mbps)' " >> $1
  echo "plot ARG1 using 14:(\$24 * 0.5 + \$25 * 0.5) with linespoint lw 2 pt 11 notitle" >> $1
}

plot_bitrate()
{
  #R,QP,QT,MeshResolution,QM,TextureResolution,ConnectivityVertexRate(Mbps),ConnectivityTextureRate(Mbps),ConnectivityTotalRate(Mbps),VertexPositionRate(Mbps),TextureCoordinateRate(Mbps),TotalGeometryRate(Mbps),TextureMapRate(Mbps),TotalRate(Mbps),D1(dB),D2(dB),Y-PSNR(dB),U-PSNR(dB),V-PSNR(dB),PCQM(dB),R-PSNR(dB),G-PSNR(dB),B-PSNR(dB),RGB-PSNR(dB),GEO-PSNR(dB)
  echo "" > $1
  echo "# Output W3C Scalable Vector Graphics" >> $1
  echo "set terminal pdf font \"Verdana,6\"" >> $1
  echo "# Read comma-delimited data from file" >> $1
  echo "set datafile separator comma" >> $1
  echo "# Use stacked histogram" >> $1
  echo "set style data histograms" >> $1
  echo "set style histogram rowstacked" >> $1
  echo "set boxwidth 0.5 relative" >> $1
  echo "set style fill solid 1.0 border -1" >> $1
  echo "set key left top" >> $1
  echo "set style line 102 lc rgb \"gray\" lw 0.25 dt \".\"" >> $1
  echo "set grid ytics mytics back ls 102" >> $1
  echo "" >> $1
  echo "# Set graph title" >> $1
  echo "set title ''" >> $1
  echo "# Set axis label" >> $1
  echo "set xrange [0:6]" >> $1
  echo "set xtics (" " 0, \"R1\" 1, \"R2\" 2, \"R3\" 3, \"R4\" 4, \"R5\" 5) font \"1\"" >> $1
  echo "set ylabel 'Total Bitrate (Mbps)'" >> $1
  echo "plot ARG1 using 12 t \"Geometry Bitrate\", \"\"  using 0:(\$12/2):(sprintf(\"%3.2f\",\$12)) with labels font \",8\" notitle, '' using 13 t \"Texture Bitrate\", \"\"  using 0:(\$12+\$13/2):(sprintf(\"%3.2f\",\$13)) with labels font \",8\" notitle, \"\"  using 0:(\$12+\$13+2):(sprintf(\"%3.2fMbps\",\$14)) with labels font \",10\" notitle" >> $1
}

plot_enc_timing()
{
  echo "" > $1
  echo "# Output W3C Scalable Vector Graphics" >> $1
  echo "set terminal pdf font \"Verdana,6\"" >> $1
  echo "# Read comma-delimited data from file" >> $1
  echo "set datafile separator comma" >> $1
  echo "# Use stacked histogram" >> $1
  echo "set style data histograms" >> $1
  echo "set style histogram rowstacked" >> $1
  echo "set boxwidth 0.5 relative" >> $1
  echo "set style fill solid 1.0 border -1" >> $1
  echo "#legend off " >> $1
  echo "set key autotitle columnhead" >> $1
  #echo "set key off " >> $1
  echo "#set grid" >> $1
  echo "set style line 102 lc rgb \"gray\" lw 0.25 dt \".\"" >> $1
  echo "set grid ytics mytics back ls 102" >> $1
  echo "" >> $1
  echo "# Set graph title" >> $1
  echo "set title ''" >> $1
  echo "# Set axis label" >> $1
  echo "set xtics nomirror rotate by 45 right" >> $1
  echo "set ylabel 'Total Bitrate (Mbps)'" >> $1
  echo "plot ARG1 using (\$26):xtic(1) t \"Geometry Encoder Time\", '' using 24 t \"Texture Encoder Time\", \"\"  using 0:(\$26/2):(sprintf(\"%3.2f seconds\",\$26)) with labels font \",5\" notitle, \"\"  using 0:(\$26+\$24/2):(sprintf(\"%3.2f seconds\",\$24)) with labels font \",5\" notitle, \"\"  using 0:(\$24+\$26+5000):(sprintf(\"%3.2f seconds\",\$24+\$26)) with labels font \",7\" notitle" >> $1
}

plot_dec_timing()
{
  echo "" > $1
  echo "# Output W3C Scalable Vector Graphics" >> $1
  echo "set terminal pdf font \"Verdana,6\"" >> $1
  echo "# Read comma-delimited data from file" >> $1
  echo "set datafile separator comma" >> $1
  echo "# Use stacked histogram" >> $1
  echo "set style data histograms" >> $1
  echo "set style histogram rowstacked" >> $1
  echo "set boxwidth 0.5 relative" >> $1
  echo "set style fill solid 1.0 border -1" >> $1
  echo "#legend off " >> $1
  echo "set key autotitle columnhead" >> $1
  #echo "set key off " >> $1
  echo "#set grid" >> $1
  echo "set style line 102 lc rgb \"gray\" lw 0.25 dt \".\"" >> $1
  echo "set grid ytics mytics back ls 102" >> $1
  echo "" >> $1
  echo "# Set graph title" >> $1
  echo "set title ''" >> $1
  echo "# Set axis label" >> $1
  echo "set xtics nomirror rotate by 45 right" >> $1
  echo "set ylabel 'Total Bitrate (Mbps)'" >> $1
  echo "plot ARG1 using (\$27):xtic(1) t \"Geometry Decoder Time\", '' using 25 t \"Texture Decoder Time\", \"\"  using 0:(\$27/2):(sprintf(\"%3.2f seconds\",\$27)) with labels font \",5\" notitle, \"\"  using 0:(\$27+\$25/2):(sprintf(\"%3.2f seconds\",\$25)) with labels font \",5\" notitle, \"\"  using 0:(\$27+\$25+5):(sprintf(\"%3.2f seconds\",\$25+\$27)) with labels font \",7\" notitle" >> $1
}

create_plots()
{
  #creating directory structure for plot results
  [ ! -d ${RESULTSdir}/results_${numFrames}frames ] && mkdir  ${RESULTSdir}/results_${numFrames}frames
  [ ! -d ${RESULTSdir}/results_${numFrames}frames/plot ] && mkdir  ${RESULTSdir}/results_${numFrames}frames/plot

  for codingMode in {AI,RA}; do
    statLog=${RESULTSdir}/results_${numFrames}frames/statistics_${seq}_${codingMode}.csv
    #plotGrid=${RESULTSdir}/results_${numFrames}frames/plot/${seq}_${codingMode}_grid.pdf
    #if [ ! -f ${plotGrid} ]; then
    #  plot_grid tmp_grid.sh
    #  echo gnuplot -c tmp_grid.sh ${statLog} 
    #  gnuplot -c tmp_grid.sh ${statLog} > ${plotGrid}
	  #  rm tmp_grid.sh
    #fi
    plotIbsm=${RESULTSdir}/results_${numFrames}frames/plot/${seq}_${codingMode}_ibsm.pdf
    if [ ! -f ${plotIbsm} ]; then
      plot_ibsm tmp_ibsm.sh
      echo gnuplot -c tmp_ibsm.sh ${statLog} 
      gnuplot -c tmp_ibsm.sh ${statLog} > ${plotIbsm}
	    rm tmp_ibsm.sh
    fi
    plotBitrate=${RESULTSdir}/results_${numFrames}frames/plot/${seq}_${codingMode}_bitrate.pdf
    if [ ! -f ${plotBitrate} ]; then
      plot_bitrate tmp_bitrate.sh
      echo gnuplot -c tmp_bitrate.sh ${statLog} 
      gnuplot -c tmp_bitrate.sh ${statLog} > ${plotBitrate}
	    rm tmp_bitrate.sh
    fi
    plotTiming=${RESULTSdir}/results_${numFrames}frames/plot/${seq}_${codingMode}_enc_timing.pdf
    if [ ! -f ${plotTiming} ]; then
      plot_enc_timing tmp_timing.sh
      echo gnuplot -c tmp_timing.sh ${statLog} 
      gnuplot -c tmp_timing.sh ${statLog} > ${plotTiming}
  	  rm tmp_timing.sh
    fi
    plotTiming=${RESULTSdir}/results_${numFrames}frames/plot/${seq}_${codingMode}_dec_timing.pdf
    if [ ! -f ${plotTiming} ]; then
      plot_dec_timing tmp_timing.sh
      echo gnuplot -c tmp_timing.sh ${statLog} 
      gnuplot -c tmp_timing.sh ${statLog} > ${plotTiming}
	    rm tmp_timing.sh
    fi
  done
}

render_mmetric_empty ()
{
  renderVideo=$1
  if [ -f ${renderVideo}.mp4 ]; then
    echo "${renderVideo}.mp4 exists"
  else
    echo "v 0 0 0 0 0 0" > tmp.obj
    echo "v 0 1 0 0 0 0" >> tmp.obj
    echo "v 0 0 1 0 0 0" >> tmp.obj
    echo "f 0 1 2" >> tmp.obj
    #rendering parameters
    width=1280
    height=720
    viewDir="0.0 0.0 1.0"
    if [ ${seq} == basketball_player ]; then
      viewDir="0.0 0.0 -1.0"
    fi
    if [ ${seq} == dancer ]; then
      viewDir="0.0 0.0 -1.0"
    fi
    endFr=$((numFrames + startFr - 1))
	  
    echo ${mmetricSW} \
      sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
      render --inputModel tmp.obj --outputImage ${renderVideo}_fr%04d.png --width=${width} --height=${height} --bboxMin "${globalMinPos}" --bboxMax "${globalMaxPos}" --viewDir "${viewDir}" --hideProgress 
    ${mmetricSW} \
      sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
      render --inputModel tmp.obj --outputImage ${renderVideo}_fr%04d.png --width=${width} --height=${height} --bboxMin "${globalMinPos}" --bboxMax "${globalMaxPos}" --viewDir "${viewDir}" --hideProgress 

    ${ffmpegSW} -y -start_number ${startFr} -i ${renderVideo}_fr%04d.png -c:v libx265 -preset ultrafast -x265-params lossless=1 -pix_fmt yuv420p -frames:v ${numFrames} ${renderVideo}.mp4
    
    rm ${renderVideo}_fr*.png
    rm tmp.obj
  fi
}

render_mmetric ()
{
  renderVideo=$1
  refMesh=$2
  refTexture=$3
  if [ -f ${renderVideo}.mp4 ]; then
    echo "${renderVideo}.mp4 exists"
  else
    #rendering parameters
    width=1280
    height=720
    viewDir="0.0 0.0 1.0"
    if [ ${seq} == baskteball_player ]; then
      viewDir="0.0 0.0 -1.0"
    fi
    if [ ${seq} == dancer ]; then
      viewDir="0.0 0.0 -1.0"
    fi
    endFr=$((numFrames + startFr - 1))
	  
    echo ${mmetricSW} \
      sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
      render --inputModel ${refMesh} --inputMap ${refTexture} --outputImage ${renderVideo}_fr%04d.png --width=${width} --height=${height} --bboxMin "${globalMinPos}" --bboxMax "${globalMaxPos}" --viewDir "${viewDir}" --hideProgress 
    ${mmetricSW} \
      sequence --firstFrame ${startFr} --lastFrame ${endFr} END \
      render --inputModel ${refMesh} --inputMap ${refTexture} --outputImage ${renderVideo}_fr%04d.png --width=${width} --height=${height} --bboxMin "${globalMinPos}" --bboxMax "${globalMaxPos}" --viewDir "${viewDir}" --hideProgress 

    ${ffmpegSW} -y -start_number ${startFr} -i ${renderVideo}_fr%04d.png -c:v libx265 -preset ultrafast -x265-params lossless=1 -pix_fmt yuv420p -frames:v ${numFrames} ${renderVideo}.mp4
    
    rm ${renderVideo}_fr*.png
  fi
}

render_sequence()
{
  #creating directory structure for results
  [ ! -d ${RESULTSdir}/results_${numFrames}frames ] && mkdir  ${RESULTSdir}/results_${numFrames}frames
  [ ! -d ${RESULTSdir}/results_${numFrames}frames/video ] && mkdir  ${RESULTSdir}/results_${numFrames}frames/video
  #now reading parameters from configuration file and collecting simulation results from HM, DRACO, and mmetric
  numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`

  #generating video for each rate point
  for codingMode in {AI,RA}; do
    for ((rate=1;rate<=$((numLines-1));rate++)); do
      #reading the coding parameters
      qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
      qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
      meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
      qm=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 5 -d","`
      textureResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 6 -d","`
      textMapSize=$((textMapDim / textureResolution))
	    if [[ ${qp} == -1 ]]; then
        echo "Creating empty video for rate R${rate}"
        renderVideo=${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R${rate}
        render_mmetric_empty ${renderVideo}
        continue
      fi
      echo "Creating video for rate R${rate}"
      renderVideo=${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R${rate}
  	  compMesh=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr%04d.obj
      compTexture=${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate}/${seq}_fr%04d.png
      render_mmetric ${renderVideo} ${compMesh} ${compTexture}
    done;
  
    #composing a video for comparison
    comparisonName=${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_comparison
    if [ ! -f ${comparisonName}.mp4 ]; then
      ${ffmpegSW} -i ${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R1.mp4 \
                  -i ${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R2.mp4 \
                  -i ${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R3.mp4 \
                  -i ${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R4.mp4 \
                  -i ${RESULTSdir}/results_${numFrames}frames/video/${seq}_${codingMode}_R5.mp4 \
                  -filter_complex "[0:v]crop=320:720:480:0[v0];[1:v]crop=320:720:480:0[v1];[2:v]crop=320:720:480:0[v2];[3:v]crop=320:720:480:0[v3];[4:v]crop=320:720:480:0[v4];\
                  [v0][v1]hstack[v01];[v01][v2]hstack[v012];[v012][v3]hstack[v0123];[v0123][v4]hstack" ${comparisonName}_NO_LEGEND.mp4
  	  ${ffmpegSW} -i ${comparisonName}_NO_LEGEND.mp4 \
                  -vf "drawtext=text='R1': fontcolor=white: fontsize=30: box=1: boxcolor=black@0.5: boxborderw=4: x=(320*1/2-text_w/2): y=(text_h), \
                      drawtext=text='R2': fontcolor=white: fontsize=30: box=1: boxcolor=black@0.5: boxborderw=4: x=(320*3/2-text_w/2): y=(text_h),\
                      drawtext=text='R3': fontcolor=white: fontsize=30: box=1: boxcolor=black@0.5: boxborderw=4: x=(320*5/2-text_w/2): y=(text_h),\
                      drawtext=text='R4': fontcolor=white: fontsize=30: box=1: boxcolor=black@0.5: boxborderw=4: x=(320*7/2-text_w/2): y=(text_h),\
                      drawtext=text='R5': fontcolor=white: fontsize=30: box=1: boxcolor=black@0.5: boxborderw=4: x=(320*9/2-text_w/2): y=(text_h)" ${comparisonName}.mp4
	    rm ${comparisonName}_NO_LEGEND.mp4
    else
      echo ${comparisonName}.mp4 exists
    fi
  done
}

zip_sequence()
{
  #creating directories if necessary
  [ ! -d ${RESULTSdir}/decoded_meshes ] && mkdir  ${RESULTSdir}/decoded_meshes
  [ ! -d ${RESULTSdir}/decoded_meshes/CAT1 ] && mkdir  ${RESULTSdir}/decoded_meshes/CAT2
  [ ! -d ${RESULTSdir}/decoded_meshes/CAT1/lossy_all_intra ]     && mkdir  ${RESULTSdir}/decoded_meshes/CAT1/lossy_all_intra
  [ ! -d ${RESULTSdir}/decoded_meshes/CAT1/lossy_random_access ] && mkdir  ${RESULTSdir}/decoded_meshes/CAT1/lossy_random_access
  
  #now reading parameters from configuration file and collecting simulation results from HM, DRACO, and mmetric
  numLines=`wc -l ${codingParameters} | awk '{ print $1 }'`
  #generating video for each rate point
  codingCode=0
  for codingMode in {AI,RA}; do
    codingCode=$((codingCode+1)) 
    if [ ${codingMode} == AI ]; then
      resultDir=${RESULTSdir}/decoded_meshes/CAT1/lossy_all_intra
    else
      resultDir=${RESULTSdir}/decoded_meshes/CAT1/lossy_random_access
    fi
    for ((rate=1;rate<=$((numLines-1));rate++)); do
      #reading the coding parameters
      qp=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 2 -d","`
      qt=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 3 -d","`
      meshResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 4 -d","`
      qm=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 5 -d","`
      textureResolution=`head -$((rate+1)) ${codingParameters} | tail -1 | cut -f 6 -d","`
      textMapSize=$((textMapDim / textureResolution))
	    if [[ ${qp} == -1 ]]; then
        continue
      fi
      #copying the decoded meshes into the all intra directory only
      decodedZIPFile=${resultDir}/P00S${seqNum}C${codingCode}R${rate}.zip
      if [ -f ${decodedZIPFile} ]; then
        echo ${decodedZIPFile} exists, skipping zip generation
      else
        echo Creating ${decodedZIPFile} with ${numFrames} frames
        for (( i=0; i<${numFrames}; i++ )); do
          f=$((i + startFr))
          fr=$(printf "%04d" $f)
          echo Processing ${i}/${numFrames} frame
          dequantMesh=${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}/${seq}_fr${fr}.obj
          if  [ -f ${dequantMesh} ]; then
            echo "ZIP FOR DECODED DRACO FRAME ${fr}"
            zip -u -j -v ${decodedZIPFile} ${dequantMesh}
          else
            echo ${dequantMesh} does not exist
          fi
          dequantText=${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate}/${seq}_fr${fr}.png
          if  [ -f ${dequantText} ]; then
            echo "ZIP FOR DECODED DRACO FRAME ${fr}"
            zip -u -j -v ${decodedZIPFile} ${dequantText}
          else
            echo ${dequantText} does not exist
          fi
        done
      fi
      decodedMD5File=${resultDir}/P00S${seqNum}C${codingCode}R${rate}.md5
      if [ -f ${decodedMD5File} ]; then
        echo ${decodedMD5File} exists, skipping md5 generation
      else
        curDir=`pwd`
        for (( i=0; i<${numFrames}; i++ )); do
          f=$((i + startFr))
          fr=$(printf "%04d" $f)
          cd ${DRACOdir}/${seq}/${meshResolution}/qp${qp}_qt${qt}
          dequantMesh=${seq}_fr${fr}.obj
          if  [ -f ${dequantMesh} ]; then
            echo -ne "MD5SUM FOR MESH FRAME ${fr}\033[0K\r"
            md5sum ${dequantMesh} >> ${decodedMD5File}
          else
            echo "${dequantMesh} does not exist" >> ${decodedMD5File}
          fi
          cd ${HMdir}/${seq}/${codingMode}/${textureResolution}/R${rate}
          dequantText=${seq}_fr${fr}.png
          if  [ -f ${dequantText} ]; then
            echo -ne "MD5SUM FOR TEXTURE FRAME ${fr}\033[0K\r"
            md5sum ${dequantText} >> ${decodedMD5File}
          else
            echo "${dequantText} does not exist" >> ${decodedMD5File}
          fi
        done
      fi
    done
  done
  #clear the last line for readability
  echo -ne "\033[0K\r"
}

#Adjust these directories for your environment
ContentDir=/mnt/88f5590e-d5bf-4c5d-afed-893a33c47b26/MPEG/DATASET/PCC/CATM
if [ ! -d ${ContentDir} ]; then
  echo "Directory (${ContentDir}) does not exist, exiting script"
  exit 1
fi
SWDir=/mnt/d21f681c-6b46-42e8-9028-c4d87f97f5d9/MPEG/SW
if [ ! -d ${SWDir} ]; then
  echo "Directory (${SWDir}) does not exist, exiting script"
  exit 1
fi
WorkDir=/mnt/88f5590e-d5bf-4c5d-afed-893a33c47b26/MPEG/MPEG134/anchor_generation
if [ ! -d ${WorkDir} ]; then
  echo "Directory (${WorkDir}) does not exist, exiting script"
  exit 1
fi
ScriptsDir=/mnt/d21f681c-6b46-42e8-9028-c4d87f97f5d9/MPEG/mpeg-vmesh-anchor/scripts
if [ ! -d ${ScriptsDir} ]; then
  echo "Directory (${ScriptsDir}) does not exist, exiting script"
  exit 1
fi

echo "Initializing script"
initialize_script
#for seq in {longdress,soldier,basketball_player,dancer,mitch,thomas,football,levi}; do 
for seq in {longdress,}; do 
  echo "Load ${seq} parameters"
  echo Load ${seq} parameters
  load_sequence_parameters
  echo Decode HM
  run_HM_lossy
  echo Decode DRACO lossy
  run_DRACO_lossy
  echo Run mmerror
  for sampleMethod in {ibsm,ibsm_90degree,ibsm_45degree}; do
    run_mmetric_lossy ${sampleMethod}
  done
  echo Collect results
  collect_results_lossy ${sampleMethod}
  echo Create plots
  create_plots
  echo Create Rendering
  render_sequence   
  echo Create Zip
  zip_sequence  
done