//----------------------------------------------------------------------------------------------------
// HDL Test Stimuli
//----------------------------------------------------------------------------------------------------

`ifndef TEST_STIMULI
`define TEST_STIMULI

`timescale 1ns/1ps


module test_stimuli
   #(parameter LATTICE_FAMILY                   = "common",
     parameter S_AHB_ADR_WIDTH                  = 32,
     parameter S_AHB_DAT_WIDTH                  = 32,
     parameter S_AHB_DAT_ENDIANESS              = "little-endian",
     parameter C_PORT_ENABLE                    = 0,
     parameter C_APB_ADR_WIDTH                  = 11,
     parameter C_APB_DAT_WIDTH                  = 32,
     parameter PAGE_PRG_BUF_ENA                 = 0,
     parameter PAGE_READ_BUF_ENA                = 0,
     parameter PAGE_PRG_BUFFER_EBR              = 0,
     parameter PAGE_PRG_BUFFER_DISTRIBUTED_RAM  = 1,
     parameter PAGE_READ_BUFFER_EBR             = 0,
     parameter PAGE_READ_BUFFER_DISTRIBUTED_RAM = 1,
     parameter PAGE_BUFFER_INTF                 = "APB",
     parameter PAGE_PROGRAM_SIZE                = 256,
     parameter PAGE_READ_SIZE                   = 256,
     parameter CLOCK_SEL                        = 1,
     parameter CPOL                             = 1,
     parameter CPHASE                           = 1,
     parameter LSB_FIRST                        = 0,
     parameter SPI_READ                         = 8'h03,
     parameter SPI_FAST_READ                    = 8'h0b,
     parameter SPI_BYTE_PRG                     = 8'h02,
     parameter SPI_PAGE_PRG                     = 8'h02,
     parameter SPI_BLK1_ERS                     = 8'h20,
     parameter SPI_BLK2_ERS                     = 8'h52,
     parameter SPI_BLK3_ERS                     = 8'hd8,
     parameter SPI_CHIP_ERS                     = 8'h60,
     parameter SPI_WRT_ENB                      = 8'h06,
     parameter SPI_WRT_DISB                     = 8'h04,
     parameter SPI_READ_STAT                    = 8'h05,
     parameter SPI_WRT_STAT                     = 8'h01,
     parameter SPI_PWD_DOWN                     = 8'hb9,
     parameter SPI_PWD_UP                       = 8'hab,
     parameter SPI_DEV_ID                       = 8'h9f,
	 parameter PCLK_PERIOD                      = 0)
   (
    // AHB-Lite signals
    input  ahbl_hclk_i,
    output [S_AHB_ADR_WIDTH-1:0] ahbl_haddr_o,
    output [S_AHB_DAT_WIDTH-1:0] ahbl_hwdata_o,
    output [2:0] ahbl_hsize_o,
    output ahbl_hwrite_o,
    output ahbl_hsel_o,
    output [2:0] ahbl_hburst_o,
    output [1:0] ahbl_htrans_o,
    output ahbl_hlock_o,
    input  [S_AHB_DAT_WIDTH-1:0] ahbl_hrdata_i,
    input  ahbl_hresp_i,
    input  ahbl_hready_i,
    
    // APB signals
    input  apb_pclk_i,
    output [C_APB_ADR_WIDTH-1:0] apb_paddr_o,
    output [C_APB_DAT_WIDTH-1:0] apb_pwdata_o,
    output apb_pwrite_o,
    output apb_psel_o,
    input  apb_pready_i,
    output apb_penable_o,
    input  [C_APB_DAT_WIDTH-1:0] apb_prdata_i,
    input  apb_pslverr_i,

    // SPI flash signals
    input  mosi_i,     // Serial data from FPGA to SPI flash
    output miso_o,     // Serial data from SPI flash to FPGA
    input  ss_n_i,     // SPI flash chip select
    input  sclk_i,     // Serial clock to SPI flash
    input  wpj_i,      // Write protect
    
    // system signals
    //input clk_i,
    input spi_clk,
    input rst_i
    );

  //State Defines for Flash Model
  localparam                     IDLE         = 4'h0            ; // State for IDLE
  localparam                     CMD_OPCODE   = 4'h1            ; // State for CMD_OPCODE
  localparam                     READ_STAT    = 4'h2            ; // State for READ STATUS
  localparam                     PAGE_ADDR    = 4'h3            ; // State for PAGE ADDR
  localparam                     PAGE_WR_DATA = 4'h4            ; // State for PAGE WR DATA
  localparam                     PAGE_RD_DATA = 4'h5            ; // State for PAGE RD DATA
  localparam                     BLK_ERASE    = 4'h6            ; // State for BLOCK ERASE

  //Register Defines for Accessing SPI Controller's registers
  localparam                     PAGE_PRG            = 11'h000         ; // Page Program register
  localparam                     PAGE_RD             = 11'h004         ; // Page Read register
  localparam                     BLK_ERASE_TYPE1     = 11'h008         ; // Block Erase Type 1 register
  localparam                     CHIP_ERASE          = 11'h014         ; // Chip Erase register
  localparam                     WR_EN_REG           = 11'h018         ; // Write enable register
  localparam                     WR_DIS_REG          = 11'h01C         ; // Write Disable register
  localparam                     MANUFACTURER_ID     = 11'h030         ; // Write Disable register
  localparam                     POWER_DOWN          = 11'h028         ; // Power Down register
  localparam                     POWER_UP            = 11'h02C         ; // Power Up register
  localparam                     SPI_ADDR_BYTE0      = 11'h040         ; // SPI Flash Address register
  localparam                     SPI_ADDR_BYTE1      = 11'h041         ; // SPI Flash Address register
  localparam                     SPI_ADDR_BYTE2      = 11'h042         ; // SPI Flash Address register
  localparam                     SPI_ADDR_BYTE3      = 11'h043         ; // SPI Flash Address register
  localparam                     SPI_ADDR_MODE       = 11'h044         ; // SPI Flash Address Mode register
  localparam                     PAGE_PRG_LEN_0      = 11'h104         ; // Page Program Length register
  localparam                     PAGE_PRG_LEN_1      = 11'h105         ; // Page Program Length register
  localparam                     PAGE_RD_LEN_0       = 11'h108         ; // Page Read Length register
  localparam                     PAGE_RD_LEN_1       = 11'h109         ; // Page Read Length register
  localparam                     AHB_PAGE_BUF_EN     = 11'h118         ; // Enable Page Buffer operation in AHB-L
  localparam                     PAGE_PRG_BUF        = 11'h200         ; // Page Program Buffer register
  localparam                     PAGE_RD_BUF         = 11'h400         ; // Page Read Buffer register
  localparam                     USER_CMD_OPCODE     = 11'h080         ; // User-Defined Command Opcode
  localparam                     USER_CMD_ADDR       = 11'h084         ; // User-Defined Address
  localparam                     USER_CMD_WR_DATA0   = 11'h088         ; // User-Defined Write Data0
  localparam                     USER_CMD_WR_DATA1   = 11'h08C         ; // User-Defined Write Data1
  localparam                     USER_CMD_DATA_DUMMY = 11'h090         ; // User-Defined Data and Dummy Length
  localparam                     USER_CMD_CFG        = 11'h094         ; // User-Defined Configurations
  localparam                     USER_CMD_RD_DATA0   = 11'h098         ; // User-Defined Read Data0
  localparam                     USER_CMD_RD_DATA1   = 11'h09C         ; // User-Defined Read Data1
  localparam                     USER_CMD_START      = 11'h0A0         ; // User-Defined Start
  localparam                     TRANS_STAT_REG      = 11'h1C0         ; // SPI Transaction Status Register
  localparam                     RD_BUFF_STAT_REG    = 11'h1C4         ; // Page Read Buffer Status Register
  localparam                     RD_BUFF_STAT_REG_1  = 11'h1C5         ; // Page Read Buffer Status Register
  localparam                     RD_BUFF_STAT_REG_2  = 11'h1C6         ; // Page Read Buffer Status Register
  localparam                     RD_BUFF_ADDR_REG    = 11'h1C8         ; // Page Read Buffer Address Register
  localparam                     RD_BUFF_ADDR_REG_1  = 11'h1C9         ; // Page Read Buffer Address Register
  localparam                     RD_BUFF_ADDR_REG_2  = 11'h1CA         ; // Page Read Buffer Address Register
  localparam                     RD_BUFF_ADDR_REG_3  = 11'h1CB         ; // Page Read Buffer Address Register

  //Page Buffer
  localparam                     PAGE_PRG_LOOP     = (PAGE_PROGRAM_SIZE>PAGE_READ_SIZE) ? 1 : PAGE_READ_SIZE/PAGE_PROGRAM_SIZE;
  localparam                     PAGE_RD_LOOP      = (PAGE_PROGRAM_SIZE>PAGE_READ_SIZE) ? 2 : 1;
  localparam                     PAGE_COMPARE      = (PAGE_PROGRAM_SIZE>PAGE_READ_SIZE) ? PAGE_PROGRAM_SIZE : PAGE_READ_SIZE;
 
  //Delay for MOSI and MISO
  localparam                     MOSI_MISO_DLY     = (CLOCK_SEL == 0) ? (PCLK_PERIOD/4) : ((PCLK_PERIOD*CLOCK_SEL)/2); 
  //localparam                     MOSI_MISO_DLY     = (CLOCK_SEL == 0) ? (PCLK_PERIOD/2) : ((PCLK_PERIOD*CLOCK_SEL)); 
  
  //Registers required to implement the Flash Model
  reg [3:0]                      state_4b                     ; // State register 
  reg [7:0]                      cmd_opcode_8b                ; // Command opcode register 
  reg [31:0]                     cmd_addr_24b_32b             ; // Command Address
  reg [7:0]                      cmd_data_8b                  ; // Command Data
  reg [7:0]                      cmd_2d_8b  [3:0]             ; // Command FIFO Register
  reg [7:0]                      data_2d_8b [32767:0]         ; // RAM Memory of Flash max memory depth
  reg [12:0]                     data_idx_13b                 ; // Index pointer to point the RAM
  reg [1:0]                      cmd_idx_2b                   ; // Command Index register
  reg [2:0]                      bit_cnt_3b                   ; // Bit Count register
  reg [4:0]                      addr_cnt_5b                  ; // Address counter
  reg [31:0]                     ahb_rd_data_32b              ; // AHB Read Data register
  reg [3:0]                      smp_cnt_3b                   ; // Bit Count register for sampled data
  reg [7:0]                      smp_data_8b                  ; // Sampled data
  reg                            ahb_intf_en_b                ; // AHB Interface case is enabled

  reg [C_APB_DAT_WIDTH-1:0]      apb_rd_data                  ;  // APB Read Data register

  //Variables for Simulation handling
  integer                        writes_i                     ; // Number of write commands
  integer                        reads_i                      ; // Number of read commands
  integer                        timeout_i                    ; // Transaction timeout
  reg [16383:0]                  test_name_s                  ; // Use for storing testcase
  integer                        num_tran_ui                  ; // Use for storing number of transcations
  integer                        user_debug_i                 ; // Use for storing debug level passed by user
  reg                            finish_on_error_b            ; // Terminate on error message
  integer                        seed_i                       ; // Random seed to control the randomization
  integer                        errs_i                       ; // Total number of errors detected
  integer                        trans_i                      ; // Total number of transactions
  integer                        idx_i                        ; // Iterative variable
  reg [12:0]                     idx_13b                      ; // Iterative variable


  reg [S_AHB_ADR_WIDTH-1:0]      ahbl_haddr                   ; // AHBl address
  reg [S_AHB_DAT_WIDTH-1:0]      ahbl_hwdata                  ; // AHBl write data
  reg [2:0]                      ahbl_hsize                   ; // AHBl size
  reg                            ahbl_hwrite                  ; // AHBl write
  reg                            ahbl_hsel                    ; // AHBl select
  reg [2:0]                      ahbl_hburst                  ; // AHBl burst
  reg [1:0]                      ahbl_htrans                  ; // AHBl htrans
  reg                            ahbl_hlock                   ; // AHBl hlock

  reg [C_APB_ADR_WIDTH-1:0]      apb_paddr                    ; // APB address
  reg [C_APB_DAT_WIDTH-1:0]      apb_pwdata                   ; // APB write data 
  reg                            apb_pwrite                   ; // APB write
  reg                            apb_psel                     ; // APB select
  reg                            apb_penable                  ; // APB enable

  reg                            slv_out_b                    ; // Slave data output
  reg [2:0]                      size_3b                      ; // AHBl size

  reg [31:0]                     burst_prg_data_2d_32b [127:0];
  reg [31:0]                     burst_rd_data_2d_32b  [127:0];
  reg [7:0]                      burst_prg_data_2d_8b  [511:0];
  reg [7:0]                      burst_rd_data_2d_8b   [511:0];
  
  //For CPOL and CPHASE settings
  wire                           sclk_wr                      ;
  wire                           sclk_rd                      ;
  wire [31:0]                    cmd_addr_24b_32b_w           ;
  reg                            ss_n_dly                     ;
  wire                           miso_w                       ; // Delayed slave data output
  wire                           mosi_w                       ; // Delayed Slave data input

  //1###################################################################################################
  // Code to assign the input signals 
  //1###################################################################################################
  assign miso_o         = miso_w;
  assign ahbl_haddr_o   = ahbl_haddr;  
  assign ahbl_hwdata_o  = ahbl_hwdata; 
  assign ahbl_hsize_o   = ahbl_hsize;  
  assign ahbl_hwrite_o  = ahbl_hwrite; 
  assign ahbl_hsel_o    = ahbl_hsel;   
  assign ahbl_hburst_o  = ahbl_hburst; 
  assign ahbl_htrans_o  = ahbl_htrans; 
  assign ahbl_hlock_o   = ahbl_hlock;  
  assign apb_paddr_o    = apb_paddr;
  assign apb_pwdata_o   = apb_pwdata;
  assign apb_pwrite_o   = apb_pwrite;
  assign apb_psel_o     = apb_psel;
  assign apb_penable_o  = apb_penable;
  
  //1###################################################################################################
  // For CPOL and CPHASE signals 
  //1################################################################################################### 
  //assign sclk_wr = CPOL ? ~spi_clk : spi_clk;
  assign sclk_wr =(~CPOL & ~CPHASE) ? spi_clk : 
                  ( CPOL & ~CPHASE) ? ~spi_clk:
                  (~CPOL &  CPHASE) ? spi_clk :
				                      ~spi_clk;
  assign sclk_rd = sclk_wr;
  assign cmd_addr_24b_32b_w = (S_AHB_ADR_WIDTH == 32) ? (LSB_FIRST[0] ? {mosi_w,cmd_addr_24b_32b[31:1]} : {cmd_addr_24b_32b[30:0],mosi_w}) : 
		                                                (LSB_FIRST[0] ? {mosi_w,cmd_addr_24b_32b[23:1]} : {cmd_addr_24b_32b[22:0],mosi_w}) ;

  //Delay MOSI and MISO
  assign #MOSI_MISO_DLY miso_w = slv_out_b;
  assign #MOSI_MISO_DLY mosi_w = mosi_i;
  //1###################################################################################################
  // Initilize all the SOC bus signals 
  //1###################################################################################################
  initial begin
    ahbl_haddr                     <= 0;
    ahbl_hwdata                    <= 0;
    ahbl_hsize                     <= 0;
    ahbl_hwrite                    <= 0;
    ahbl_hsel                      <= 0;
    ahbl_hburst                    <= 0;
    ahbl_htrans                    <= 0;
    ahbl_hlock                     <= 0;

    apb_paddr                      <= 0;
    apb_pwdata                     <= 0;
    apb_pwrite                     <= 0;
    apb_psel                       <= 0;
    apb_penable                    <= 0;
    ahb_intf_en_b                  <= 0;
    ahb_rd_data_32b                <= 0;
    idx_i                          <= 0;
    size_3b                        <= 0;
    for (idx_i = 0; idx_i < 32768; idx_i = idx_i + 1) begin
      data_2d_8b[idx_i] = 8'hFF;
    end
  end


  //1###################################################################################################
  // Initilize all the variables 
  //1###################################################################################################
  initial begin
    timeout_i                      = 1000;
    finish_on_error_b              = 0;
    //2-----------------------------------------------------------------------------------------------
    // Plus args is defined 
    //2-----------------------------------------------------------------------------------------------
    if ($test$plusargs("sdvt_finish") == 1) begin
      finish_on_error_b              = 1;
    end
  end


  //1###################################################################################################
  // Initilize and run testcase 
  //1###################################################################################################
  initial begin
    errs_i                         = 0;
    trans_i                        = 0;
    wait_clocks(20);
    test_runner();
    wait_clocks(10);
    post_process();
    wait_clocks(10);
    $finish(1);
  end


  //1-------------------------------------------------------------------------------------------------
  // write_apb :This method is used for writing CSR register through APB Interface
  //1-------------------------------------------------------------------------------------------------
  task write_apb;
    input    [31:0]                     p_addr_32b                     ;
    input    [31:0]                     p_data_32b                     ;
    integer m_i_i;
    integer num_tran;
    begin
      num_tran = (C_APB_DAT_WIDTH == 32) ? 1 : 4;
      for (m_i_i = 0; m_i_i < num_tran; m_i_i = m_i_i + 1) begin
        @ (posedge apb_pclk_i);
        $write("INFO :: @%0dns %m() :: APB Writing address 0x%x data 0x%x\n",$time,p_addr_32b,p_data_32b);
        apb_psel                       <= 1; 
        apb_paddr                      <= p_addr_32b[10:0] + m_i_i;
        if (C_APB_DAT_WIDTH == 32)
          apb_pwdata                   <= p_data_32b; 
        else
          case (m_i_i)
            0 : apb_pwdata             <= p_data_32b[7:0]; 
            1 : apb_pwdata             <= p_data_32b[15:8]; 
            2 : apb_pwdata             <= p_data_32b[23:16]; 
            3 : apb_pwdata             <= p_data_32b[31:24]; 
          endcase
        apb_pwrite                     <= 1;
        @ (posedge apb_pclk_i);
        apb_penable                    <= 1;
        @ (posedge apb_pclk_i);
        //3---------------------------------------------------------------------------------------------
        // Wait for transfer to finish 
        //3---------------------------------------------------------------------------------------------
        while (apb_pready_i == 0) begin
          @ (posedge apb_pclk_i);
        end
        apb_penable                    <= 0;
        //apb_psel                       <= 0;
        apb_psel                       <= 1;
        apb_pwrite                     <= 0;
        @ (posedge apb_pclk_i);
        apb_penable                    <= 1;
        @ (posedge apb_pclk_i);
        apb_penable                    <= 0;
      end
    end
  endtask


  //1-------------------------------------------------------------------------------------------------
  // write_apb :This method is used for writing CSR register through APB Interface
  //1-------------------------------------------------------------------------------------------------
  task write_only_apb;
    input    [31:0]                     p_addr_32b                     ;
    input    [31:0]                     p_data_32b                     ;
    integer m_i_i;
    integer num_tran;
    begin
      num_tran = (C_APB_DAT_WIDTH == 32) ? 1 : 4;
      for (m_i_i = 0; m_i_i < num_tran; m_i_i = m_i_i + 1) begin
        @ (posedge apb_pclk_i);
        $write("INFO :: @%0dns %m() :: APB Writing address 0x%x data 0x%x\n",$time,p_addr_32b,p_data_32b);
        apb_psel                       <= 1; 
        apb_paddr                      <= p_addr_32b[10:0] + m_i_i;
        if (C_APB_DAT_WIDTH == 32)
          apb_pwdata                   <= p_data_32b; 
        else
          case (m_i_i)
            0 : apb_pwdata             <= p_data_32b[7:0]; 
            1 : apb_pwdata             <= p_data_32b[15:8]; 
            2 : apb_pwdata             <= p_data_32b[23:16]; 
            3 : apb_pwdata             <= p_data_32b[31:24]; 
          endcase
        apb_pwrite                     <= 1;
        @ (posedge apb_pclk_i);
        apb_penable                    <= 1;
        @ (posedge apb_pclk_i);
        //3---------------------------------------------------------------------------------------------
        // Wait for transfer to finish 
        //3---------------------------------------------------------------------------------------------
        while (apb_pready_i == 0) begin
          @ (posedge apb_pclk_i);
        end
        apb_penable                    <= 0;
        apb_psel                       <= 0;
        apb_pwrite                     <= 0;
      end
    end
  endtask
  
  
  //1-------------------------------------------------------------------------------------------------
  // write_apb :This method is used for writing CSR register through APB Interface
  //1-------------------------------------------------------------------------------------------------
  task write_apb_8b;
    input    [31:0]                     p_addr_32b                     ;
    input    [31:0]                     p_data_32b                     ;
    integer m_i_i;
    integer num_tran;
    begin
      num_tran = 1 ;
      for (m_i_i = 0; m_i_i < num_tran; m_i_i = m_i_i + 1) begin
        @ (posedge apb_pclk_i);
        $write("INFO :: @%0dns %m() :: APB Writing address 0x%x data 0x%x\n",$time,p_addr_32b,p_data_32b);
        apb_psel                       <= 1; 
        apb_paddr                      <= p_addr_32b[10:0] + m_i_i;
        if (C_APB_DAT_WIDTH == 32)
          apb_pwdata                   <= p_data_32b; 
        else
          case (m_i_i)
            0 : apb_pwdata             <= p_data_32b[7:0]; 
            1 : apb_pwdata             <= p_data_32b[15:8]; 
            2 : apb_pwdata             <= p_data_32b[23:16]; 
            3 : apb_pwdata             <= p_data_32b[31:24]; 
          endcase
        apb_pwrite                     <= 1;
        @ (posedge apb_pclk_i);
        apb_penable                    <= 1;
        @ (posedge apb_pclk_i);
        //3---------------------------------------------------------------------------------------------
        // Wait for transfer to finish 
        //3---------------------------------------------------------------------------------------------
        while (apb_pready_i == 0) begin
          @ (posedge apb_pclk_i);
        end
        apb_penable                    <= 0;
        //apb_psel                       <= 0;
        apb_psel                       <= 1;
        apb_pwrite                     <= 0;
      end
    end
  endtask
  

  //1-------------------------------------------------------------------------------------------------
  // read_apb :This method is used for reading register through APB Interface
  //1-------------------------------------------------------------------------------------------------
  task read_apb;
    input    [31:0]                     p_addr_32b                     ;
    begin
      @ (posedge apb_pclk_i);
      apb_psel                       <= 1; 
      apb_paddr                      <= p_addr_32b[10:0]; 
      apb_pwrite                     <= 0;
      @ (posedge apb_pclk_i);
      apb_penable                    <= 1;
      @ (posedge apb_pclk_i);
      //3---------------------------------------------------------------------------------------------
      // Wait for transfer to finish 
      //3---------------------------------------------------------------------------------------------
      while (apb_pready_i == 0) begin
        @ (posedge apb_pclk_i);
      end
      apb_penable                    <= 0;
      apb_psel                       <= 0;
      apb_rd_data                     = apb_prdata_i;
      $write("INFO :: @%0dns %m() :: APB Reading address 0x%x data 0x%x\n",$time,p_addr_32b,apb_prdata_i);
    end
  endtask

  //1-------------------------------------------------------------------------------------------------
  // write_ahb :This method is used for writing CSR 
  //1-------------------------------------------------------------------------------------------------
  task write_ahb;
    input    [31:0]      p_addr_32b                     ;
    input    [31:0]      p_data_32b                     ;
    input    [3:0]       p_size_3b                      ;
    begin
      @ (posedge ahbl_hclk_i);
      $write("INFO :: @%0dns %m() :: AHB Writing address 0x%x data 0x%x\n",$time,p_addr_32b,p_data_32b);
      ahbl_haddr                    <= p_addr_32b; 
      ahbl_htrans                   <= 2;
      ahbl_hsel                     <= 1;
      ahbl_hwrite                   <= 1;
      ahbl_hburst                   <= 0;
      ahbl_hsize                    <= p_size_3b;
      size_3b                       <= p_size_3b;
      @ (posedge ahbl_hclk_i);
      ahbl_haddr                    <= 0;
      ahbl_hwrite                   <= 0;
      ahbl_htrans                   <= 0;

      if( S_AHB_DAT_WIDTH == 32 && p_size_3b == 0) begin
        if( S_AHB_DAT_ENDIANESS == "little-endian")
          case (p_addr_32b[1:0])
            2'b00 : ahbl_hwdata     <= {24'd0, p_data_32b[7:0]};
            2'b01 : ahbl_hwdata     <= {16'd0, p_data_32b[7:0],  8'd0};
            2'b10 : ahbl_hwdata     <= { 8'd0, p_data_32b[7:0], 16'd0};
            2'b11 : ahbl_hwdata     <= {       p_data_32b[7:0], 24'd0};
          endcase
        else // !if( S_AHB_DAT_ENDIANESS == "little-endian")
          case (p_addr_32b[1:0])
            2'b00 : ahbl_hwdata     <= {       p_data_32b[7:0], 24'd0};
            2'b01 : ahbl_hwdata     <= { 8'd0, p_data_32b[7:0], 16'd0};
            2'b10 : ahbl_hwdata     <= {16'd0, p_data_32b[7:0],  8'd0};
            2'b11 : ahbl_hwdata     <= {24'd0, p_data_32b[7:0]};
          endcase
      end
      else
        ahbl_hwdata                 <= p_data_32b;

      @ (posedge ahbl_hclk_i);
      //3---------------------------------------------------------------------------------------------
      // Transfer is completed 
      //3---------------------------------------------------------------------------------------------
      while (ahbl_hready_i == 0) begin
        @ (posedge ahbl_hclk_i);
      end
      ahbl_htrans                   <= 0;
      ahbl_hwrite                   <= 0;
      ahbl_hburst                   <= 0;
      ahbl_hsize                    <= 0;
      ahbl_hsel                     <= 0;
    end
  endtask

  task burst_write_ahb;
    input    [31:0]      p_addr_32b                     ;
    input    [2:0]       p_size_3b                      ;
    input    [31:0]      p_burst_32b                    ;
    integer              m_i_i;
    reg      [31:0]      addr_32b                       ;
    begin
      @ (posedge ahbl_hclk_i);
      $write("INFO :: @%0dns %m() :: AHB Burst Write address 0x%x\n",$time,p_addr_32b);
      ahbl_haddr                    <= p_addr_32b; 
      ahbl_htrans                   <= 2;
      ahbl_hsel                     <= 1;
      ahbl_hwrite                   <= 1;
      ahbl_hburst                   <= 1;
      ahbl_hsize                    <= p_size_3b;
      size_3b                       <= p_size_3b;
      addr_32b                      <= p_addr_32b;
      @ (posedge ahbl_hclk_i);
      
      //No wait states are expected
      for (m_i_i = 0; m_i_i < p_burst_32b; m_i_i = m_i_i + 1) begin
        if(S_AHB_DAT_WIDTH == 32) begin
          ahbl_hwdata               <= burst_prg_data_2d_32b[addr_32b[8:2]];
          addr_32b                  <= addr_32b + 4;
        end
        else begin
          ahbl_hwdata               <= burst_prg_data_2d_8b[addr_32b[8:0]];
          addr_32b                  <= addr_32b + 1;
        end
          
        if(m_i_i == (p_burst_32b-1)) begin
          ahbl_haddr                <= 0;
          ahbl_htrans               <= 0;
          ahbl_hwrite               <= 0;
          ahbl_hburst               <= 0;
          ahbl_hsize                <= 0;
        end
        else
          ahbl_htrans               <= 3;
        
        @ (posedge ahbl_hclk_i);
      end

      //3---------------------------------------------------------------------------------------------
      // Transfer is completed 
      //3---------------------------------------------------------------------------------------------
      ahbl_hsel                 <= 0;
    end
  endtask

  task burst_write_ahb_error_chk;
    input    [31:0]      p_addr_32b                     ;
    input    [2:0]       p_size_3b                      ;
    input    [2:0]       p_burst_type_3b                ;
    input    [31:0]      p_burst_32b                    ;
    integer              m_i_i;
    integer              burst_len;
    begin
      case (p_burst_type_3b)
        3'b000 : burst_len = 1;
        3'b001 : burst_len = p_burst_32b;
        3'b010 : burst_len = 4;
        3'b011 : burst_len = 4;
        3'b100 : burst_len = 8;
        3'b101 : burst_len = 8;
        3'b110 : burst_len = 16;
        3'b111 : burst_len = 16;
      endcase
        
      @ (posedge ahbl_hclk_i);
      $write("INFO :: @%0dns %m() :: AHB Burst Write Error Check address 0x%x\n",$time,p_addr_32b);
      ahbl_haddr                    <= p_addr_32b; 
      ahbl_htrans                   <= 2;
      ahbl_hsel                     <= 1;
      ahbl_hwrite                   <= 1;
      ahbl_hburst                   <= p_burst_type_3b;
      ahbl_hsize                    <= p_size_3b;
      @ (posedge ahbl_hclk_i);
      
      for (m_i_i = 0; m_i_i < burst_len; m_i_i = m_i_i + 1) begin
          ahbl_hwdata               <= $random;
          
        if(m_i_i == (burst_len-1)) begin
          ahbl_haddr                <= 0;
          ahbl_htrans               <= 0;
          ahbl_hwrite               <= 0;
          ahbl_hburst               <= 0;
          ahbl_hsize                <= 0;
        end
        else
          ahbl_htrans               <= 3;
        
        if (ahbl_hready_i == 0 && ahbl_hresp_i == 1) begin
          ahbl_haddr                <= 0;
          ahbl_htrans               <= 0;
          ahbl_hwrite               <= 0;
          ahbl_hburst               <= 0;
          ahbl_hsize                <= 0;
          m_i_i                      = burst_len;
          @ (posedge ahbl_hclk_i);
        end
        else
          @ (posedge ahbl_hclk_i);
      end
      
      if (burst_len == 1) begin
        ahbl_haddr                <= 0;
        ahbl_htrans               <= 0;
        ahbl_hwrite               <= 0;
        ahbl_hburst               <= 0;
        ahbl_hsize                <= 0;
        @ (posedge ahbl_hclk_i);
      end

      //3---------------------------------------------------------------------------------------------
      // Transfer is completed 
      //3---------------------------------------------------------------------------------------------
      ahbl_hsel                 <= 0;
    end
  endtask

  //1-------------------------------------------------------------------------------------------------
  // read_ahb :This method is used for reading CSR 
  //1-------------------------------------------------------------------------------------------------
  task read_ahb;
    input    [31:0]      p_addr_32b                     ;
    input    [2:0]       p_size_3b                      ;
    begin
      @ (posedge ahbl_hclk_i);
      ahbl_haddr                <= p_addr_32b; 
      ahbl_htrans               <= 2;
      ahbl_hsel                 <= 1;
      ahbl_hwrite               <= 0;
      ahbl_hburst               <= 0;
      ahbl_hsize                <= p_size_3b;
      size_3b                   <= p_size_3b;
      while (ahbl_hready_i == 0) begin
        @ (posedge ahbl_hclk_i);
      end
      @ (posedge ahbl_hclk_i);
      ahbl_haddr                <= 0;
      ahbl_htrans               <= 0;
      @ (posedge ahbl_hclk_i);
      //3---------------------------------------------------------------------------------------------
      // Transfer is completed 
      //3---------------------------------------------------------------------------------------------
      while (ahbl_hready_i == 0) begin
        @ (posedge ahbl_hclk_i);
      end
      ahbl_haddr                <= 0; 
      ahbl_htrans               <= 0;
      ahbl_hwrite               <= 0;
      ahbl_hburst               <= 0;
      ahbl_hsize                <= 0;
      ahb_rd_data_32b           <= ahbl_hrdata_i;
      ahbl_hsel                 <= 0;
      $write("INFO :: @%0dns %m() :: Reading address 0x%x data 0x%x\n",
        $time,p_addr_32b,ahbl_hrdata_i);
      @ (posedge ahbl_hclk_i);
    end
  endtask

  task burst_read_ahb;
    input    [31:0]      p_addr_32b                     ;
    input    [2:0]       p_size_3b                      ;
    input    [31:0]      p_burst_32b                    ;
    integer              m_i_i;
    reg      [31:0]      addr_32b                       ;
    begin
      @ (posedge ahbl_hclk_i);
      $write("INFO :: @%0dns %m() :: AHB Burst Read address 0x%x\n",$time,p_addr_32b);
      ahbl_haddr                <= p_addr_32b; 
      ahbl_htrans               <= 2;
      ahbl_hsel                 <= 1;
      ahbl_hwrite               <= 0;
      ahbl_hburst               <= 1;
      ahbl_hsize                <= p_size_3b;
      size_3b                   <= p_size_3b;
      addr_32b                  <= p_addr_32b;
            
      @ (posedge ahbl_hclk_i);

      //No wait states are expected
      for (m_i_i = 0; m_i_i < p_burst_32b; m_i_i = m_i_i + 1) begin        
        if(m_i_i == (p_burst_32b-1)) begin
          ahbl_haddr            <= 0;
          ahbl_htrans           <= 0;
          ahbl_hwrite           <= 0;
          ahbl_hburst           <= 0;
          ahbl_hsize            <= 0;
        end
        else
          ahbl_htrans           <= 3;
        
        @ (negedge ahbl_hclk_i);
        
        while (ahbl_hready_i == 0) begin
          @ (negedge ahbl_hclk_i);
        end
        
        if(S_AHB_DAT_WIDTH == 32) begin
          burst_rd_data_2d_32b[addr_32b[8:2]] <= ahbl_hrdata_i;
          addr_32b              <= addr_32b + 4;
        end
        else begin
          burst_rd_data_2d_8b[addr_32b[8:0]]  <= ahbl_hrdata_i;
          addr_32b              <= addr_32b + 1;
        end
        
        @ (posedge ahbl_hclk_i);
      end
      
      //3---------------------------------------------------------------------------------------------
      // Transfer is completed 
      //3---------------------------------------------------------------------------------------------
      ahbl_hsel             <= 0;
    end
  endtask

  task burst_read_ahb_error_chk;
    input    [31:0]      p_addr_32b                     ;
    input    [2:0]       p_size_3b                      ;
    input    [2:0]       p_burst_type_3b                ;
    input    [31:0]      p_burst_32b                    ;
    integer              m_i_i;
    integer              burst_len;
    begin
      case (p_burst_type_3b)
        3'b000 : burst_len = 1;
        3'b001 : burst_len = p_burst_32b;
        3'b010 : burst_len = 4;
        3'b011 : burst_len = 4;
        3'b100 : burst_len = 8;
        3'b101 : burst_len = 8;
        3'b110 : burst_len = 16;
        3'b111 : burst_len = 16;
      endcase
      
      @ (posedge ahbl_hclk_i);
      $write("INFO :: @%0dns %m() :: AHB Burst Read Error Check address 0x%x\n",$time,p_addr_32b);
      ahbl_haddr                <= p_addr_32b; 
      ahbl_htrans               <= 2;
      ahbl_hsel                 <= 1;
      ahbl_hwrite               <= 0;
      ahbl_hburst               <= p_burst_type_3b;
      ahbl_hsize                <= p_size_3b;     
      @ (posedge ahbl_hclk_i);

      for (m_i_i = 0; m_i_i < burst_len; m_i_i = m_i_i + 1) begin        
        if(m_i_i == (burst_len-1)) begin
          ahbl_haddr            <= 0;
          ahbl_htrans           <= 0;
          ahbl_hwrite           <= 0;
          ahbl_hburst           <= 0;
          ahbl_hsize            <= 0;
        end
        else
          ahbl_htrans           <= 3;
        
        if (ahbl_hready_i == 0 && ahbl_hresp_i == 1) begin
          ahbl_haddr            <= 0;
          ahbl_htrans           <= 0;
          ahbl_hburst           <= 0;
          ahbl_hsize            <= 0;
          m_i_i                  = burst_len;
          @ (posedge ahbl_hclk_i);
        end
        else
          @ (posedge ahbl_hclk_i);
      end
      
      if (burst_len == 1) begin
        ahbl_haddr                <= 0;
        ahbl_htrans               <= 0;
        ahbl_hburst               <= 0;
        ahbl_hsize                <= 0;
        @ (posedge ahbl_hclk_i);
      end
      
      //3---------------------------------------------------------------------------------------------
      // Transfer is completed 
      //3---------------------------------------------------------------------------------------------
      ahbl_hsel             <= 0;
    end
  endtask

  //1-------------------------------------------------------------------------------------------------
  // wait_clocks :Wait for some duration 
  //1-------------------------------------------------------------------------------------------------
  //           p_delay_32b :Number of clocks delay 
  //1-------------------------------------------------------------------------------------------------
  task wait_clocks;
    input    [31:0]                     p_delay_32b                    ;
    begin
      //3---------------------------------------------------------------------------------------------
      // Wait for the delay 
      //3---------------------------------------------------------------------------------------------
      repeat (p_delay_32b) begin
        @ (posedge apb_pclk_i);
      end
    end
  endtask
  
  // Delay ss_n_i signal
  always @ (posedge sclk_wr or posedge rst_i) begin
    if (rst_i == 1) begin
	    ss_n_dly <= 1'h0;
	end
	else begin
	    ss_n_dly <= ss_n_i;
	end
  end
  
  // Control MISO based on CPOL
  always @ (posedge sclk_rd or posedge rst_i) begin
    if (rst_i == 1) begin
      slv_out_b         <= 0;
	end
	else begin
	    if(state_4b == PAGE_RD_DATA) begin
		    if (ss_n_i == 1) begin
                slv_out_b   <= 1'b0;
            end
            else begin
                slv_out_b   <= data_2d_8b[cmd_addr_24b_32b+data_idx_13b][bit_cnt_3b];
			end
		end
		else begin
            slv_out_b   <= 1'b0;
		end
	end
  end
  //1-------------------------------------------------------------------------------------------------
  // This block implements the SPI Flash model
  //1-------------------------------------------------------------------------------------------------
  always @ (posedge sclk_wr or posedge rst_i) begin
    //--------------------------------------------------------------------
    // Reset all the varaibles
    //--------------------------------------------------------------------
    if (rst_i == 1) begin
      state_4b          <= IDLE;
      cmd_opcode_8b     <= 0;
      bit_cnt_3b        <= 0;
      addr_cnt_5b       <= 0;
      cmd_idx_2b        <= 0;
      data_idx_13b      <= 0;
      cmd_addr_24b_32b  <= 0;
      cmd_data_8b       <= 0;
    end else begin
    //--------------------------------------------------------------------
    // process the state machine
    //--------------------------------------------------------------------
      case (state_4b) 
        //----------------------------------------------------------------
        // IDLE State - Wait for Chip to de-assert
        //----------------------------------------------------------------
        IDLE: begin
          if (~ss_n_i) begin
			if (CPHASE) begin
                bit_cnt_3b    <= LSB_FIRST[0] ? 0 : 7;
			end
			else begin
                cmd_opcode_8b <= LSB_FIRST[0] ? {7'h0,mosi_w} : {mosi_w,7'h0};
                bit_cnt_3b    <= LSB_FIRST[0] ? 1 : 6;
			end
            data_idx_13b  <= 0;
            state_4b      <= CMD_OPCODE;
          end
        end

        //----------------------------------------------------------------
        // CMD_OPCODE State - Sample the command opcode driven by SPI Controller
        //----------------------------------------------------------------
        CMD_OPCODE: begin
              cmd_opcode_8b[bit_cnt_3b] <= mosi_w;
              cmd_2d_8b[cmd_idx_2b]     <= LSB_FIRST[0] ?  {mosi_w,cmd_opcode_8b[6:0]} : {cmd_opcode_8b[7:1],mosi_w};

          if (~LSB_FIRST[0] && bit_cnt_3b == 0) begin
            if (({cmd_opcode_8b[7:1],mosi_w} == SPI_WRT_ENB ) || 
                ({cmd_opcode_8b[7:1],mosi_w} == SPI_WRT_DISB) || 
                ({cmd_opcode_8b[7:1],mosi_w} == SPI_CHIP_ERS) || 
                ({cmd_opcode_8b[7:1],mosi_w} == SPI_PWD_UP  ) || 
                ({cmd_opcode_8b[7:1],mosi_w} == SPI_PWD_DOWN))begin
              state_4b   <= IDLE;
              cmd_idx_2b <= cmd_idx_2b + 1;
            end else if ({cmd_opcode_8b[7:1],mosi_w} == SPI_READ_STAT) begin
              state_4b   <= READ_STAT;
              bit_cnt_3b <= 7;
              cmd_idx_2b <= 0;
            end else if ({cmd_opcode_8b[7:1],mosi_w} == SPI_PAGE_PRG) begin
              state_4b    <= PAGE_ADDR;
              addr_cnt_5b <= (S_AHB_ADR_WIDTH == 32) ? 31 : 23;
              bit_cnt_3b  <= 7;
              cmd_idx_2b  <= cmd_idx_2b + 1;
            end else if ({cmd_opcode_8b[7:1],mosi_w} == SPI_READ) begin
              state_4b    <= PAGE_ADDR;
              addr_cnt_5b <= (S_AHB_ADR_WIDTH == 32) ? 31 : 23;
              bit_cnt_3b  <= 7;
            end else if ({cmd_opcode_8b[7:1],mosi_w} == SPI_BLK1_ERS) begin
              state_4b    <= PAGE_ADDR;
              addr_cnt_5b <= (S_AHB_ADR_WIDTH == 32) ? 31 : 23;
              bit_cnt_3b  <= 7;
              cmd_idx_2b  <= cmd_idx_2b + 1;
            end
          end else if (LSB_FIRST[0] && bit_cnt_3b == 7) begin
            if (({mosi_w,cmd_opcode_8b[6:0]} == SPI_WRT_ENB ) || 
                ({mosi_w,cmd_opcode_8b[6:0]} == SPI_WRT_DISB) || 
                ({mosi_w,cmd_opcode_8b[6:0]} == SPI_CHIP_ERS) || 
                ({mosi_w,cmd_opcode_8b[6:0]} == SPI_PWD_UP  ) || 
                ({mosi_w,cmd_opcode_8b[6:0]} == SPI_PWD_DOWN))begin
              state_4b   <= IDLE;
              cmd_idx_2b <= cmd_idx_2b + 1;
            end else if ({mosi_w,cmd_opcode_8b[6:0]} == SPI_READ_STAT) begin
              state_4b   <= READ_STAT;
              bit_cnt_3b <= 7;
              cmd_idx_2b <= 0;
            end else if ({mosi_w,cmd_opcode_8b[6:0]} == SPI_PAGE_PRG) begin
              state_4b    <= PAGE_ADDR;
              addr_cnt_5b <= (S_AHB_ADR_WIDTH == 32) ? 31 : 23;
              bit_cnt_3b  <= 0;
              cmd_idx_2b  <= cmd_idx_2b + 1;
            end else if ({mosi_w,cmd_opcode_8b[6:0]} == SPI_READ) begin
              state_4b    <= PAGE_ADDR;
              addr_cnt_5b <= (S_AHB_ADR_WIDTH == 32) ? 31 : 23;
              bit_cnt_3b  <= 0;
            end else if ({mosi_w,cmd_opcode_8b[6:0]} == SPI_BLK1_ERS) begin
              state_4b    <= PAGE_ADDR;
              addr_cnt_5b <= (S_AHB_ADR_WIDTH == 32) ? 31 : 23;
              bit_cnt_3b  <= 0;
              cmd_idx_2b  <= cmd_idx_2b + 1;
            end
          end else begin
            bit_cnt_3b    <= LSB_FIRST[0] ? bit_cnt_3b + 1 : bit_cnt_3b - 1;
          end
        end

        //----------------------------------------------------------------
        // PAGE_ADDR State - Sample the Page write address
        //----------------------------------------------------------------
        PAGE_ADDR: begin
          addr_cnt_5b  <= addr_cnt_5b - 1;
          if (S_AHB_ADR_WIDTH == 32)
            cmd_addr_24b_32b       <= LSB_FIRST[0] ? {mosi_w,cmd_addr_24b_32b[31:1]} : {cmd_addr_24b_32b[30:0],mosi_w};
          else
            cmd_addr_24b_32b[23:0] <= LSB_FIRST[0] ? {mosi_w,cmd_addr_24b_32b[23:1]} : {cmd_addr_24b_32b[22:0],mosi_w};
          
          if (addr_cnt_5b == 0) begin
            if (cmd_opcode_8b == SPI_READ) begin
              state_4b <= PAGE_RD_DATA;
              bit_cnt_3b  <= LSB_FIRST[0] ? bit_cnt_3b + 1 : bit_cnt_3b - 1;
			  slv_out_b   <= data_2d_8b[cmd_addr_24b_32b_w+data_idx_13b][bit_cnt_3b];
            end else if(cmd_opcode_8b == SPI_BLK1_ERS) begin
              state_4b <= BLK_ERASE;
            end else begin
              state_4b <= PAGE_WR_DATA;
            end
          end
        end

        //----------------------------------------------------------------
        // PAGE_WR_DATA State - Sample the Page write data
        //----------------------------------------------------------------
        PAGE_WR_DATA: begin
          bit_cnt_3b  <= LSB_FIRST[0] ? bit_cnt_3b + 1 : bit_cnt_3b - 1;
          cmd_data_8b <= LSB_FIRST[0] ? {mosi_w,cmd_data_8b[7:1]} : {cmd_data_8b[6:0],mosi_w};
          if ((~LSB_FIRST[0] && bit_cnt_3b == 0) || 
              ( LSB_FIRST[0] && bit_cnt_3b == 7)) begin
            data_2d_8b[cmd_addr_24b_32b+data_idx_13b] <= LSB_FIRST[0] ? {mosi_w,cmd_data_8b[7:1]} : {cmd_data_8b[6:0],mosi_w};
            data_idx_13b                              <= data_idx_13b + 1;
          end

          if (ss_n_i) begin
            state_4b <= IDLE;
          end
        end

        //----------------------------------------------------------------
        // PAGE_RD_DATA State - Sample the Page read data
        //----------------------------------------------------------------
        PAGE_RD_DATA: begin
          bit_cnt_3b  <= LSB_FIRST[0] ? bit_cnt_3b + 1 : bit_cnt_3b - 1;
          if ((~LSB_FIRST[0] && bit_cnt_3b == 0) || 
              ( LSB_FIRST[0] && bit_cnt_3b == 7)) begin
            data_idx_13b             <= data_idx_13b + 1;
          end

          if (ss_n_i == 1) begin
            state_4b    <= IDLE;
          end
        end

        //----------------------------------------------------------------
        // READ_STAT State - Drive the read status value
        //----------------------------------------------------------------
        READ_STAT: begin
          bit_cnt_3b  <=  bit_cnt_3b - 1;
          if (bit_cnt_3b == 0) begin
            state_4b <= IDLE;
          end
        end

        //----------------------------------------------------------------
        // BLK_ERASE State - Erase the flash data
        //----------------------------------------------------------------
        BLK_ERASE: begin
          for (idx_13b = 0; idx_13b < 4095; idx_13b = idx_13b + 1) begin
            data_2d_8b[{cmd_addr_24b_32b[11],idx_13b[11:0]}] = 8'hFF;
          end
          state_4b <= IDLE;
        end

      endcase
    end
  end


  //1-------------------------------------------------------------------------------------------------
  // This block implements the Sample Debug model
  //1-------------------------------------------------------------------------------------------------
  always @ (posedge sclk_i or posedge rst_i) begin
    //--------------------------------------------------------------------
    // Reset all the varaibles
    //--------------------------------------------------------------------
    if (rst_i == 1) begin
      smp_cnt_3b  <= 7;
      smp_data_8b <= 0;
    end else begin
      //--------------------------------------------------------------------
      // process the state machine
      //--------------------------------------------------------------------
      if (ss_n_i == 0) begin
        if (smp_cnt_3b == 0) begin
          if (state_4b == PAGE_RD_DATA) begin
            smp_cnt_3b <= 7;
          end else begin
            smp_cnt_3b <= 7;
          end
        end else begin
          smp_cnt_3b <= smp_cnt_3b - 1;
          if (state_4b == PAGE_RD_DATA) 
            smp_data_8b[smp_cnt_3b] <= slv_out_b;
          else
            smp_data_8b[smp_cnt_3b] <= mosi_w;
        end
      end else begin
        smp_cnt_3b <= 7;
      end
    end
  end


  //1-------------------------------------------------------------------------------------------------
  // chkr_spi : This method implements the chkr_spi 
  //1-------------------------------------------------------------------------------------------------
  task chkr_spi;
    input [7:0]                    p_cmd_8b                     ; // Expected command type
    input [7:0]                    p_len_8b                     ; // Expected length
    input [31:0]                   p_addr_32b                   ; // Expected address      

    begin
      trans_i                      = trans_i + 1;
      //------------------------------------------------------------------------
      // Make comparison based on command type
      //------------------------------------------------------------------------
      case (p_cmd_8b)
        //----------------------------------------------------------------------
        // SPI Write enable command
        //----------------------------------------------------------------------
        SPI_WRT_ENB: begin
          if (p_cmd_8b === cmd_opcode_8b) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end
        end

        //----------------------------------------------------------------------
        // SPI Write Disable command
        //----------------------------------------------------------------------
        SPI_WRT_DISB: begin
          if (p_cmd_8b === cmd_opcode_8b) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end
        end

        //----------------------------------------------------------------------
        // SPI Chip erase command
        //----------------------------------------------------------------------
        SPI_CHIP_ERS: begin
          //--------------------------------------------------------------------
          // WRT END Command should have received first
          //--------------------------------------------------------------------
          if (SPI_WRT_ENB === cmd_2d_8b[0]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
          end

          //--------------------------------------------------------------------
          // SPI_CHIP_ERS Command should have received second
          //--------------------------------------------------------------------
          if (SPI_CHIP_ERS === cmd_2d_8b[1]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-2 SPI_CHIP_ERS expected 0x%x, Got 0x%x\n",$time,SPI_CHIP_ERS,cmd_2d_8b[1]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-2 SPI_CHIP_ERS expected 0x%x, Got 0x%x\n",$time,SPI_CHIP_ERS,cmd_2d_8b[1]);
          end

          //--------------------------------------------------------------------
          // SPI_READ_STAT Command should have received last
          //--------------------------------------------------------------------
          if (SPI_READ_STAT === cmd_2d_8b[2]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
          end
        end

        //----------------------------------------------------------------------
        // SPI Power up command
        //----------------------------------------------------------------------
        SPI_PWD_UP: begin
          if (p_cmd_8b === cmd_opcode_8b) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end
        end

        //----------------------------------------------------------------------
        // SPI Power down command
        //----------------------------------------------------------------------
        SPI_PWD_DOWN: begin
          if (p_cmd_8b === cmd_opcode_8b) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode expected 0x%x, Got 0x%x\n",$time,p_cmd_8b,cmd_opcode_8b);
          end
        end

        //----------------------------------------------------------------------
        // SPI Blcok Erase Type 1 command
        //----------------------------------------------------------------------
        SPI_BLK1_ERS: begin
          //--------------------------------------------------------------------
          // WRT END Command should have received first
          //--------------------------------------------------------------------
          if (SPI_WRT_ENB === cmd_2d_8b[0]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
          end

          //--------------------------------------------------------------------
          // SPI_BLK1_ERS Command should have received second
          //--------------------------------------------------------------------
          if (SPI_BLK1_ERS === cmd_2d_8b[1]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-2 SPI_BLK1_ERS expected 0x%x, Got 0x%x\n",$time,SPI_BLK1_ERS,cmd_2d_8b[1]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-2 SPI_BLK1_ERS expected 0x%x, Got 0x%x\n",$time,SPI_BLK1_ERS,cmd_2d_8b[1]);
          end

          //Compare Address
          if (p_addr_32b[31:0] === cmd_addr_24b_32b) begin
            $write("MSG :: @%0dns %m() :: Match Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
          end

          //--------------------------------------------------------------------
          // SPI_READ_STAT Command should have received last
          //--------------------------------------------------------------------
          if (SPI_READ_STAT === cmd_2d_8b[2]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
          end
        end

        //----------------------------------------------------------------------
        // SPI Byte/Page Program command
        //----------------------------------------------------------------------
        SPI_PAGE_PRG: begin
          //--------------------------------------------------------------------
          // WRT END Command should have received first
          //--------------------------------------------------------------------
		 
          if (SPI_WRT_ENB === cmd_2d_8b[0]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
          end
	

          //--------------------------------------------------------------------
          // SPI_PAGE_PRG Command should have received second
          //--------------------------------------------------------------------
		   
		  if (SPI_PAGE_PRG === cmd_2d_8b[1]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-2 SPI_PAGE_PRG expected 0x%x, Got 0x%x\n",$time,SPI_PAGE_PRG,cmd_2d_8b[1]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-2 SPI_PAGE_PRG expected 0x%x, Got 0x%x\n",$time,SPI_PAGE_PRG,cmd_2d_8b[1]);
          end
	

          //Compare Address
		   
          if (p_addr_32b[31:0] === cmd_addr_24b_32b) begin
            $write("MSG :: @%0dns %m() :: Match Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
          end
	

          //--------------------------------------------------------------------
          // SPI_READ_STAT Command should have received last
          //--------------------------------------------------------------------
		   
          if (SPI_READ_STAT === cmd_2d_8b[2]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
          end
	
        end
		
        //----------------------------------------------------------------------
        // SPI Read command
        //----------------------------------------------------------------------
        SPI_READ: begin
		  
          //--------------------------------------------------------------------
          // WRT END Command should have received first
          //--------------------------------------------------------------------
          if (SPI_READ === cmd_2d_8b[0]) begin
            $write("MSG :: @%0dns %m() :: Match Command Opcode-1 SPI_READ expected 0x%x, Got 0x%x\n",$time,SPI_READ,cmd_2d_8b[0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Opcode-1 SPI_READ expected 0x%x, Got 0x%x\n",$time,SPI_READ,cmd_2d_8b[0]);
          end

          if (p_addr_32b[31:0] === cmd_addr_24b_32b) begin
            $write("MSG :: @%0dns %m() :: Match Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
          end
        end


      endcase


    end
  endtask


  //1-------------------------------------------------------------------------------------------------
  // ahb_checker : This method implements the chkr_spi for write and read command using AHB Interface
  //1-------------------------------------------------------------------------------------------------
  task ahb_checker;
    input                          p_mode_b                     ; // Mode input to select whether Page program or read
    input [31:0]                   p_addr_32b                   ; // Expected address      
    input [31:0]                   p_data_32b                   ; // Expected data

    begin
      trans_i                      = trans_i + 1;
      //------------------------------------------------------------------------
      // Program command chkr_spi
      //------------------------------------------------------------------------
      if (p_mode_b == 0) begin
        //--------------------------------------------------------------------
        // WRT EN Command should have received first
        //--------------------------------------------------------------------
        if (SPI_WRT_ENB === cmd_2d_8b[0]) begin
          $write("MSG :: @%0dns %m() :: Match Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
        end else begin
          process_err();
          $write("ERROR :: @%0dns %m() :: Command Opcode-1 WRT_ENB expected 0x%x, Got 0x%x\n",$time,SPI_WRT_ENB,cmd_2d_8b[0]);
        end

        //--------------------------------------------------------------------
        // SPI_PAGE_PRG Command should have received second
        //--------------------------------------------------------------------
        if (SPI_PAGE_PRG === cmd_2d_8b[1]) begin
          $write("MSG :: @%0dns %m() :: Match Command Opcode-2 SPI_PAGE_PRG expected 0x%x, Got 0x%x\n",$time,SPI_PAGE_PRG,cmd_2d_8b[1]);
        end else begin
          process_err();
          $write("ERROR :: @%0dns %m() :: Command Opcode-2 SPI_PAGE_PRG expected 0x%x, Got 0x%x\n",$time,SPI_PAGE_PRG,cmd_2d_8b[1]);
        end

        //Compare Address
        if (p_addr_32b[31:0] === cmd_addr_24b_32b) begin
          $write("MSG :: @%0dns %m() :: Match Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
        end else begin
          process_err();
          $write("ERROR :: @%0dns %m() :: Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
        end
        //Compare Data
        if (S_AHB_DAT_WIDTH == 32) begin
          if (size_3b == 2) begin
            if ((p_data_32b[7:0] === data_2d_8b[cmd_addr_24b_32b+0]) && (p_data_32b[15:8] === data_2d_8b[cmd_addr_24b_32b+1]) && (p_data_32b[23:16] === data_2d_8b[cmd_addr_24b_32b+2]) && (p_data_32b[31:24] === data_2d_8b[cmd_addr_24b_32b+3])) begin
              $write("MSG :: @%0dns %m() :: Match Command Data expected 0x%x, Got 0x%x\n",$time,p_data_32b,{data_2d_8b[cmd_addr_24b_32b+3],data_2d_8b[cmd_addr_24b_32b+2],data_2d_8b[cmd_addr_24b_32b+1],data_2d_8b[cmd_addr_24b_32b+0]});
            end else begin
              process_err();
              $write("ERROR :: @%0dns %m() :: Command Data expected 0x%x, Got 0x%x\n",$time,p_data_32b,{data_2d_8b[cmd_addr_24b_32b+3],data_2d_8b[cmd_addr_24b_32b+2],data_2d_8b[cmd_addr_24b_32b+1],data_2d_8b[cmd_addr_24b_32b+0]});
            end
          end else if (size_3b == 0) begin
            if (p_data_32b[7:0] === data_2d_8b[cmd_addr_24b_32b+0]) begin
              $write("MSG :: @%0dns %m() :: Match Command Data expected 0x%x, Got 0x%x\n",$time,p_data_32b,data_2d_8b[cmd_addr_24b_32b+0]);
            end else begin
              process_err();
              $write("ERROR :: @%0dns %m() :: Command Data expected 0x%x, Got 0x%x\n",$time,p_data_32b,data_2d_8b[cmd_addr_24b_32b+0]);
            end
          end
        end else begin
          if ((p_data_32b[7:0] === data_2d_8b[cmd_addr_24b_32b+0])) begin
            $write("MSG :: @%0dns %m() :: Match Command Data expected 0x%x, Got 0x%x\n",$time,p_data_32b[7:0],data_2d_8b[cmd_addr_24b_32b+0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Data expected 0x%x, Got 0x%x\n",$time,p_data_32b[7:0],data_2d_8b[cmd_addr_24b_32b+0]);
          end
        end
        //--------------------------------------------------------------------
        // SPI_READ_STAT Command should have received last
        //--------------------------------------------------------------------
        if (SPI_READ_STAT === cmd_2d_8b[2]) begin
          $write("MSG :: @%0dns %m() :: Match Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
        end else begin
          process_err();
          $write("ERROR :: @%0dns %m() :: Command Opcode-3 SPI_READ_STAT expected 0x%x, Got 0x%x\n",$time,SPI_READ_STAT,cmd_2d_8b[2]);
        end

      //------------------------------------------------------------------------
      // Read command chkr_spi
      //------------------------------------------------------------------------
      end else begin
        //--------------------------------------------------------------------
        // SPI_READ Command should have received first
        //--------------------------------------------------------------------
        if (SPI_READ === cmd_2d_8b[0]) begin
          $write("MSG :: @%0dns %m() :: Match Command Opcode-1 SPI_READ expected 0x%x, Got 0x%x\n",$time,SPI_READ,cmd_2d_8b[0]);
        end else begin
          process_err();
          $write("ERROR :: @%0dns %m() :: Command Opcode-1 SPI_READ expected 0x%x, Got 0x%x\n",$time,SPI_READ,cmd_2d_8b[0]);
        end

        //Compare Address
        if (p_addr_32b[31:0] === cmd_addr_24b_32b) begin
          $write("MSG :: @%0dns %m() :: Match Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
        end else begin
          process_err();
          $write("ERROR :: @%0dns %m() :: Command Address expected 0x%x, Got 0x%x\n",$time,p_addr_32b[23:0],cmd_addr_24b_32b[23:0]);
        end
        //Compare Data
        if (S_AHB_DAT_WIDTH == 32) begin
          if (size_3b == 2) begin
            if ((p_data_32b[7:0] === data_2d_8b[cmd_addr_24b_32b+0]) && (p_data_32b[15:8] === data_2d_8b[cmd_addr_24b_32b+1]) && (p_data_32b[23:16] === data_2d_8b[cmd_addr_24b_32b+2]) && (p_data_32b[31:24] === data_2d_8b[cmd_addr_24b_32b+3])) begin
              $write("MSG :: @%0dns %m() :: Match Command Data expected 0x%x, Got 0x%x\n",$time,{data_2d_8b[cmd_addr_24b_32b+3],data_2d_8b[cmd_addr_24b_32b+2],data_2d_8b[cmd_addr_24b_32b+1],data_2d_8b[cmd_addr_24b_32b+0]},p_data_32b);
            end else begin
              process_err();
              $write("ERROR :: @%0dns %m() :: Command Data expected 0x%x, Got 0x%x\n",$time,{data_2d_8b[cmd_addr_24b_32b+3],data_2d_8b[cmd_addr_24b_32b+2],data_2d_8b[cmd_addr_24b_32b+1],data_2d_8b[cmd_addr_24b_32b+0]},p_data_32b);
            end
          end else if (size_3b == 0) begin
            if (p_data_32b[7:0] === data_2d_8b[cmd_addr_24b_32b+0]) begin
              $write("MSG :: @%0dns %m() :: Match Command Data expected 0x%x, Got 0x%x\n",$time,data_2d_8b[cmd_addr_24b_32b+0],p_data_32b);
            end else begin
              process_err();
              $write("ERROR :: @%0dns %m() :: Command Data expected 0x%x, Got 0x%x\n",$time,data_2d_8b[cmd_addr_24b_32b+0],p_data_32b);
            end
          end
        end else begin
          if ((p_data_32b[7:0] === data_2d_8b[cmd_addr_24b_32b+0])) begin
            $write("MSG :: @%0dns %m() :: Match Command Data expected 0x%x, Got 0x%x\n",$time,data_2d_8b[cmd_addr_24b_32b+0],p_data_32b[7:0]);
          end else begin
            process_err();
            $write("ERROR :: @%0dns %m() :: Command Data expected 0x%x, Got 0x%x\n",$time,data_2d_8b[cmd_addr_24b_32b+0],p_data_32b[7:0]);
          end
        end

      end


    end
  endtask
  

  //1-------------------------------------------------------------------------------------------------
  // process_err :This method process the error 
  //1-------------------------------------------------------------------------------------------------
  //             parameter :No parameter 
  //1-------------------------------------------------------------------------------------------------
  task process_err;
    begin
      errs_i                         = errs_i + 1;
      //3---------------------------------------------------------------------------------------------
      // Check if finish was passed 
      //3---------------------------------------------------------------------------------------------
      if (finish_on_error_b == 1) begin
        $finish(1);
      end
    end
  endtask


  //1###################################################################################################
  // Include all testcases here 
  //1###################################################################################################
  task test_basic;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] erase_addr_32b;
    reg [31:0] byte_addr_32b;
    reg [31:0] prg_data_2d_32b [255:0];
    reg [31:0] rd_data_2d_32b  [255:0];
    reg [7:0]  prg_data_2d_8b  [255:0];
    reg [7:0]  rd_data_2d_8b   [255:0];
    reg [2:0]  size_3b;
    reg [31:0] data_32b;
    integer m_i_i;
    integer m_j_i;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        if (C_PORT_ENABLE == 1) begin
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i == 1) begin
               read_apb(addr_32b);
		  end
          if( S_AHB_ADR_WIDTH == 32) begin
            //Set the flash addressing mode to 32-bit mode
            addr_32b       = 0;
            addr_32b[10:0] = SPI_ADDR_MODE;
            data_32b       = 0;
            data_32b[7:0]  = 8'h01;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
          end
          if(S_AHB_DAT_WIDTH == 32 && C_APB_DAT_WIDTH == 32) begin
            addr_32b[31:12] = 20'h00000;
            addr_32b[11:2]  = $random;
            addr_32b[1:0]   = 2'h0;
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              prg_data_2d_32b[m_j_i][31:0]  = $random;
              ahb_intf_en_b                 = 1;
              size_3b                       = 2;
              //Initiate the transaction through AHB
              write_ahb(prg_addr_32b,prg_data_2d_32b[m_j_i],size_3b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
		      end
              //Call the chkr_spi to check Flash model receives all the commands successfully
              ahb_checker(0,prg_addr_32b,prg_data_2d_32b[m_j_i]);
			  //Increment AHB write address
              prg_addr_32b    = prg_addr_32b+4;
            end
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through AHB
              ahb_intf_en_b   = 1;
              size_3b         = 2;
              read_ahb(rd_addr_32b,size_3b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
		      end
			  //Store AHB read data
			  rd_data_2d_32b[m_j_i] = ahb_rd_data_32b;
              //Call the chkr_spi to check Flash model receives all the commands successfully
              ahb_checker(1,rd_addr_32b,rd_data_2d_32b[m_j_i]);
              ahb_intf_en_b   = 0;
			  //Increment AHB read address
              rd_addr_32b     = rd_addr_32b+4;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i]);
            end
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            erase_addr_32b  = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              prg_data_2d_32b[m_j_i][31:0]  = $random;
              ahb_intf_en_b                 = 1;
              size_3b                       = 2;
              //Initiate the transaction through AHB
              write_ahb(prg_addr_32b,prg_data_2d_32b[m_j_i],size_3b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
		      end
              //Call the chkr_spi to check Flash model receives all the commands successfully
              ahb_checker(0,prg_addr_32b,prg_data_2d_32b[m_j_i]);
			  //Increment AHB write address
              prg_addr_32b    = prg_addr_32b+4;
            end
            //Set the spi flash address
            addr_32b       = 0;
            addr_32b[10:0] = SPI_ADDR_BYTE0;
            data_32b       = erase_addr_32b;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
            //Program the value to initiate block erase type 1 transaction
            addr_32b       = 0;
            addr_32b[10:0] = BLK_ERASE_TYPE1;
            data_32b       = 32'h0000_0001;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            //Call the chkr_spi to check Flash model receives the opcode successfully
            chkr_spi(SPI_BLK1_ERS,0,erase_addr_32b);
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through AHB
              ahb_intf_en_b   = 1;
              size_3b         = 2;
              read_ahb(rd_addr_32b,size_3b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
		      end
			  //Store AHB read data
              rd_data_2d_32b[m_j_i] = ahb_rd_data_32b;
              //Call the chkr_spi to check Flash model receives all the commands successfully
              ahb_checker(1,rd_addr_32b,rd_data_2d_32b[m_j_i]);
			  //Increment AHB read address
              rd_addr_32b     = rd_addr_32b+4;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              if (rd_data_2d_32b[m_j_i] != 32'hFFFFFFFF) begin
                process_err();
                $write("ERROR :: @%0dns %m() :: Read Data expected after Erase 32'hFFFFFF,Got 0x%x\n",$time,rd_data_2d_32b[m_j_i]);
              end
            end
          end
          addr_32b[31:12] = 20'h00000;
          addr_32b[11:2]  = $random;
          addr_32b[11:2]  = addr_32b[11:2] < 256 ? addr_32b[11:2]+256:addr_32b[11:2];
          addr_32b[1:0]   = 2'h0;
          prg_addr_32b    = addr_32b-64;
          rd_addr_32b     = addr_32b-64;
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word write transaction
            prg_data_2d_8b[m_j_i]  = $random;
            ahb_intf_en_b          = 1;
            size_3b                = 0;
            //Initiate the transaction through AHB
            write_ahb(prg_addr_32b,prg_data_2d_8b[m_j_i],size_3b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            byte_addr_32b   = prg_addr_32b;
            //Call the chkr_spi to check Flash model receives all the commands successfully
            ahb_checker(0,byte_addr_32b,prg_data_2d_8b[m_j_i]);
			//Increment AHB write address
            prg_addr_32b    = prg_addr_32b+1;
          end
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word read transaction & Initiate the transaction through AHB
            ahb_intf_en_b   = 1;
            size_3b         = 0;
            read_ahb(rd_addr_32b,size_3b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            if( S_AHB_DAT_WIDTH == 32) begin
              if( S_AHB_DAT_ENDIANESS == "little-endian")
                case (rd_addr_32b[1:0])
                  2'b00: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[7:0];
                  2'b01: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[15:8];
                  2'b10: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[23:16];
                  2'b11: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[31:24];
                endcase
              else //if( S_AHB_DAT_ENDIANESS != "little-endian")
                case (rd_addr_32b[1:0])
                  2'b00: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[31:24];
                  2'b01: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[23:16];
                  2'b10: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[15:8];
                  2'b11: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[7:0];
                endcase
            end
            else begin//if( S_AHB_DAT_WIDTH != 32)
			  //Store AHB read data
              rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[7:0];
			end
            //Call the chkr_spi to check Flash model receives all the commands successfully
            byte_addr_32b      = rd_addr_32b;
            ahb_checker(1,byte_addr_32b,rd_data_2d_8b[m_j_i]);
            ahb_intf_en_b   = 0;
			//Increment AHB read address
            rd_addr_32b     = rd_addr_32b+1;
          end
          //compare the data for write and read
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            do_data_compare(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i]);
          end
          prg_addr_32b    = addr_32b-64;
          rd_addr_32b     = addr_32b-64;
          erase_addr_32b  = addr_32b-64;
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word write transaction
            prg_data_2d_8b[m_j_i]  = $random;
            ahb_intf_en_b          = 1;
            size_3b                = 0;
            //Initiate the transaction through AHB
            write_ahb(prg_addr_32b,prg_data_2d_8b[m_j_i],size_3b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            byte_addr_32b      = prg_addr_32b;
            //Call the chkr_spi to check Flash model receives all the commands successfully
            ahb_checker(0,byte_addr_32b,prg_data_2d_8b[m_j_i]);
			//Increment AHB write address
            prg_addr_32b       = prg_addr_32b+1;
          end
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = erase_addr_32b;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Program the value to initiate block erase type 1 transaction
          addr_32b       = 0;
          addr_32b[10:0] = BLK_ERASE_TYPE1;
          data_32b       = 32'h0000_0001;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i == 1) begin
               read_apb(addr_32b);
		  end
          //Call the chkr_spi to check Flash model receives the opcode successfully
          chkr_spi(SPI_BLK1_ERS,0,erase_addr_32b);
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word read transaction & Initiate the transaction through AHB
            ahb_intf_en_b   = 1;
            size_3b         = 0;
            read_ahb(rd_addr_32b,size_3b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            if( S_AHB_DAT_WIDTH == 32) begin
              if( S_AHB_DAT_ENDIANESS == "little-endian")
                case (rd_addr_32b[1:0])
                  2'b00: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[7:0];
                  2'b01: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[15:8];
                  2'b10: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[23:16];
                  2'b11: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[31:24];
                endcase
              else //( S_AHB_DAT_ENDIANESS != "little-endian")
                case (rd_addr_32b[1:0])
                  2'b00: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[31:24];
                  2'b01: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[23:16];
                  2'b10: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[15:8];
                  2'b11: rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[7:0];
                endcase
            end
            else begin //if( S_AHB_DAT_WIDTH != 32)
			  //Store AHB read data
              rd_data_2d_8b[m_j_i] = ahb_rd_data_32b[7:0];
			end 
            //Call the chkr_spi to check Flash model receives all the commands successfully
            byte_addr_32b        = rd_addr_32b;
            ahb_checker(1,byte_addr_32b,rd_data_2d_8b[m_j_i]);
            rd_addr_32b          = rd_addr_32b+1;
          end
          //Checking default data after erase command
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            if (rd_data_2d_8b[m_j_i] != 8'hFF) begin
              process_err();
              $write("ERROR :: @%0dns %m() :: Read Data expected after Erase 8'hFF,Got 0x%x\n",$time,rd_data_2d_8b[m_j_i]);
            end
          end
        end else begin
          if( S_AHB_DAT_WIDTH == 32) begin
            addr_32b[31:12] = 20'h00000;
            addr_32b[11:2]  = $random;
            addr_32b[1:0]   = 2'h0;
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              prg_data_2d_32b[m_j_i][31:0]  = $random;
              ahb_intf_en_b                 = 1;
              size_3b                       = 2;
              //Initiate the transaction through AHB
              write_ahb(prg_addr_32b,prg_data_2d_32b[m_j_i],size_3b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
		      end
              //Call the chkr_spi to check Flash model receives all the commands successfully
              ahb_checker(0,prg_addr_32b,prg_data_2d_32b[m_j_i]);
			  //Increment AHB write address
              prg_addr_32b    = prg_addr_32b+4;
            end
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through AHB
              ahb_intf_en_b   = 1;
              size_3b         = 2;
              read_ahb(rd_addr_32b,size_3b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
		      end
              rd_data_2d_32b[m_j_i] = ahb_rd_data_32b;
              //Call the chkr_spi to check Flash model receives all the commands successfully
              ahb_checker(1,rd_addr_32b,prg_data_2d_32b[m_j_i]);
              ahb_intf_en_b   = 0;
			  //Increment AHB read address
              rd_addr_32b     = rd_addr_32b+4;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i]);
            end
          end
          addr_32b[31:12] = 20'h00000;
          addr_32b[11:2]  = $random;
          addr_32b[11:2]  = addr_32b[11:2] < 256 ? addr_32b[11:2]+256:addr_32b[11:2];
          addr_32b[1:0]   = 2'h0;
          prg_addr_32b    = addr_32b-64;
          rd_addr_32b     = addr_32b-64;
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word write transaction
            prg_data_2d_8b[m_j_i]  = $random;
            ahb_intf_en_b          = 1;
            size_3b                = 0;
            //Initiate the transaction through AHB
            write_ahb(prg_addr_32b,prg_data_2d_8b[m_j_i],size_3b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            byte_addr_32b      = prg_addr_32b;
            //Call the chkr_spi to check Flash model receives all the commands successfully
            ahb_checker(0,byte_addr_32b,prg_data_2d_8b[m_j_i]);
			//Wait for the transaction to finish
            prg_addr_32b    = prg_addr_32b+1;
          end
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word read transaction & Initiate the transaction through AHB
            ahb_intf_en_b   = 1;
            size_3b         = 0;
            read_ahb(rd_addr_32b,size_3b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			//Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
		    end
            rd_data_2d_8b[m_j_i] = ahb_rd_data_32b;
            //Call the chkr_spi to check Flash model receives all the commands successfully
            byte_addr_32b      = rd_addr_32b;
            ahb_checker(1,byte_addr_32b,rd_data_2d_8b[m_j_i]);
            ahb_intf_en_b   = 0;
			//Increment AHB read address
            rd_addr_32b     = rd_addr_32b+1;
          end
          //compare the data for write and read
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            do_data_compare(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i]);
          end
        end 
      end 
    end
  endtask
  
  
  // User-Defined Command Register
  task test_basic_user_defined_command;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] erase_addr_32b;
    reg [31:0] byte_addr_32b;
    reg [63:0] prg_data_2d_64b [255:0];
    reg [63:0] rd_data_2d_64b  [255:0];
    reg [31:0] prg_data_2d_32b [255:0];
    reg [31:0] rd_data_2d_32b  [255:0];
    reg [7:0]  prg_data_2d_8b  [255:0];
    reg [7:0]  rd_data_2d_8b   [255:0];
    reg [2:0]  size_3b;
    reg [31:0] data_32b;
    reg [31:0] rd_data_byte;
    integer m_i_i;
    integer m_j_i;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        if (C_PORT_ENABLE == 1) begin
          if(C_APB_DAT_WIDTH == 32) begin
            //Set the flash addressing mode to 32-bit mode
            addr_32b       = 0;
            addr_32b[10:0] = SPI_ADDR_MODE;
            data_32b       = 0;
            data_32b[7:0]  = 8'h01;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
          end
          //4-byte APB Data Width
          if(C_APB_DAT_WIDTH == 32) begin
		    // APB Write and Read : 4-byte data
            addr_32b[31:12] = 20'h00000;
            addr_32b[11:2]  = $random;
            addr_32b[1:0]   = 2'h0;
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction using User-defined command register
              prg_data_2d_32b[m_j_i]  = $random;
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h02;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the command address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              data_32b       = prg_addr_32b;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
              data_32b       = {24'h0, 4'h0, 4'h4}; //Write 4-byte data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h1};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Write Data0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_WR_DATA0;
              data_32b       = prg_data_2d_32b[m_j_i];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_PAGE_PRG,0,prg_addr_32b);
			  //Increment user-defined write address
              prg_addr_32b    = prg_addr_32b+4;
            end
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through APB
		      //Using user-defined command registers
		      //Set the flash address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              //Initiate the transaction through APB
              write_apb(addr_32b,rd_addr_32b);		    
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h03;   // Read Data
              //data_32b       = 32'h0B;   // Fast Read
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
			  //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
			  data_32b       = {24'h0, 4'h0, 4'h4}; // Read 4-bytes of data, without dummy
			  //data_32b       = {24'h0, 4'h0, 4'h4}; // Read 4-bytes of data, with dummy of 1 bytes (7 -> 8 bytes)
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h0};// Dummy Disabled
              //data_32b       = {29'h0, 1'h1, 1'h1, 1'h0}; // Dummy Enabled
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_READ,0,rd_addr_32b);
		      //Set the register for Read Data 0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_RD_DATA0;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store user-defined read data
              rd_data_2d_32b[m_j_i] = apb_prdata_i;
			  //Increment user-defined read address
              rd_addr_32b     = rd_addr_32b+4;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i]);
            end
            //Set the flash addressing mode to 32-bit mode
            addr_32b       = 0;
            addr_32b[10:0] = SPI_ADDR_MODE;
            data_32b       = 0;
            data_32b[7:0]  = 8'h01;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    // APB Write and Read : 8-byte data
            addr_32b[31:12] = 20'h00000;
            addr_32b[11:2]  = $random;
            addr_32b[1:0]   = 2'h0;
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              prg_data_2d_64b[m_j_i][31:0]   = $random;
              prg_data_2d_64b[m_j_i][63:32]  = $random;
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h02;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the command address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              data_32b       = prg_addr_32b;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
              data_32b       = {24'h0, 4'h0, 4'h8}; //Write 8-byte data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h1};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Write Data0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_WR_DATA0;
              data_32b       = prg_data_2d_64b[m_j_i][31:0];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
              //Set Write Data1
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_WR_DATA1;
              data_32b       = prg_data_2d_64b[m_j_i][63:32];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_PAGE_PRG,0,prg_addr_32b);
			  //Increment user-defined write address
              prg_addr_32b    = prg_addr_32b+8;
            end
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through APB
		      //Using user-defined command registers
		      //Set the flash address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              //Initiate the transaction through APB
              write_apb(addr_32b,rd_addr_32b);		    
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h03;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
			  //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
			  data_32b       = {24'h0, 4'h0, 4'h8}; // Read 8-bytes of data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h0};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_READ,0,rd_addr_32b);
		      //Set the register for Read Data 0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_RD_DATA0;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store user-defined read data
              rd_data_2d_64b[m_j_i][31:0] = apb_prdata_i;
		      //Set the register for Read Data 1
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_RD_DATA1;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store user-defined read data
              rd_data_2d_64b[m_j_i][63:32] = apb_prdata_i;
			  //Increment user-defined read address
              rd_addr_32b     = rd_addr_32b+8;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_64b[m_j_i][31:0],rd_data_2d_64b[m_j_i][31:0]);
              do_data_compare(prg_data_2d_64b[m_j_i][63:32],rd_data_2d_64b[m_j_i][63:32]);
            end
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            erase_addr_32b  = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction using user-defined command register
              prg_data_2d_32b[m_j_i][31:0]  = $random;
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h02;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the command address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              data_32b       = prg_addr_32b;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
              data_32b       = {24'h0, 4'h0, 4'h4}; //Write 4-byte data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h1};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Write Data0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_WR_DATA0;
              data_32b       = prg_data_2d_32b[m_j_i][31:0];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_PAGE_PRG,0,prg_addr_32b);
			  //Increment user-defined write address
              prg_addr_32b    = prg_addr_32b+4;
            end
            //Program the value to initiate block erase type 1 transaction using user-defined command registers
		    //Set the command opcode
            addr_32b       = 0;
            addr_32b[10:0] = USER_CMD_OPCODE;
            data_32b       = 32'h20;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    //Set the command address
            addr_32b       = 0;
            addr_32b[10:0] = USER_CMD_ADDR;
            data_32b       = erase_addr_32b;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    //Set Data and Dummy Length
            addr_32b       = 0;
            addr_32b[10:0] = USER_CMD_DATA_DUMMY;
            data_32b       = {24'h0, 4'h0, 4'h0}; // Erase data
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    //Set Configurations
            addr_32b       = 0;
            addr_32b[10:0] = USER_CMD_CFG;
            data_32b       = {29'h0, 1'h0, 1'h1, 1'h1};
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    //Set Start
            addr_32b       = 0;
            addr_32b[10:0] = USER_CMD_START;
            data_32b       = 32'h1;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = TRANS_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
			  //Wait for the transaction to finish
            while (apb_prdata_i == 1) begin
                 read_apb(addr_32b);
            end
            chkr_spi(SPI_BLK1_ERS,0,erase_addr_32b);
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through APB
		      //Using user-defined command registers
		      //Set the flash address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              //Initiate the transaction through APB
              write_apb(addr_32b,rd_addr_32b);		    
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h03;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
			  //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
			  data_32b       = {24'h0, 4'h0, 4'h4}; // Read 4-bytes of data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h0};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_READ,0,rd_addr_32b);
		      //Set the register for Read Data 0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_RD_DATA0;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store user-defined read data
              rd_data_2d_32b[m_j_i] = apb_prdata_i;
			  //Increment user-defined read address
              rd_addr_32b     = rd_addr_32b+4;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              if (rd_data_2d_32b[m_j_i] != 32'hFFFFFFFF) begin
                process_err();
                $write("ERROR :: @%0dns %m() :: Read Data expected after Erase 32'hFFFFFF,Got 0x%x\n",$time,rd_data_2d_32b[m_j_i]);
              end
            end
            //Set the flash addressing mode to 32-bit mode
            addr_32b       = 0;
            addr_32b[10:0] = SPI_ADDR_MODE;
            data_32b       = 0;
            data_32b[7:0]  = 8'h01;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
		    // APB Write and Read : 1-byte data
            addr_32b[31:12] = 20'h00000;
            addr_32b[11:2]  = $random;
            addr_32b[1:0]   = 2'h0;
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction using User-defined command register
              prg_data_2d_32b[m_j_i]  = $random;
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h02;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the command address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              data_32b       = prg_addr_32b;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
              data_32b       = {24'h0, 4'h0, 4'h1}; //Write 1-byte data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h1};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Write Data0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_WR_DATA0;
              data_32b       = prg_data_2d_32b[m_j_i][7:0];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_PAGE_PRG,0,prg_addr_32b);
			  //Increment user-defined write address
              prg_addr_32b    = prg_addr_32b+1;
            end
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through APB
		      //Using user-defined command registers
		      //Set the flash address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              //Initiate the transaction through APB
              write_apb(addr_32b,rd_addr_32b);		    
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h03;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
			  //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
			  data_32b       = {24'h0, 4'h0, 4'h1}; // Read 1-byte of data
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h0};
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_READ,0,rd_addr_32b);
		      //Set the register for Read Data 0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_RD_DATA0;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store user-defined read data
              rd_data_2d_8b[m_j_i] = apb_prdata_i;
			  //Increment user-defined read address
              rd_addr_32b     = rd_addr_32b+1;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_32b[m_j_i][7:0],rd_data_2d_8b[m_j_i]);
            end
          end 
          else begin
            //Set the flash addressing mode to 32-bit mode
            addr_32b       = 0;
            addr_32b[10:0] = SPI_ADDR_MODE;
            data_32b       = 0;
            data_32b[7:0]  = 8'h0;
		    // APB Write and Read : 1-byte data
            addr_32b[31:12] = 20'h00000;
            addr_32b[11:2]  = $random;
            addr_32b[11:2]  = addr_32b[11:2] < 256 ? addr_32b[11:2]+256:addr_32b[11:2];
            addr_32b[1:0]   = 2'h0;
            prg_addr_32b    = addr_32b-64;
            rd_addr_32b     = addr_32b-64;
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction using User-defined command register
              prg_data_2d_32b[m_j_i]  = $random;
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h02;
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set the command address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              data_32b       = prg_addr_32b[7:0];
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
              data_32b       = {24'h0, 4'h0, 4'h1}; //Write 1-byte data
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h1};
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set Write Data0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_WR_DATA0;
              data_32b       = prg_data_2d_32b[m_j_i][7:0];
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_PAGE_PRG,0,prg_addr_32b[7:0]);
			  //Increment user-defined write address
              prg_addr_32b    = prg_addr_32b+1;
            end
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through APB
		      //Using user-defined command registers
		      //Set the flash address
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_ADDR;
              data_32b       = rd_addr_32b[7:0];
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);		    
		      //Set the command opcode
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_OPCODE;
              data_32b       = 32'h03;
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
			  //Set Data and Dummy Length
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_DATA_DUMMY;
			  data_32b       = {24'h0, 4'h0, 4'h1}; // Read 1-byte of data
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set Configurations
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_CFG;
              data_32b       = {29'h0, 1'h0, 1'h1, 1'h0};
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set Start
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_START;
              data_32b       = 32'h1;
              //Initiate the transaction through APB
              write_apb_8b(addr_32b,data_32b);
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Wait for the transaction to finish
              while (apb_prdata_i == 1) begin
                   read_apb(addr_32b);
              end
			  chkr_spi(SPI_READ,0,rd_addr_32b[7:0]);
		      //Set the register for Read Data 0
              addr_32b       = 0;
              addr_32b[10:0] = USER_CMD_RD_DATA0;
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store user-defined read data
              rd_data_2d_8b[m_j_i] = apb_prdata_i;
			  //Increment user-defined read address
              rd_addr_32b     = rd_addr_32b+1;
            end
            //compare the data for write and read
            for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_32b[m_j_i][7:0],rd_data_2d_8b[m_j_i][7:0]);
            end
		  end
        end
      end 
    end
  endtask

  task test_page;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] prg_data_2d_32b [1023:0];
    reg [31:0] rd_data_2d_32b  [1023:0];
    reg [7:0]  prg_data_2d_8b  [4095:0];
    reg [7:0]  rd_data_2d_8b   [4095:0];
    reg [31:0] data_32b;
	reg [9:0]  prev_buf_in_cnt;
	reg [9:0]  cur_buf_in_cnt;
	reg [10:0] prev_rd_addr;
    integer m_i_i;
    integer m_j_i;
    integer m_k_i;
	integer loop_limit;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
	  //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = TRANS_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
	  //Wait for the transaction to finish
      while (apb_prdata_i == 1) begin
           read_apb(addr_32b);
      end
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        //Set the address
        addr_32b        = 0;
        addr_32b[13:10] = $random;
        addr_32b[9:0]   = 9'h000;
        prg_addr_32b    = addr_32b;
        rd_addr_32b     = addr_32b;
        for (m_k_i = 0; m_k_i < PAGE_PRG_LOOP; m_k_i = m_k_i + 1) begin
          prg_addr_32b   = prg_addr_32b + m_k_i*PAGE_PROGRAM_SIZE;
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = prg_addr_32b;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the flash addressing mode to 24-bit or 32-bit mode
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_MODE;
          data_32b       = 0;
          data_32b[7:0]  = (S_AHB_ADR_WIDTH == 32) ? 8'h01 : 8'h00;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the page program length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Generate word data
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG_BUF;
          if (C_APB_DAT_WIDTH == 32) begin
            for (m_j_i = 0; m_j_i < (PAGE_PROGRAM_SIZE>>2); m_j_i = m_j_i + 1) begin
              prg_data_2d_32b[addr_32b[8:2]][31:0]  = $random;
              //Initiate the transaction through APB
              write_apb(addr_32b,prg_data_2d_32b[addr_32b[8:2]]);
			  //Increment APB write address
              addr_32b     = addr_32b+4;
            end
          end
          else begin
            for (m_j_i = 0; m_j_i < (PAGE_PROGRAM_SIZE>>2); m_j_i = m_j_i + 1) begin
              data_32b = $random;
              prg_data_2d_8b[addr_32b[8:0]  ][7:0]  = data_32b[ 7: 0];
              prg_data_2d_8b[addr_32b[8:0]+1][7:0]  = data_32b[15: 8];
              prg_data_2d_8b[addr_32b[8:0]+2][7:0]  = data_32b[23:16];
              prg_data_2d_8b[addr_32b[8:0]+3][7:0]  = data_32b[31:24];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
			  //Increment APB write address
              addr_32b     = addr_32b+4;
            end
          end
          //address limit check
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            data_32b  = $random;
            //Initiate the transaction through APB
            //write_apb(addr_32b,data_32b);
            write_only_apb(addr_32b,data_32b);
			//Increment APB write address
            addr_32b     = addr_32b+4;
          end
          //Run the page program in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i[0] == 1) begin
               read_apb(addr_32b);
          end
        end
		// Access Page Read Buffer Registers once all data are written on the buffer
        for (m_k_i = 0; m_k_i < PAGE_RD_LOOP; m_k_i = m_k_i + 1) begin
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = rd_addr_32b + m_k_i*PAGE_READ_SIZE;
          //Set the page read length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Run the page read in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i == 1) begin
              read_apb(addr_32b);
          end
          //Read word data
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD_BUF;
          if (C_APB_DAT_WIDTH == 32) begin
            for (m_j_i = 0; m_j_i < (PAGE_READ_SIZE>>2); m_j_i = m_j_i + 1) begin
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store APB read data
              rd_data_2d_32b[addr_32b[8:2]] = apb_rd_data;
			  //Increment APB read address
              addr_32b                      = addr_32b+4;
            end
            //address limit check
            for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Increment APB read address
              addr_32b                      = addr_32b+4;
            end
          end
          else begin
            for (m_j_i = 0; m_j_i < PAGE_READ_SIZE; m_j_i = m_j_i + 1) begin
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Store APB read data
              rd_data_2d_8b[addr_32b[8:0]]  = apb_rd_data[7:0];
			  //Increment APB read address
              addr_32b                      = addr_32b+1;
            end
            //address limit check
            for (m_j_i = 0; m_j_i < 256; m_j_i = m_j_i + 1) begin
              //Initiate the transaction through APB
              read_apb(addr_32b);
			  //Increment APB read address
              addr_32b                      = addr_32b+1;
            end
          end
          //compare the data for write and read
          if (C_APB_DAT_WIDTH == 32) begin
            for (m_j_i = 0; m_j_i < (PAGE_READ_SIZE>>2); m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i]);
            end
          end
          else begin
            for (m_j_i = 0; m_j_i < PAGE_READ_SIZE; m_j_i = m_j_i + 1) begin
              do_data_compare(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i]);
            end
          end
		end
      end
    end
  endtask

  task test_page_with_rd_buf_stat_addr;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] prg_data_2d_32b [1023:0];
    reg [31:0] rd_data_2d_32b  [1023:0];
    reg [7:0]  prg_data_2d_8b  [4095:0];
    reg [7:0]  rd_data_2d_8b   [4095:0];
    reg [31:0] data_32b;
	reg [9:0]  prev_buf_in_cnt;
	reg [9:0]  cur_buf_in_cnt;
	reg [10:0] prev_rd_addr;
	reg [10:0] done_rd_addr_cntr;
	reg [31:0] buf_cntr;
	reg        buf_stat;
	reg [10:0] prev_addr;
	reg [31:0] rd_data_cntr;
    integer m_i_i;
    integer m_j_i;
    integer m_k_i;
	integer loop_limit;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
	  //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = TRANS_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
	  //Wait for the transaction to finish
      while (apb_prdata_i == 1) begin
           read_apb(addr_32b);
      end
	  //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = RD_BUFF_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
      //W1 to clear on Read Buffer Status Register
      addr_32b       = 0;
      addr_32b[10:0] = RD_BUFF_STAT_REG;
      data_32b       = 0;
      data_32b[31:0] = 32'hFFFFFFFF;
      //Initiate the transaction through APB
      write_apb(addr_32b,data_32b);
	  //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = RD_BUFF_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        //Set the address
        addr_32b        = 0;
        addr_32b[13:10] = $random;
        addr_32b[9:0]   = 9'h000;
        prg_addr_32b    = addr_32b;
        rd_addr_32b     = addr_32b;
        for (m_k_i = 0; m_k_i < PAGE_PRG_LOOP; m_k_i = m_k_i + 1) begin
          prg_addr_32b   = prg_addr_32b + m_k_i*PAGE_PROGRAM_SIZE;
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = prg_addr_32b;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the flash addressing mode to 24-bit or 32-bit mode
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_MODE;
          data_32b       = 0;
          data_32b[7:0]  = (S_AHB_ADR_WIDTH == 32) ? 8'h01 : 8'h00;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the page program length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Generate word data
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG_BUF;
          if (C_APB_DAT_WIDTH == 32) begin
            for (m_j_i = 0; m_j_i < (PAGE_PROGRAM_SIZE>>2); m_j_i = m_j_i + 1) begin
              prg_data_2d_32b[addr_32b[8:2]][31:0]  = $random;
              //Initiate the transaction through APB
              write_apb(addr_32b,prg_data_2d_32b[addr_32b[8:2]]);
			  //Increment APB write address
              addr_32b     = addr_32b+4;
            end
          end
          else begin
            for (m_j_i = 0; m_j_i < (PAGE_PROGRAM_SIZE>>2); m_j_i = m_j_i + 1) begin
              data_32b = $random;
              prg_data_2d_8b[addr_32b[8:0]  ][7:0]  = data_32b[ 7: 0];
              prg_data_2d_8b[addr_32b[8:0]+1][7:0]  = data_32b[15: 8];
              prg_data_2d_8b[addr_32b[8:0]+2][7:0]  = data_32b[23:16];
              prg_data_2d_8b[addr_32b[8:0]+3][7:0]  = data_32b[31:24];
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
			  //Increment APB write address
              addr_32b     = addr_32b+4;
            end
          end
          //address limit check
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            data_32b  = $random;
            //Initiate the transaction through APB
            //write_apb(addr_32b,data_32b);
            write_only_apb(addr_32b,data_32b);
			//Increment APB write address
            addr_32b     = addr_32b+4;
          end
          //Run the page program in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i[0] == 1) begin
               read_apb(addr_32b);
          end
        end
		// Access Page Read Buffer Registers once there is available data on the buffer
        for (m_k_i = 0; m_k_i < PAGE_RD_LOOP; m_k_i = m_k_i + 1) begin
		  prev_buf_in_cnt = 0;
		  prev_rd_addr   = 11'h400;
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = rd_addr_32b + m_k_i*PAGE_READ_SIZE;
          //Set the page read length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Run the page read in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  rd_data_cntr = 0;
          if (C_APB_DAT_WIDTH == 32) begin
		    if (apb_prdata_i == 1) begin
				while (rd_data_cntr != PAGE_PROGRAM_SIZE) begin
		          //Set the register for Read Transaction Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
				  //Wait for the buffer to be not empty with content counter not less than 1
			      while (apb_prdata_i[0] == 0 & apb_prdata_i[18:8] < 11'h1) begin
                      read_apb(addr_32b);
			      end
				  //Get the buffer content value 
			      cur_buf_in_cnt = apb_prdata_i[18:8];
				  //Wait for the buffer content value to increment
			      if (cur_buf_in_cnt > prev_buf_in_cnt) begin
                    //Read word data
                    addr_32b       = 0;
                    addr_32b[10:0] = prev_rd_addr;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
			        //Store APB read data
                    rd_data_2d_32b[addr_32b[8:2]] = apb_rd_data;
		            //Set the register for Read Buffer Address Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_ADDR_REG;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
			        //Increment APB read address
			    	prev_rd_addr = apb_rd_data + 11'h4;
			        //Increment read data counter
					rd_data_cntr = rd_data_cntr + 4;
			      end
				  //Store current buffer content value to previous value
			      prev_buf_in_cnt = cur_buf_in_cnt;
				end
                //W1 to clear on Read Buffer Status Register
                addr_32b       = 0;
                addr_32b[10:0] = RD_BUFF_STAT_REG;
                data_32b       = 0;
                data_32b[31:0] = 32'hFFFFFFFF;
                //Initiate the transaction through APB
                write_apb(addr_32b,data_32b);
		    end
		    else begin
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
		    end
		  end
		  else begin
			buf_stat = 1;
			buf_cntr = 32'h0;
			prev_addr = 11'h0;
		    if (apb_prdata_i == 1) begin
			  while (rd_data_cntr != PAGE_PROGRAM_SIZE) begin
	            //Set the register for Read Transaction Status Register
                addr_32b       = 0;
                addr_32b[10:0] = RD_BUFF_STAT_REG;
                //Initiate the transaction through APB
                read_apb(addr_32b);
				//Get buffer status
	            buf_stat       = apb_prdata_i[0];
	            //Set the register for Read Transaction Status Register
                addr_32b       = 0;
                addr_32b[10:0] = RD_BUFF_STAT_REG_1;
                //Initiate the transaction through APB
                read_apb(addr_32b);
				//Get buffer counter
	            buf_cntr[7:0] =  apb_prdata_i;
	            //Set the register for Read Transaction Status Register
                addr_32b       = 0;
                addr_32b[10:0] = RD_BUFF_STAT_REG_2;
                //Initiate the transaction through APB
                read_apb(addr_32b);
				//Get buffer counter
	            buf_cntr[10:8] =  apb_prdata_i[2:0];
				//Wait for the buffer to be not empty with content counter not less than 1
	            while (buf_stat == 0 & buf_cntr < 1) begin
	              //Set the register for Read Transaction Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
	              buf_stat       = apb_prdata_i[0];
	              //Set the register for Read Transaction Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG_1;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
	              buf_cntr[7:0] =  apb_prdata_i;
	              //Set the register for Read Transaction Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG_2;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
	              buf_cntr[10:8] =  apb_prdata_i[2:0];
	            end
				//Get the buffer content value
	            cur_buf_in_cnt = buf_cntr;
				//Wait for the buffer content value to increment
	            if (cur_buf_in_cnt > prev_buf_in_cnt) begin
                  //Read word data
                  addr_32b       = 0;
                  addr_32b[10:0] = prev_rd_addr;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
                  rd_data_2d_8b[addr_32b[8:0]] = apb_rd_data[7:0];
	              //Set the register for Read Buffer Address Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_ADDR_REG;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
				  //Get the previous address read
		          prev_addr[7:0] = apb_rd_data[7:0];
	              //Set the register for Read Buffer Address Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_ADDR_REG_1;
                  //Initiate the transaction through APB
                  read_apb(addr_32b);
				  //Get the previous address read
		          prev_addr[10:8] = apb_rd_data[2:0];
			      //Increment APB read address
	 		      prev_rd_addr = prev_addr + 11'h1;					
			      //Increment read data counter
		          rd_data_cntr = rd_data_cntr + 1;
	            end
				//Store current buffer content value to previous value
	            prev_buf_in_cnt = cur_buf_in_cnt;
		      end
              //W1 to clear on Read Buffer Status Register
              addr_32b       = 0;
              addr_32b[10:0] = RD_BUFF_STAT_REG;
              data_32b       = 0;
              data_32b[31:0] = 32'hFFFFFFFF;
              //Initiate the transaction through APB
              write_apb(addr_32b,data_32b);
		    end
		    else begin
		      //Set the register for Read Transaction Status Register
              addr_32b       = 0;
              addr_32b[10:0] = TRANS_STAT_REG;
              //Initiate the transaction through APB
              read_apb(addr_32b);
		    end
          end		  
		end
        //compare the data for write and read
        if (C_APB_DAT_WIDTH == 32) begin
          for (m_j_i = 0; m_j_i < (PAGE_READ_SIZE>>2); m_j_i = m_j_i + 1) begin
            do_data_compare(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i]);
          end
        end
        else begin
          for (m_j_i = 0; m_j_i < PAGE_READ_SIZE; m_j_i = m_j_i + 1) begin
            do_data_compare(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i]);
          end
        end
      end
    end
  endtask
  
  task test_page_ahb;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] prg_data_2d_32b [1023:0];
    reg [31:0] rd_data_2d_32b  [1023:0];
    reg [7:0]  prg_data_2d_8b  [4095:0];
    reg [7:0]  rd_data_2d_8b   [4095:0];
    reg [31:0] data_32b;
    integer m_i_i;
    integer m_j_i;
    integer m_k_i;
    integer ptr;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
	  //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = TRANS_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
	  //Wait for the transaction to finish
      while (apb_prdata_i == 1) begin
           read_apb(addr_32b);
      end
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        //Enable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h01;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Set the address
        addr_32b        = 0;
        addr_32b[13:10] = $random;
        addr_32b[9:0]   = 9'h000;
        prg_addr_32b    = addr_32b;
        rd_addr_32b     = addr_32b;
        //Write word data
        for (m_k_i = 0; m_k_i < PAGE_PRG_LOOP; m_k_i = m_k_i + 1) begin
          if (S_AHB_DAT_WIDTH == 32)
            for (m_j_i = 0; m_j_i < (PAGE_PROGRAM_SIZE>>2); m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              ptr                           = m_k_i*(PAGE_PROGRAM_SIZE>>2) + m_j_i;
              prg_data_2d_32b[ptr][31:0]    = $random^(m_k_i+1);
              ahb_intf_en_b                 = 1;
              size_3b                       = 2;
              //Initiate the transaction through AHB
              write_ahb(addr_32b,prg_data_2d_32b[ptr],size_3b);
			  //Increment AHB write address
              addr_32b    = addr_32b+4;
            end
          else
            for (m_j_i = 0; m_j_i < PAGE_PROGRAM_SIZE; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              ptr                           = m_k_i*PAGE_PROGRAM_SIZE + m_j_i;
              prg_data_2d_8b[ptr][7:0]      = $random^(m_k_i+1);
              ahb_intf_en_b                 = 1;
              size_3b                       = 0;
              //Initiate the transaction through AHB
              write_ahb(addr_32b,prg_data_2d_8b[ptr],size_3b);
			  //Increment AHB write address
              addr_32b    = addr_32b+1;
            end
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = prg_addr_32b + m_k_i*PAGE_PROGRAM_SIZE;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the flash addressing mode to 24-bit or 32-bit mode
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_MODE;
          data_32b       = 0;
          data_32b[7:0]  = (S_AHB_ADR_WIDTH == 32) ? 8'h01 : 8'h00;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the page program length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Run the page program in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i == 1) begin
               read_apb(addr_32b);
          end
        end
        for (m_k_i = 0; m_k_i < PAGE_RD_LOOP; m_k_i = m_k_i + 1) begin
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = rd_addr_32b + m_k_i*PAGE_READ_SIZE;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the page read length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_READ_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Run the page read in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
	      //Wait for the transaction to finish
          while (apb_prdata_i == 1) begin
               read_apb(addr_32b);
          end
          //Read word data
          if (S_AHB_DAT_WIDTH == 32)
            for (m_j_i = 0; m_j_i < (PAGE_READ_SIZE>>2); m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through AHB
              ahb_intf_en_b   = 1;
              size_3b         = 2;
              read_ahb(rd_addr_32b,size_3b);
              ptr             = m_j_i + m_k_i*(PAGE_READ_SIZE>>2);
              rd_data_2d_32b[ptr] = ahb_rd_data_32b;
              ahb_intf_en_b   = 0;
			  //Increment AHB read address
              rd_addr_32b     = rd_addr_32b+4;
            end
          else
            for (m_j_i = 0; m_j_i < PAGE_READ_SIZE; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word read transaction & Initiate the transaction through AHB
              ahb_intf_en_b   = 1;
              size_3b         = 0;
              read_ahb(rd_addr_32b,size_3b);
              ptr             = m_j_i + m_k_i*PAGE_READ_SIZE;
              rd_data_2d_8b[ptr] = ahb_rd_data_32b[7:0];
              ahb_intf_en_b   = 0;
			  //Increment AHB read address
              rd_addr_32b     = rd_addr_32b+1;
            end
        end
        //compare the data for write and read
        if (S_AHB_DAT_WIDTH == 32)
          for (m_j_i = 0; m_j_i < (PAGE_COMPARE>>2); m_j_i = m_j_i + 1) begin
            do_data_compare2(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i],m_j_i);
          end
        else
          for (m_j_i = 0; m_j_i < PAGE_COMPARE; m_j_i = m_j_i + 1) begin
            do_data_compare2(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i],m_j_i);
          end
        //Disable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h00;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
      end
    end
  endtask
  
  task test_page_ahb_with_rd_buf_stat_addr;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] prg_data_2d_32b [1023:0];
    reg [31:0] rd_data_2d_32b  [1023:0];
    reg [7:0]  prg_data_2d_8b  [4095:0];
    reg [7:0]  rd_data_2d_8b   [4095:0];
    reg [31:0] data_32b;
	reg [9:0]  prev_buf_in_cnt;
	reg [9:0]  cur_buf_in_cnt;
	reg [31:0] prev_rd_addr;
	reg [10:0] done_rd_addr_cntr;
	reg [31:0] rd_data_cntr;
	reg [31:0] buf_cntr;
	reg        buf_stat;
	reg [31:0] prev_addr;
    integer m_i_i;
    integer m_j_i;
    integer m_k_i;
    integer ptr;
	integer loop_limit;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
	  //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = TRANS_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
      //Wait for the transaction to finish
      while (apb_prdata_i == 1) begin
           read_apb(addr_32b);
      end
	  if (C_APB_DAT_WIDTH == 32) begin
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG;
        //Initiate the transaction through APB
        read_apb(addr_32b);
        //W1 to clear on Read Buffer Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG;
        data_32b       = 0;
        data_32b[31:0] = 32'hFFFFFFFF;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	  end
	  else begin
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG_1;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG_2;
        //Initiate the transaction through APB
        read_apb(addr_32b);
        //W1 to clear on Read Buffer Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG;
        data_32b       = 0;
        data_32b[31:0] = 32'hFFFFFFFF;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG_1;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	    //Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = RD_BUFF_STAT_REG_2;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	  end
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        //Enable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h01;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Set the address
        addr_32b        = 0;
        addr_32b[13:10] = $random;
        addr_32b[9:0]   = 9'h000;
        prg_addr_32b    = addr_32b;
        rd_addr_32b     = addr_32b;
		prev_rd_addr    = addr_32b; 
        //Write word data
        for (m_k_i = 0; m_k_i < PAGE_PRG_LOOP; m_k_i = m_k_i + 1) begin
          if (S_AHB_DAT_WIDTH == 32)
            for (m_j_i = 0; m_j_i < (PAGE_PROGRAM_SIZE>>2); m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              ptr                           = m_k_i*(PAGE_PROGRAM_SIZE>>2) + m_j_i;
              prg_data_2d_32b[ptr][31:0]    = $random^(m_k_i+1);
              ahb_intf_en_b                 = 1;
              size_3b                       = 2;
              //Initiate the transaction through AHB
              write_ahb(addr_32b,prg_data_2d_32b[ptr],size_3b);
			  //Increment AHB write address
              addr_32b    = addr_32b+4;
            end
          else
            for (m_j_i = 0; m_j_i < PAGE_PROGRAM_SIZE; m_j_i = m_j_i + 1) begin
              //Program the value to initiate word write transaction
              ptr                           = m_k_i*PAGE_PROGRAM_SIZE + m_j_i;
              prg_data_2d_8b[ptr][7:0]      = $random^(m_k_i+1);
              ahb_intf_en_b                 = 1;
              size_3b                       = 0;
              //Initiate the transaction through AHB
              write_ahb(addr_32b,prg_data_2d_8b[ptr],size_3b);
			  //Increment AHB write address
              addr_32b    = addr_32b+1;
            end
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = prg_addr_32b + m_k_i*PAGE_PROGRAM_SIZE;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the flash addressing mode to 24-bit or 32-bit mode
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_MODE;
          data_32b       = 0;
          data_32b[7:0]  = (S_AHB_ADR_WIDTH == 32) ? 8'h01 : 8'h00;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the page program length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Run the page program in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_PRG;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  //Wait for the transaction to finish
          while (apb_prdata_i == 1) begin
               read_apb(addr_32b);
          end
        end
        for (m_k_i = 0; m_k_i < PAGE_RD_LOOP; m_k_i = m_k_i + 1) begin
	      if (C_APB_DAT_WIDTH == 32) begin
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
            //W1 to clear on Read Buffer Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG;
            data_32b       = 0;
            data_32b[31:0] = 32'hFFFFFFFF;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
	      end
	      else begin
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG_1;
            //Initiate the transaction through APB
            read_apb(addr_32b);
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG_2;
            //Initiate the transaction through APB
            read_apb(addr_32b);
            //W1 to clear on Read Buffer Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG;
            data_32b       = 0;
            data_32b[31:0] = 32'hFFFFFFFF;
            //Initiate the transaction through APB
            write_apb(addr_32b,data_32b);
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG;
            //Initiate the transaction through APB
            read_apb(addr_32b);
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG_1;
            //Initiate the transaction through APB
            read_apb(addr_32b);
	        //Set the register for Read Transaction Status Register
            addr_32b       = 0;
            addr_32b[10:0] = RD_BUFF_STAT_REG_2;
            //Initiate the transaction through APB
            read_apb(addr_32b);
	      end
          //Set the spi flash address
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_BYTE0;
          data_32b       = rd_addr_32b + m_k_i*PAGE_READ_SIZE;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Set the page read length
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD_LEN_0;
          data_32b       = 0;
          data_32b[15:0] = PAGE_READ_SIZE[15:0];
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
          //Run the page read in SPI interface
          addr_32b       = 0;
          addr_32b[10:0] = PAGE_RD;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		  prev_buf_in_cnt = 0;
		  ptr = 0;
		  rd_data_cntr = 0;
		  if (apb_prdata_i == 1) begin
            if (S_AHB_DAT_WIDTH == 32) begin
			  if (C_APB_DAT_WIDTH == 32) begin
				  //Wait for read data counter value to be equal to PAGE_READ_SIZE
				  while (rd_data_cntr != PAGE_READ_SIZE) begin
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Wait for the buffer to be not empty with content counter not less than 1
			        while (apb_prdata_i[0] == 0 & apb_prdata_i[18:8] < 11'h1) begin
                        read_apb(addr_32b);
			        end
				    //Get the buffer content value
			        cur_buf_in_cnt = apb_prdata_i[18:8];
				    //Wait for the buffer content value to increment
			        if (cur_buf_in_cnt > prev_buf_in_cnt) begin
                        //Read word data
                        addr_32b       = 0;
                        addr_32b[10:0] = rd_addr_32b;
                        //Program the value to initiate word read transaction & Initiate the transaction through AHB
                        ahb_intf_en_b   = 1;
                        size_3b         = 2;
                        read_ahb(rd_addr_32b,size_3b);
                        rd_data_2d_32b[ptr] = ahb_rd_data_32b;
                        ptr             = ptr + 1;
                        ahb_intf_en_b   = 0;
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
			            //Increment APB read address 
			    	    rd_addr_32b = apb_rd_data + 11'h4;
			            //Increment read data counter
						rd_data_cntr = rd_data_cntr + 4;
			    	end
				    //Store current buffer content value to previous value
			    	prev_buf_in_cnt = cur_buf_in_cnt;
				  end
                  //W1 to clear on Read Buffer Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG;
                  data_32b       = 0;
                  data_32b[31:0] = 32'hFFFFFFFF;
                  //Initiate the transaction through APB
                  write_apb(addr_32b,data_32b);
			  end
			  else begin // C_APB_DAT_WIDTH == 8
			      buf_stat = 1;
			      buf_cntr = 32'h0;
			      prev_addr = 32'h0;
				  //Wait for read data counter value to be equal to PAGE_READ_SIZE
				  while (rd_data_cntr != PAGE_READ_SIZE) begin
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Get buffer status
					buf_stat       = apb_prdata_i[0];
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG_1;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Get buffer counter
					buf_cntr[7:0] =  apb_prdata_i;
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG_2;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Get buffer counter
					buf_cntr[10:8] =  apb_prdata_i[2:0];
				    //Wait for the buffer to be not empty with content counter not less than 1
			        while (buf_stat == 0 & buf_cntr < 1) begin
		               //Set the register for Read Transaction Status Register
                       addr_32b       = 0;
                       addr_32b[10:0] = RD_BUFF_STAT_REG;
                       //Initiate the transaction through APB
                       read_apb(addr_32b);
					   buf_stat       = apb_prdata_i[0];
		               //Set the register for Read Transaction Status Register
                       addr_32b       = 0;
                       addr_32b[10:0] = RD_BUFF_STAT_REG_1;
                       //Initiate the transaction through APB
                       read_apb(addr_32b);
				       //Get buffer counter
					   buf_cntr[7:0] =  apb_prdata_i;
		               //Set the register for Read Transaction Status Register
                       addr_32b       = 0;
                       addr_32b[10:0] = RD_BUFF_STAT_REG_2;
                       //Initiate the transaction through APB
                       read_apb(addr_32b);
				       //Get buffer counter
					   buf_cntr[10:8] =  apb_prdata_i[2:0];
			        end
				    //Get the buffer content value
			        cur_buf_in_cnt = buf_cntr;
				    //Wait for the buffer content value to increment
			        if (cur_buf_in_cnt > prev_buf_in_cnt) begin
                        //Read word data
                        addr_32b       = 0;
                        addr_32b[10:0] = rd_addr_32b;
                        //Program the value to initiate word read transaction & Initiate the transaction through AHB
                        ahb_intf_en_b   = 1;
                        size_3b         = 2;
                        read_ahb(rd_addr_32b,size_3b);
                        rd_data_2d_32b[ptr] = ahb_rd_data_32b;
                        ptr             = ptr + 1;
                        ahb_intf_en_b   = 0;
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[7:0] = apb_rd_data[7:0];
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG_1;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[15:8] = apb_rd_data;
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG_2;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[23:16] = apb_rd_data[7:0];
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG_3;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[31:24] = apb_rd_data;
			            //Increment APB read address
			    	    rd_addr_32b = prev_addr + 32'h4;
			            //Increment read data counter
						rd_data_cntr = rd_data_cntr + 4;
			    	end
				    //Store current buffer content value to previous value
			    	prev_buf_in_cnt = cur_buf_in_cnt;
				  end
                  //W1 to clear on Read Buffer Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG;
                  data_32b       = 0;
                  data_32b[31:0] = 32'hFFFFFFFF;
                  //Initiate the transaction through APB
                  write_apb(addr_32b,data_32b);
			  end
			end
            else begin // S_AHB_DAT_WIDTH == 8
			  if (C_APB_DAT_WIDTH == 32) begin
				  //Wait for read data counter value to be equal to PAGE_READ_SIZE
			      while (rd_data_cntr != PAGE_READ_SIZE) begin
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Wait for the buffer to be not empty with content counter not less than 1
			        while (apb_prdata_i[0] == 0 & apb_prdata_i[18:8] < 11'h1) begin
                        read_apb(addr_32b);
			        end
				    //Get the buffer content value
			        cur_buf_in_cnt = apb_prdata_i[18:8];
				    //Wait for the buffer content value to increment
			        if (cur_buf_in_cnt > prev_buf_in_cnt) begin
                        //Read word data
                        addr_32b       = 0;
                        addr_32b[10:0] = rd_addr_32b;
                        //Program the value to initiate word read transaction & Initiate the transaction through AHB
                        ahb_intf_en_b   = 1;
                        size_3b         = 0;
                        read_ahb(rd_addr_32b,size_3b);
                        rd_data_2d_8b[ptr] = ahb_rd_data_32b[7:0];
                        ptr             = ptr + 1;
                        ahb_intf_en_b   = 0;
			            //Increment read data counter
					    rd_data_cntr   = rd_data_cntr + 1;
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
			            //Increment APB read address
			    	    rd_addr_32b = apb_rd_data + 11'h1;
			    	end
				    //Store current buffer content value to previous value
			    	prev_buf_in_cnt = cur_buf_in_cnt;
				   end
                  //W1 to clear on Read Buffer Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG;
                  data_32b       = 0;
                  data_32b[31:0] = 32'hFFFFFFFF;
                  //Initiate the transaction through APB
                  write_apb(addr_32b,data_32b);
			  end
			  else begin // C_APB_DAT_WIDTH == 8
			      buf_stat = 1;
			      buf_cntr = 32'h0;
			      prev_addr = 32'h0;
				  //Wait for read data counter value to be equal to PAGE_READ_SIZE
                  while (rd_data_cntr != PAGE_READ_SIZE)begin
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Get buffer status
					buf_stat       = apb_prdata_i[0];
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG_1;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Get buffer counter
					buf_cntr[7:0] =  apb_prdata_i;
		            //Set the register for Read Transaction Status Register
                    addr_32b       = 0;
                    addr_32b[10:0] = RD_BUFF_STAT_REG_2;
                    //Initiate the transaction through APB
                    read_apb(addr_32b);
				    //Get buffer counter
					buf_cntr[10:8] =  apb_prdata_i[2:0];
				    //Wait for the buffer to be not empty with content counter not less than 1
			        while (buf_stat == 0 & buf_cntr < 1) begin
		               //Set the register for Read Transaction Status Register
                       addr_32b       = 0;
                       addr_32b[10:0] = RD_BUFF_STAT_REG;
                       //Initiate the transaction through APB
                       read_apb(addr_32b);
				       //Get buffer status
					   buf_stat       = apb_prdata_i[0];
		               //Set the register for Read Transaction Status Register
                       addr_32b       = 0;
                       addr_32b[10:0] = RD_BUFF_STAT_REG_1;
                       //Initiate the transaction through APB
                       read_apb(addr_32b);
				       //Get buffer counter
					   buf_cntr[7:0] =  apb_prdata_i;
		               //Set the register for Read Transaction Status Register
                       addr_32b       = 0;
                       addr_32b[10:0] = RD_BUFF_STAT_REG_2;
                       //Initiate the transaction through APB
                       read_apb(addr_32b);
				       //Get buffer counter
					   buf_cntr[10:8] =  apb_prdata_i[2:0];
			        end
				    //Get the buffer content value
			        cur_buf_in_cnt = buf_cntr;
				    //Wait for the buffer content value to increment
			        if (cur_buf_in_cnt > prev_buf_in_cnt) begin
                        //Read word data
                        addr_32b       = 0;
                        addr_32b[10:0] = rd_addr_32b;
                        //Program the value to initiate word read transaction & Initiate the transaction through AHB
                        ahb_intf_en_b   = 1;
                        size_3b         = 0;
                        read_ahb(rd_addr_32b,size_3b);
                        rd_data_2d_8b[ptr] = ahb_rd_data_32b;
                        ptr             = ptr + 1;
                        ahb_intf_en_b   = 0;
			            //Increment read data counter
					    rd_data_cntr    = rd_data_cntr + 1;
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[7:0] = apb_rd_data[7:0];
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG_1;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[15:8] = apb_rd_data;
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG_2;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[23:16] = apb_rd_data[7:0];
		                //Set the register for Read Buffer Address Register
                        addr_32b       = 0;
                        addr_32b[10:0] = RD_BUFF_ADDR_REG_3;
                        //Initiate the transaction through APB
                        read_apb(addr_32b);
						prev_addr[31:24] = apb_rd_data;
			            //Increment APB read address
			    	    rd_addr_32b = prev_addr + 32'h1;
			    	end
				    //Store current buffer content value to previous value
			    	prev_buf_in_cnt = cur_buf_in_cnt;
                  end
				  //W1 to clear on Read Buffer Status Register
                  addr_32b       = 0;
                  addr_32b[10:0] = RD_BUFF_STAT_REG;
                  data_32b       = 0;
                  data_32b[31:0] = 32'hFFFFFFFF;
                  //Initiate the transaction through APB
                  write_apb(addr_32b,data_32b);
			  end
			end
		  end
        //end
		else begin
		  //Set the register for Read Transaction Status Register
          addr_32b       = 0;
          addr_32b[10:0] = TRANS_STAT_REG;
          //Initiate the transaction through APB
          read_apb(addr_32b);
		end
        if (S_AHB_DAT_WIDTH == 32) begin
          size_3b         = 2;
          read_ahb(rd_addr_32b,size_3b);
          rd_data_2d_32b[ptr] = ahb_rd_data_32b;
		end
        //compare the data for write and read
        if (S_AHB_DAT_WIDTH == 32)
          for (m_j_i = 0; m_j_i < (PAGE_COMPARE>>2); m_j_i = m_j_i + 1) begin
            do_data_compare2(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i],m_j_i);
          end
        else
          for (m_j_i = 0; m_j_i < PAGE_COMPARE; m_j_i = m_j_i + 1) begin
            do_data_compare2(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i],m_j_i);
          end
        //Disable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h00;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
      end
    end
	end
  endtask

  task test_page_ahb_burst;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] data_32b;
    integer m_i_i;
    integer m_j_i;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
      //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = TRANS_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
      while (apb_prdata_i == 1) begin
           read_apb(addr_32b);
      end
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        //Enable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h01;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Set the address
        addr_32b       = 0;
        addr_32b[11:2]  = $random;
        addr_32b[1:0]   = 9'h000;
        prg_addr_32b    = addr_32b;
        rd_addr_32b     = addr_32b;
        //Program the value
        if (S_AHB_DAT_WIDTH == 32)
          for (m_j_i = 0; m_j_i < 128; m_j_i = m_j_i + 1) begin
            burst_prg_data_2d_32b[m_j_i][31:0]  = $random;
          end
        else
          for (m_j_i = 0; m_j_i < 512; m_j_i = m_j_i + 1) begin
            burst_prg_data_2d_8b[m_j_i][7:0]  = $random;
          end
        ahb_intf_en_b   = 1;
        if (S_AHB_DAT_WIDTH == 32) begin
          size_3b         = 2;
          //Initiate the transaction through AHB
          burst_write_ahb(prg_addr_32b,size_3b,(PAGE_PROGRAM_SIZE>>2));
        end
        else begin
          size_3b         = 0;
          //Initiate the transaction through AHB
          burst_write_ahb(prg_addr_32b,size_3b,PAGE_PROGRAM_SIZE);
        end
        wait_clocks(10);
        //Set the spi flash address
        addr_32b       = 0;
        addr_32b[10:0] = SPI_ADDR_BYTE0;
        data_32b       = prg_addr_32b;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Set the flash addressing mode to 24-bit or 32-bit mode
        addr_32b       = 0;
        addr_32b[10:0] = SPI_ADDR_MODE;
        data_32b       = 0;
        data_32b[7:0]  = (S_AHB_ADR_WIDTH == 32) ? 8'h01 : 8'h00;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Set the page program length
        addr_32b       = 0;
        addr_32b[10:0] = PAGE_PRG_LEN_0;
        data_32b       = 0;
        data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Run the page program in SPI interface
        addr_32b       = 0;
        addr_32b[10:0] = PAGE_PRG;
        data_32b       = 0;
        data_32b[7:0]  = 8'h01;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
		//Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = TRANS_STAT_REG;
        //Initiate the transaction through APB
        read_apb(addr_32b);
		//Wait for the transaction to finish
        while (apb_prdata_i == 1) begin
             read_apb(addr_32b);
        end
        //Set the page read length
        addr_32b       = 0;
        addr_32b[10:0] = PAGE_RD_LEN_0;
        data_32b       = 0;
        data_32b[15:0] = PAGE_PROGRAM_SIZE[15:0];
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Run the page read in SPI interface
        addr_32b       = 0;
        addr_32b[10:0] = PAGE_RD;
        data_32b       = 0;
        data_32b[7:0]  = 8'h01;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
		//Set the register for Read Transaction Status Register
        addr_32b       = 0;
        addr_32b[10:0] = TRANS_STAT_REG;
        //Initiate the transaction through APB
        read_apb(addr_32b);
	    //Wait for the transaction to finish
        while (apb_prdata_i == 1) begin
             read_apb(addr_32b);
        end
        //Program the value to initiate word read transaction & Initiate the transaction through AHB
        ahb_intf_en_b   = 1;
        if (S_AHB_DAT_WIDTH == 32) begin
          size_3b         = 2;
          burst_read_ahb(rd_addr_32b,size_3b,(PAGE_PROGRAM_SIZE>>2));
        end
        else begin
          size_3b         = 0;
          burst_read_ahb(rd_addr_32b,size_3b,PAGE_PROGRAM_SIZE);
        end
        //Wait for transaction completion
        wait_clocks(10);
        ahb_intf_en_b   = 0;
        //compare the data for write and read
        addr_32b        = rd_addr_32b;
        if (S_AHB_DAT_WIDTH == 32)
          for (m_j_i = 0; m_j_i < (PAGE_READ_SIZE>>2); m_j_i = m_j_i + 1) begin
            do_data_compare(burst_prg_data_2d_32b[addr_32b[8:2]],burst_rd_data_2d_32b[addr_32b[8:2]]);
            addr_32b      = addr_32b + 4;
          end
        else
          for (m_j_i = 0; m_j_i < PAGE_READ_SIZE; m_j_i = m_j_i + 1) begin
            do_data_compare(burst_prg_data_2d_8b[addr_32b[8:0]],burst_rd_data_2d_8b[addr_32b[8:0]]);
            addr_32b      = addr_32b + 1;
          end
        //Disable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h00;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
      end
    end
  endtask

  task test_page_ahb_error;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] erase_addr_32b;
    reg [31:0] byte_addr_32b;
    reg [31:0] prg_data_2d_32b;
    reg [31:0] rd_data_2d_32b;
    reg [7:0]  prg_data_2d_8b;
    reg [7:0]  rd_data_2d_8b;
    reg [2:0]  size_3b;
    reg [31:0] data_32b;
    integer m_i_i;
    integer m_j_i;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
      //Set the register for Read Transaction Status Register
      addr_32b       = 0;
      addr_32b[10:0] = TRANS_STAT_REG;
      //Initiate the transaction through APB
      read_apb(addr_32b);
	  //Wait for the transaction to finish
      while (apb_prdata_i == 1) begin
           read_apb(addr_32b);
      end
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        //Enable Page Buffer in AHB-L
        addr_32b       = 0;
        addr_32b[10:0] = AHB_PAGE_BUF_EN;
        data_32b       = 0;
        data_32b[7:0]  = 8'h01;
        //Initiate the transaction through APB
        write_apb(addr_32b,data_32b);
        //Incorrect size
        addr_32b       = 0;
        addr_32b[11:2]  = $random;
        addr_32b[1:0]   = 9'h000;
        prg_addr_32b    = addr_32b;
        rd_addr_32b     = addr_32b;
        for (m_j_i = 0; m_j_i < 8; m_j_i = m_j_i + 1) begin
          //Program the value to initiate word write transaction
          prg_data_2d_32b = $random;
          ahb_intf_en_b   = 1;
          size_3b         = m_j_i;
          //Initiate the transaction through AHB
          write_ahb(prg_addr_32b,prg_data_2d_32b,size_3b);
          wait_clocks(10);
        end
        for (m_j_i = 0; m_j_i < 8; m_j_i = m_j_i + 1) begin
          //Program the value to initiate word read transaction & Initiate the transaction through AHB
          ahb_intf_en_b   = 1;
          size_3b         = m_j_i;
          read_ahb(rd_addr_32b,size_3b);
          //Wait for transaction completion
          wait_clocks(10);
          ahb_intf_en_b   = 0;
        end
        //no data check, error response check only
        wait_clocks(50);
        for (m_j_i = 0; m_j_i < 8; m_j_i = m_j_i + 1) begin
          //Program the value to initiate burst write transaction
          ahb_intf_en_b   = 1;
          size_3b         = (S_AHB_DAT_WIDTH == 32) ? 2 : 0;
          //Initiate the transaction through AHB
          burst_write_ahb_error_chk(prg_addr_32b,size_3b,m_j_i,32);
          wait_clocks(10);
        end
        for (m_j_i = 0; m_j_i < 8; m_j_i = m_j_i + 1) begin
          //Program the value to initiate burst read transaction & Initiate the transaction through AHB
          ahb_intf_en_b   = 1;
          size_3b         = (S_AHB_DAT_WIDTH == 32) ? 2 : 0;
          burst_read_ahb_error_chk(rd_addr_32b,size_3b,m_j_i,32);
          //Wait for transaction completion
          wait_clocks(10);
          ahb_intf_en_b   = 0;
        end
        //no data check, error response check only
        wait_clocks(50);
      end
    end
  endtask

// Below task issues another AHB-L access while the previous access just started
// wait_clocks used is wait_clocks(10)
  task test_ahb_wait;
    // Address and Data register
    reg [31:0] addr_32b;
    reg [31:0] prg_addr_32b;
    reg [31:0] rd_addr_32b;
    reg [31:0] erase_addr_32b;
    reg [31:0] byte_addr_32b;
    reg [31:0] prg_data_2d_32b [255:0];
    reg [31:0] rd_data_2d_32b  [255:0];
    reg [7:0]  prg_data_2d_8b  [255:0];
    reg [7:0]  rd_data_2d_8b   [255:0];
    reg [2:0]  size_3b;
    reg [31:0] data_32b;
    integer m_i_i;
    integer m_j_i;
    begin
      //Wait time
      wait_clocks(500*(CLOCK_SEL+1));
      //2-----------------------------------------------------------------------------------------------
      // Generate num tran commands 
      //2-----------------------------------------------------------------------------------------------
      for (m_i_i = 0; m_i_i < num_tran_ui; m_i_i = m_i_i + 1) begin
        if( S_AHB_ADR_WIDTH == 32) begin
          //Set the flash addressing mode to 32-bit mode
          addr_32b       = 0;
          addr_32b[10:0] = SPI_ADDR_MODE;
          data_32b       = 0;
          data_32b[7:0]  = 8'h01;
          //Initiate the transaction through APB
          write_apb(addr_32b,data_32b);
        end
        //Set address
        addr_32b[31:12] = 20'h00000;
        addr_32b[11:2]  = $random;
        addr_32b[1:0]   = 2'h0;
        prg_addr_32b    = addr_32b-64;
        rd_addr_32b     = addr_32b-64;
        if( S_AHB_DAT_WIDTH == 32) begin
          for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word write transaction
            prg_data_2d_32b[m_j_i][31:0]  = $random;
            ahb_intf_en_b                 = 1;
            size_3b                       = 2;
            //Initiate the transaction through AHB
            write_ahb(prg_addr_32b,prg_data_2d_32b[m_j_i],size_3b);
            wait_clocks(5);
            prg_addr_32b    = prg_addr_32b+4;
          end
          for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 2) begin
            //Program the value to initiate word read transaction & Initiate the transaction through AHB
            ahb_intf_en_b   = 1;
            size_3b         = 2;
            fork 
              begin
                read_ahb(rd_addr_32b,size_3b);
                rd_data_2d_32b[m_j_i] = ahb_rd_data_32b;
              end
              
              begin
                wait_clocks(10);
                read_ahb(rd_addr_32b+4,size_3b);
                rd_data_2d_32b[m_j_i+1] = ahb_rd_data_32b;
              end
            join
            ahb_intf_en_b   = 0;
            rd_addr_32b     = rd_addr_32b+8;
          end
          //compare the data for write and read
          for (m_j_i = 0; m_j_i < 16; m_j_i = m_j_i + 1) begin
            do_data_compare(prg_data_2d_32b[m_j_i],rd_data_2d_32b[m_j_i]);
          end
        end
        else begin
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            //Program the value to initiate word write transaction
            prg_data_2d_8b[m_j_i][7:0]    = $random;
            ahb_intf_en_b                 = 1;
            size_3b                       = 0;
            //Initiate the transaction through AHB
            write_ahb(prg_addr_32b,prg_data_2d_8b[m_j_i],size_3b);
            wait_clocks(10);
            prg_addr_32b    = prg_addr_32b+1;
          end
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 2) begin
            //Program the value to initiate word read transaction & Initiate the transaction through AHB
            ahb_intf_en_b   = 1;
            size_3b         = 0;
            fork 
              begin
                read_ahb(rd_addr_32b,size_3b);
                rd_data_2d_8b[m_j_i] = ahb_rd_data_32b;
              end
              
              begin
                wait_clocks(10);
                read_ahb(rd_addr_32b+1,size_3b);
                rd_data_2d_8b[m_j_i+1] = ahb_rd_data_32b;
              end
            join
            ahb_intf_en_b   = 0;
            rd_addr_32b     = rd_addr_32b+2;
          end
          //compare the data for write and read
          for (m_j_i = 0; m_j_i < 64; m_j_i = m_j_i + 1) begin
            do_data_compare(prg_data_2d_8b[m_j_i],rd_data_2d_8b[m_j_i]);
          end
        end
      end
    end
  endtask

  //1-------------------------------------------------------------------------------------------------
  // post_process :This task does the post processing 
  //1-------------------------------------------------------------------------------------------------
  //             parameter :No parameter 
  //1-------------------------------------------------------------------------------------------------
  task post_process;
    begin
      $write("MSG :: @%0dns %m() :: #####################################################################\n",
        $time);
      $write("MSG :: @%0dns %m() ::               Errors   detected in CHECKER %4d\n",$time,errs_i);
      $write("MSG :: @%0dns %m() ::               Number of Transactions       %4d\n",$time,trans_i);
      $write("MSG :: @%0dns %m() :: #####################################################################\n",
        $time);
      //3---------------------------------------------------------------------------------------------
      // Declare pass of fail based on sim status 
      //3---------------------------------------------------------------------------------------------
      if (errs_i == 0 && trans_i != 0) begin
        $write("MSG :: @%0dns %m() ::                       SIMULATION PASSED\n",$time);
      //3---------------------------------------------------------------------------------------------
      // If sim error is set, then there was simulation error 
      //3---------------------------------------------------------------------------------------------
      end else begin
        $write("MSG :: @%0dns %m() ::                       SIMULATION FAILED\n",$time);
      end
      $write("MSG :: @%0dns %m() :: #####################################################################\n",
        $time);
    end
  endtask


  //1-------------------------------------------------------------------------------------------------
  // test_runner :This task runs testcase 
  //1-------------------------------------------------------------------------------------------------
  //             parameter :No parameter 
  //1-------------------------------------------------------------------------------------------------
  task test_runner;
    begin
      $write("MSG :: @%0dns %m() :: \n----------------------------------------------------------------------------------------------------\n                                        SPI Flash Controller IP\n----------------------------------------------------------------------------------------------------\n\n",
        $time);
      //3---------------------------------------------------------------------------------------------
      // Check if seed was passed 
      //3---------------------------------------------------------------------------------------------
      if ($value$plusargs("sdvt_seed=%d", seed_i)) begin
        $write("MSG :: @%0dns %m() :: Setting random seed to user passed %0d\n",$time,seed_i);
      //3---------------------------------------------------------------------------------------------
      // Use seed as 1 
      //3---------------------------------------------------------------------------------------------
      end else begin
        seed_i                         = 1;
        $write("MSG :: @%0dns %m() :: Setting random seed to default %0d\n",$time,seed_i);
      end
      //3---------------------------------------------------------------------------------------------
      // Check if cmds was passed 
      //3---------------------------------------------------------------------------------------------
      if ($value$plusargs("cmds=%d", num_tran_ui)) begin
        //4-------------------------------------------------------------------------------------------
        // Print message only when num_tran_ui is changed 
        //4-------------------------------------------------------------------------------------------
        if (num_tran_ui > 0) begin
          $write("MSG :: @%0dns %m() :: Changing num_tran_ui to %0d\n",$time,num_tran_ui);
        //4-------------------------------------------------------------------------------------------
        // Make sure user did not pass -1 or 0 
        //4-------------------------------------------------------------------------------------------
        end else begin
          num_tran_ui                    = 100;
        end
      end else begin
        num_tran_ui                      = 1;
        $write("MSG :: @%0dns %m() :: Setting num_tran_ui to default %0d\n",$time,num_tran_ui);
      end
      //3---------------------------------------------------------------------------------------------
      // Check if finish was passed 
      //3---------------------------------------------------------------------------------------------
      if ($value$plusargs("sdvt_finish=%b", finish_on_error_b)) begin
      end
      //3---------------------------------------------------------------------------------------------
      // Check if testcase was passed 
      //3---------------------------------------------------------------------------------------------
      if ($value$plusargs("test=%s", test_name_s)) begin
        $write("MSG :: @%0dns %m() :: Running testcase %0s\n",$time,test_name_s);
        //4-------------------------------------------------------------------------------------------
        // Run testcase test_basic 
        //4-------------------------------------------------------------------------------------------
        if (test_name_s == "test_basic") begin
          test_basic();
        //4-------------------------------------------------------------------------------------------
        // Run testcase test_page 
        //4-------------------------------------------------------------------------------------------
        end else if (test_name_s == "test_page") begin
          if (PAGE_PRG_BUF_ENA[0] && PAGE_READ_BUF_ENA[0])
            test_page();
          else
            $write("MSG :: @%0dns %m() :: test_page was not executed. Need to enable both Page Program and Page Read Buffers ...............\n",
              $time);
        //4-------------------------------------------------------------------------------------------
        // If not valid testcase name is passed, then it is error 
        //4-------------------------------------------------------------------------------------------
        end else begin
          $write("MSG :: @%0dns %m() :: No valid testcase name passed with +test option, so calling testcase-test_basic ...................\n",
            $time);
          test_basic();
        end
      //3---------------------------------------------------------------------------------------------
      // If testcase name is not passed, then it is error 
      //3---------------------------------------------------------------------------------------------
      end else begin
          $write("MSG :: @%0dns %m() :: No testcase name passed with +test option, so calling testcase-test_basic, test-page...............\n",
          $time);
          test_basic();
          test_ahb_wait();
          if (PAGE_PRG_BUF_ENA[0] && PAGE_READ_BUF_ENA[0]) begin
            if (PAGE_BUFFER_INTF == "APB") begin
              test_page();
              test_page_with_rd_buf_stat_addr();
			end
            else begin
              test_page_ahb();
			  test_page_ahb_with_rd_buf_stat_addr();
              test_page_ahb_burst();
              test_page_ahb_error();
            end
          end
          test_basic_user_defined_command();
      end
    end
  endtask

  //1-------------------------------------------------------------------------------------------------
  // do_data_compare : This method implements the data comparison 
  //1-------------------------------------------------------------------------------------------------
  task do_data_compare;
    input [31:0]                   p_exe_data_32b                   ; // Expected data 
    input [31:0]                   p_got_data_32b                   ; // Got data 

      //Compare Address
      if (p_exe_data_32b === p_got_data_32b) begin
        $write("MSG :: @%0dns %m() :: Data expected 0x%x, Got 0x%x\n",$time,p_exe_data_32b,p_got_data_32b);
      end else begin
        process_err();
        $write("ERROR :: @%0dns %m() :: Data expected 0x%x, Got 0x%x\n",$time,p_exe_data_32b,p_got_data_32b);
      end

  endtask

  task do_data_compare2;
    input [31:0]                   p_exe_data_32b                   ; // Expected data 
    input [31:0]                   p_got_data_32b                   ; // Got data 
    input [31:0]                   index_32b                        ;
      //Compare Address
      if (p_exe_data_32b === p_got_data_32b) begin
        $write("MSG :: @%0dns %m() :: Data expected 0x%x, Got 0x%x :: Index %d\n",$time,p_exe_data_32b,p_got_data_32b,index_32b);
      end else begin
        process_err();
        $write("ERROR :: @%0dns %m() :: Data expected 0x%x, Got 0x%x :: Index %d\n",$time,p_exe_data_32b,p_got_data_32b,index_32b);
      end

  endtask

endmodule

`endif