
#include <stdio.h>
#include "gpio_regs.h"
#include "hal.h"
#include "image.h"
#include "led.h"
#include "memory_controller.h"
#include "reg_access.h"
#include "riscv.h"
#include "spi_flash_cntl.h"
#include "sys_platform.h"

struct local_uart_instance local_uart_core;
struct flash_cntl_instance this_flash_cntl;

/*
 * Change the following values according to the image
 * source and destination address:
 *
 * APPLICATION_IMG_FLASH_ADDR must match with the
 * image start address in flash device
 *
 * APPLICATION_MEMORY_START_ADDR must be 16-byte aligned
 * e.g. 0x40000010, 0x40000020 etc.
 * It must match with the image entry point when loaded
 * to DDR memory
 */
#define APPLICATION_IMG_FLASH_ADDR 		0x00000000
#define APPLICATION_MEMORY_START_ADDR 	0x50000000

#define CPU_CLOCK_HZ          CPU_INST_SYS_CLOCK_FREQ * 1000000
#define LED_PATTERN 0xAA

static int lscc_uart_putc(char c, FILE *file)
{
#ifdef LSCC_STDIO_UART_APB
		return local_uart_putc(&local_uart_core, c);
#else
		return EOF;
#endif
}

static int lscc_uart_getc(FILE *file)
{
	(void) file;
	return EOF;
}

static int lscc_uart_flush(FILE *file)
{
	(void) file;
	return 0;
}

void main()
{
	uint8_t manu_id=0;
	uint8_t clk_div = 1;
	uint8_t clk_mode = 0;
	uint8_t spi_ret;
	uint32_t app_base = APPLICATION_MEMORY_START_ADDR;

	image_status_t image_status;
	addr_mode mode = ADDR_MODE_4B;
	ddr_ret ret;

    led_init((uint32_t *) (GPIO0_INST_GPIO_MEM_MAP_BASE_ADDR + GPIO_WR_DATA), GPIO0_INST_LINES_NUM);
    led_set(LED_PATTERN);

    local_uart_init(&local_uart_core, CPU_INST_LOCAL_UART_BASE_ADDR, CPU_CLOCK_HZ, CPU_INST_STANDARD_BAUD_RATE, 1, 8);
    iob_init(lscc_uart_putc, lscc_uart_getc, lscc_uart_flush);

    printf("*****************************************\r\n");
    printf("* RISC-V RX Reference Design Bootloader *\r\n");
    printf("*****************************************\r\n");

    printf("\r\nSPI Flash Controller Initialization: ");
    spi_ret = spi_flash_cntl_init(&this_flash_cntl, SPI_FLASH1_INST_SPI_FLASH_CONTROLLER_MEM_MAP_APB_BASE_ADDR, PAGE_SIZE, PAGE_SIZE, mode);

    if(spi_ret == 0)
	{
    	printf("Passed\r\n");
	}
    else
    {
    	printf("Failed\r\n");
    	goto boot_fail;
    }

    /*
     * Set the clock rate to 25MHz
     * Set the clock mode to mode 0
     */
    spi_flash_cntl_set_clock_rate(&this_flash_cntl, clk_div);
    spi_flash_cntl_set_clock_mode(&this_flash_cntl, clk_mode);

    spi_flash_cntl_id_read(&this_flash_cntl, &manu_id);
    printf("  Flash manufacturer ID: %x\r\n", manu_id);

    printf("\r\nMemory Controller Initialization   : ");
	ret = mem_cntl_init();

	switch(ret)
	{
		case NO_FAIL:
			printf("Passed\r\n");
			break;
		case CBT_FAIL:
			printf("Command Bus Training Failed\r\n");
			goto boot_fail;
		case WR_LVL_FAIL:
			printf("Write Leveling Failed\r\n");
			goto boot_fail;
		case RD_DQS_TRN_FAIL:
			printf("Read DQS Training Failed\r\n");
			goto boot_fail;
		case RD_DATA_EYE_TRN_FAIL:
			printf("Read Data Eye Training Failed\r\n");
			goto boot_fail;
		case WR_TRN_FAIL:
			printf("Write Training Failed\r\n");
			goto boot_fail;
		case OTHER_FAIL:
			printf("Others Failure\r\n");
			goto boot_fail;
	}

	printf("\r\nLoad and CRC check image\r\n");
	printf("SRC : Flash addr %x\r\n", APPLICATION_IMG_FLASH_ADDR);
	printf("DEST: DDR addr %x\r\n",  APPLICATION_MEMORY_START_ADDR);
    image_status = read_flash_image(APPLICATION_IMG_FLASH_ADDR, (uint32_t *) APPLICATION_MEMORY_START_ADDR);

    if(image_status == IMAGE_SUCCESS)
    {
    	printf("Passed\r\n");
    }
    else if(image_status == IMAGE_MAGIC_ERR)
    {
    	printf("Failed (Invalid image)\r\n");
    	goto boot_fail;
    } else if(image_status == IMAGE_CRC32_ERR)
    {
    	printf("Failed (Bad CRC)\r\n");
    	goto boot_fail;
    }

	printf("\r\nJumping to the application\r\n");
	printf("..............................\r\n");
	printf("..............................\r\n");
	printf("..............................\r\n");
	printf("..............................\r\n");
	printf("\r\n");

	/*
	 * Flush the cache.
	 */
	asm volatile ("fence.i");

	/*
	 * Jump to the application address
	 */
	asm volatile ("jalr zero, %0" : : "r" (app_base));

	/*
	 * Should not reach here
	 */
	boot_fail:
		printf("BOOT FAILED. PLEASE RESET THE BOARD\r\n");
		while(1)
		{}

}

