//----------------------------------------------------------------------------------------------------
// HDL Test Bench
//----------------------------------------------------------------------------------------------------

//0###################################################################################################
// Include files 
//0###################################################################################################
`timescale 1ns/1ps
`include "test_stimuli.v"

module tb_top();

  //----------------------------------------------------------------------
  // DUT Parameters
  //----------------------------------------------------------------------
  `include "dut_params.v"

  parameter HCLK_PERIOD   =   10;
  parameter PCLK_PERIOD   =   20;
  
  //SPI IO Interface signals
  //wire                           clk_i                        ; // System clock input
  wire                           rst_i                        ; // Active High system reset input
  wire                           sclk_o                       ; // SCK from Master
  wire                           ss_n_o                       ; // Chip Select from Master
  wire                           wpj_o                        ; // Write protect from Master
  wire                           miso_i                       ; // Slave data output
  wire                           mosi_o                       ; // Slave data input

  wire                           ahbl_hclk_i                  ; // AHB Clock
  wire [S_AHB_ADR_WIDTH-1:0]     ahbl_haddr_i                 ; // AHB Address bus
  wire [S_AHB_DAT_WIDTH-1:0]     ahbl_hwdata_i                ; // AHB Data bus
  wire [2:0]                     ahbl_hsize_i                 ; // AHB Size input
  wire                           ahbl_hwrite_i                ; // AHB write input
  wire                           ahbl_hsel_i                  ; // AHB Select input
  wire [2:0]                     ahbl_hburst_i                ; // AHB Burst input
  wire [1:0]                     ahbl_htrans_i                ; // AHB trans input
  wire                           ahbl_hlock_i                 ; // AHB Lock input
  wire [S_AHB_DAT_WIDTH-1:0]     ahbl_hrdata_o                ; // AHB Read data output
  wire                           ahbl_hresp_o                 ; // AHB Response output
  wire                           ahbl_hready_o                ; // AHB Ready output

  wire                           apb_pclk_i                   ; // APB Clock
  wire [C_APB_ADR_WIDTH-1:0]     apb_paddr_i                  ; // APB Address bus
  wire [C_APB_DAT_WIDTH-1:0]     apb_pwdata_i                 ; // APB Write Data bus
  wire                           apb_pwrite_i                 ; // APB Write Input
  wire                           apb_psel_i                   ; // APB Select Input
  wire                           apb_penable_i                ; // APB Enable Input
  wire                           apb_pready_o                 ; // APB Ready output
  wire [C_APB_DAT_WIDTH-1:0]     apb_prdata_o                 ; // APB Read data output
  wire                           apb_pslverr_o                ; // APB Slave Error output

  //reg                            clk                          ; // System clock input
  reg                            hclk                         ; // AHB clock input
  reg                            pclk                         ; // APB clock input
  reg                            rst                          ; // Active High system reset input
  
  //Generate an internal SPI clock for CPOL and CPHASE control
  wire [3:0]                     div_num                      ;
  wire                           sclk                         ;
  reg [3:0]                      clk_cnt                      ;
  reg                            clk_div                      ;


  GSR GSR_INST (
      .GSR_N(1'b1),
      .CLK(hclk)
      //.CLK(clk)
  );

  //1###################################################################################################
  // Code to assign the input signals 
  //1###################################################################################################
  //assign clk_i      = clk;
  assign ahbl_hclk_i  = hclk;
  assign apb_pclk_i   = pclk;
  assign rst_i        = rst;
  assign div_num      = CLOCK_SEL;
  assign sclk         = div_num > 0   ? clk_div      : pclk;

  always @(posedge pclk or posedge rst_i)
    if(rst_i) begin
	    clk_cnt <= 4'h0;
	    clk_div <= 1'b0;
    end
    else if(clk_cnt == div_num - 4'd1) begin
	    clk_div <= ~clk_div;
	    clk_cnt <= 4'h0;
    end 
    else if (div_num > 4'd0)
      clk_cnt <= clk_cnt + 4'h1;

  //1###################################################################################################
  // Initilize and run testcase 
  //1###################################################################################################
  initial begin
    //clk                            = 0;
    hclk                           = 0;
    pclk                           = 0;
    drive_reset(1,10);
    drive_reset(0,10);
  end

  //1-------------------------------------------------------------------------------------------------
  // System Clock generator 
  //1-------------------------------------------------------------------------------------------------
  // always 
  //   begin
  //     #5;
  //     clk                              = ~clk;
  //   end
  
  always 
    begin
      #(HCLK_PERIOD/2);
      hclk                              = ~hclk;
    end

  always 
    begin
      #(PCLK_PERIOD/2);
      pclk                              = ~pclk;
    end

  //1-------------------------------------------------------------------------------------------------
  // drive_reset :This method is used for driving reset pin of Timer 
  //1-------------------------------------------------------------------------------------------------
  //             p_value_b :Value to drive on bus 
  //           p_delay_32b :Number of clocks delay 
  //1-------------------------------------------------------------------------------------------------
  task drive_reset;
    input                               p_value_b                      ;
    input    [31:0]                     p_delay_32b                    ;
    begin
      //3---------------------------------------------------------------------------------------------
      // Wait for the delay 
      //3---------------------------------------------------------------------------------------------
      repeat (p_delay_32b) begin
        //@ (posedge clk);
        @ (posedge pclk);
      end
      $write("INFO :: @%0dns %m() :: Driving reset pin to %0d\n",$time,p_value_b);
      rst                            <= p_value_b;
    end
  endtask

  //1###################################################################################################
  // Connect the SPI Master instance 
  //1###################################################################################################
    `include "dut_inst.v"
    test_stimuli   
       #(.S_AHB_ADR_WIDTH                   (S_AHB_ADR_WIDTH                  ),
         .S_AHB_DAT_WIDTH                   (S_AHB_DAT_WIDTH                  ),
         .S_AHB_DAT_ENDIANESS               (S_AHB_DAT_ENDIANESS              ),
         .C_PORT_ENABLE                     (C_PORT_ENABLE                    ),
         .C_APB_ADR_WIDTH                   (C_APB_ADR_WIDTH                  ),
         .C_APB_DAT_WIDTH                   (C_APB_DAT_WIDTH                  ),
         .PAGE_PRG_BUF_ENA                  (PAGE_PRG_BUF_ENA                 ),
         .PAGE_READ_BUF_ENA                 (PAGE_READ_BUF_ENA                ),
         .PAGE_PRG_BUFFER_EBR               (PAGE_PRG_BUFFER_EBR              ),
         .PAGE_PRG_BUFFER_DISTRIBUTED_RAM   (PAGE_PRG_BUFFER_DISTRIBUTED_RAM  ),
         .PAGE_READ_BUFFER_EBR              (PAGE_READ_BUFFER_EBR             ),
         .PAGE_READ_BUFFER_DISTRIBUTED_RAM  (PAGE_READ_BUFFER_DISTRIBUTED_RAM ),
         .PAGE_BUFFER_INTF                  (PAGE_BUFFER_INTF                 ),
         .PAGE_PROGRAM_SIZE                 (PAGE_PROGRAM_SIZE                ),
         .PAGE_READ_SIZE                    (PAGE_READ_SIZE                   ),
         .CLOCK_SEL                         (CLOCK_SEL                        ),
		 .CPOL                              (CPOL                             ),
		 .CPHASE                            (CPHASE                           ),
         .LSB_FIRST                         (LSB_FIRST                        ),
         .SPI_READ                          (SPI_READ                         ),
         .SPI_FAST_READ                     (SPI_FAST_READ                    ),
         .SPI_BYTE_PRG                      (SPI_BYTE_PRG                     ),
         .SPI_PAGE_PRG                      (SPI_PAGE_PRG                     ),
         .SPI_BLK1_ERS                      (SPI_BLK1_ERS                     ),
         .SPI_BLK2_ERS                      (SPI_BLK2_ERS                     ),
         .SPI_BLK3_ERS                      (SPI_BLK3_ERS                     ),
         .SPI_CHIP_ERS                      (SPI_CHIP_ERS                     ),
         .SPI_WRT_ENB                       (SPI_WRT_ENB                      ),
         .SPI_WRT_DISB                      (SPI_WRT_DISB                     ),
         .SPI_READ_STAT                     (SPI_READ_STAT                    ),
         .SPI_WRT_STAT                      (SPI_WRT_STAT                     ),
         .SPI_PWD_DOWN                      (SPI_PWD_DOWN                     ),
         .SPI_PWD_UP                        (SPI_PWD_UP                       ),
         .SPI_DEV_ID                        (SPI_DEV_ID                       ),
		 .PCLK_PERIOD                       (PCLK_PERIOD                      ))
    U_test_stimuli
       (
         // SPI Interface Signals
         //.clk_i                      (clk_i                ),
         .rst_i                      (rst_i                ),
         .sclk_i                     (sclk_o               ),
         .spi_clk                    (sclk                 ),
         .ss_n_i                     (ss_n_o               ),
         .wpj_i                      (wpj_o                ),
         .miso_o                     (miso_i               ),
         .mosi_i                     (mosi_o               ),
         // AHB Interface Signals
         .ahbl_hclk_i                (ahbl_hclk_i          ),
         .ahbl_haddr_o               (ahbl_haddr_i         ),
         .ahbl_hwdata_o              (ahbl_hwdata_i        ),
         .ahbl_hsize_o               (ahbl_hsize_i         ),
         .ahbl_hwrite_o              (ahbl_hwrite_i        ),
         .ahbl_hsel_o                (ahbl_hsel_i          ),
         .ahbl_hburst_o              (ahbl_hburst_i        ),
         .ahbl_htrans_o              (ahbl_htrans_i        ),
         .ahbl_hlock_o               (ahbl_hlock_i         ),
         .ahbl_hrdata_i              (ahbl_hrdata_o        ),
         .ahbl_hresp_i               (ahbl_hresp_o         ),
         .ahbl_hready_i              (ahbl_hready_o        ),
         // APB Interface Signals
         .apb_pclk_i                 (apb_pclk_i           ),
         .apb_paddr_o                (apb_paddr_i          ),
         .apb_pwdata_o               (apb_pwdata_i         ),
         .apb_pwrite_o               (apb_pwrite_i         ),
         .apb_psel_o                 (apb_psel_i           ),
         .apb_penable_o              (apb_penable_i        ),
         .apb_pready_i               (apb_pready_o         ),
         .apb_prdata_i               (apb_prdata_o         ),
         .apb_pslverr_i              (apb_pslverr_o        )
       ); 

endmodule
