import struct
import zlib
import sys
import argparse
import os

# Define command-line arguments
parser = argparse.ArgumentParser()
parser.add_argument("input_file", help="path to the input firmware file")
parser.add_argument("output_file", help="path to the output firmware image file")
args = parser.parse_args()

# Read in the firmware data from the input file
try:
    with open(args.input_file, 'rb') as f:
        firmware_data = f.read()
except FileNotFoundError:
    # Handle file not found error
    print(f"Error: could not find input firmware file at {args.input_file}")
    print("Usage: python generate_firmware_image.py <input_file> <output_file>")
    sys.exit(1)
except:
    # Handle other errors
    print("An error occurred while reading the input firmware file.")
    sys.exit(1)

# Define the magic number and calculate the CRC32 checksum
magic_number = 0x4C534343 #LSCC
crc32_checksum = zlib.crc32(firmware_data)

# Define the header fields using little-endian byte order for RISC-V
header = struct.pack('<III244x', magic_number, len(firmware_data), crc32_checksum)

# Concatenate the header and firmware image data
firmware_image = header + firmware_data

# Write the firmware image to the output file
try:
    # Check if the output file already exists
    if os.path.exists(args.output_file):
        print(f"Warning: output file {args.output_file} already exists and will be overwritten.")
        
    # Write the firmware image to the output file (overwriting if it already exists)
    with open(args.output_file, 'wb') as f:
        f.write(firmware_image)
except:
    print(f"An error occurred while writing the output firmware image file at {args.output_file}")
    sys.exit(1)
